package org.lsst.ccs.drivers.lambda;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
 
import org.lsst.ccs.drivers.i2c.I2c;

/**
 *  Routines for controlling a Lambda CPFE1000 power supply
 *
 *  @author  Owen Saxton
 */
public class Cpfe1000 extends I2c implements PowerSupplyDriver {

    /**
     *  Constants and data.
     */
    public static final int
        MODEL_UNKNOWN = -1,
        MODEL_12 = 0,
        MODEL_28 = 1,
        MODEL_48 = 2,
        MODEL_I12 = 3,
        MODEL_I28 = 4,
        MODEL_I48 = 5;

    public static final int
        STATUS_OFF  = 0x01,
        STATUS_DSAB = 0x02,
        STATUS_IOGB = 0x04,
        STATUS_OTW  = 0x08,
        STATUS_OTP  = 0x10,
        STATUS_VBAD = 0x20,
        STATUS_IBAD = 0x40,
        STATUS_MASK = 0x7f;

    private static final int
        REG_STATUS       = 0x07,
        REG_SERIAL_NO    = 0x01,
        REG_FIRMWARE_REV = 0x02,
        REG_PRODUCT_REV  = 0x03,
        REG_UNIT_PART_NO = 0x08,
        REG_MFG_DATE     = 0x09,
        REG_MFG_LOCATION = 0x10,
        REG_VOLTAGE      = 0x04,
        REG_CURRENT      = 0x05,
        REG_TEMPERATURE  = 0x06,
        REG_TURN_ON      = 0x1a,
        REG_TURN_OFF     = 0x0a,
        LENG_STATUS         = 1,
        LENG_SERIAL_NO      = 20,
        LENG_FIRMWARE_REV   = 3,
        LENG_FIRMWARE_REV_I = 4,
        LENG_PRODUCT_REV    = 4,
        LENG_UNIT_PART_NO   = 11,
        LENG_UNIT_PART_NO_I = 12,
        LENG_MFG_DATE       = 8,
        LENG_MFG_LOCATION   = 3,
        LENG_VOLTAGE        = 2,
        LENG_CURRENT        = 2,
        LENG_TEMPERATURE    = 2,
        LENG_TURN_ON        = 1,
        LENG_TURN_OFF       = 1,
        ADC_BAD_MASK      = 0xfc00;

    private static final Map<String, Integer> modelMap = new HashMap<>();
    static {
        modelMap.put("12", MODEL_12);
        modelMap.put("28", MODEL_28);
        modelMap.put("48", MODEL_48);
        modelMap.put("I1", MODEL_I12);
        modelMap.put("I2", MODEL_I28);
        modelMap.put("I4", MODEL_I48);
    }

    private static final double[]
        vScales = {0.034, 0.0777, 0.1355, 0.01668, 0.03973, 0.06772},
        iScales = {0.0815, 0.0488, 0.0282, 0.0815, 0.0488, 0.0282},
        tScales = {1.8, 2.048};

    private static final int[]
        lengFirmwareRevs = {LENG_FIRMWARE_REV, LENG_FIRMWARE_REV_I},
        lengUnitPartNos = {LENG_UNIT_PART_NO, LENG_UNIT_PART_NO_I};

    private int busAddr, model, lengFirmwareRev, lengUnitPartNo;
    private double vScale, iScale, tScale;


    /**
     *  Opens a connection.
     *
     *  @param  ident     The serial port name
     *  @param  addr      The I2C bus address
     *  @throws  DriverException
     */
    @Override
    public void open(String ident, int addr) throws DriverException
    {
        super.open(ident, 0);
        busAddr = addr;
        model = MODEL_UNKNOWN;
        lengUnitPartNo = LENG_UNIT_PART_NO;
        try {
            String mName = getUnitPartNo();
            if (mName.startsWith("CPFE1000F")) {
                Integer mod = modelMap.get(mName.substring(9));
                model = mod == null ? model : mod;
            }
            if (model == MODEL_UNKNOWN) {
                throw new DriverException("Unrecognized Lambda PS model: " + mName);
            }
            vScale = vScales[model];
            iScale = iScales[model];
            tScale = tScales[model / 3];
            lengFirmwareRev = lengFirmwareRevs[model / 3];
            lengUnitPartNo = lengUnitPartNos[model / 3];
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Gets the model.
     *
     *  @return  The model type
     */
    public int getModel()
    {
        return model;
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNo() throws DriverException
    {
        return readString(REG_SERIAL_NO, LENG_SERIAL_NO);
    }


    /**
     *  Gets the firmware revision.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getFirmwareRev() throws DriverException
    {
        return readString(REG_FIRMWARE_REV, lengFirmwareRev);
    }


    /**
     *  Gets the product revision.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getProductRev() throws DriverException
    {
        return readString(REG_PRODUCT_REV, LENG_PRODUCT_REV);
    }


    /**
     *  Gets the unit part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getUnitPartNo() throws DriverException
    {
        return readString(REG_UNIT_PART_NO, lengUnitPartNo);
    }


    /**
     *  Gets the manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getManufactureDate() throws DriverException
    {
        return readString(REG_MFG_DATE, LENG_MFG_DATE);
    }


    /**
     *  Gets the manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getManufactureLoc() throws DriverException
    {
        return readString(REG_MFG_LOCATION, LENG_MFG_LOCATION);
    }


    /**
     *  Reads the status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readStatus() throws DriverException
    {
        return readInt(REG_STATUS, LENG_STATUS) & STATUS_MASK;
    }


    /**
     *  Reads the voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readVoltage() throws DriverException
    {
        return fixValue(readInt(REG_VOLTAGE, LENG_VOLTAGE)) * vScale;
    }


    /**
     *  Reads the current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readCurrent() throws DriverException
    {
        return fixValue(readInt(REG_CURRENT, LENG_CURRENT)) * iScale;
    }


    /**
     *  Reads the baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return (fixValue(readInt(REG_TEMPERATURE, LENG_TEMPERATURE)) - 610) / tScale + 25.0;
    }


    /**
     *  Turns on the power.
     *
     *  @throws  DriverException
     */
    public void powerOn() throws DriverException
    {
        write(busAddr, REG_TURN_ON, new byte[LENG_TURN_ON]);
    }


    /**
     *  Turns off the power.
     *
     *  @throws  DriverException
     */
    public void powerOff() throws DriverException
    {
        write(busAddr, REG_TURN_OFF, new byte[LENG_TURN_OFF]);
    }


    /**
     *  Reads an integer.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The integer value
     *  @throws  DriverException
     */
    private int readInt(int reg, int count) throws DriverException
    {
        byte[] data = doRead(reg, count);
        int value = data[0] & 0xff;
        if (count == 2) {
            value = (value << 8) | (data[1] & 0xff);
        }

        return value;
    }


    /**
     *  Reads a string.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The string value
     *  @throws  DriverException
     */
    private String readString(int reg, int count) throws DriverException
    {
        return new String(doRead(reg, count));
    }


    /**
     *  Reads data.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The array of read bytes
     *  @throws  DriverException
     */
    private byte[] doRead(int reg, int count) throws DriverException
    {
        byte[] data = new byte[count];
        int nread = read(busAddr, reg, data);
        if (nread < count) {
            throw new DriverException("Data deficit: " + nread + "/" + count + " bytes read");
        }
        return data;
    }


    /**
     *  Fixes a value.
     *
     *  If the raw value is 2^10 or greater, NaN is returned.
     *
     *  @param  raw    The raw value
     *  @return  The calculated value
     */
    private static double fixValue(int raw)
    {
        return (raw & ADC_BAD_MASK) == 0 ? raw : Double.NaN;
    }


    @Override
    public void open(DriverConstants.ConnType connType, String ident, int addr) throws DriverException
    {
        if (connType == DriverConstants.ConnType.SERIAL) {
            open(ident, addr);
        }
        else {
            throw new DriverException("Invalid connection type: " + connType);
        }
    }


    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readCurrent();
    }


    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readCurrent();
    }


    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
    }


    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readVoltage();
    }


    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readVoltage();
    }


    @Override
    public void setVoltage(double value, int chan)
    {
    }


    @Override
    public void setOffDelay(double value, int chan)
    {
    }


    @Override
    public void setOnDelay(double value, int chan)
    {
    }


    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return (readStatus() & STATUS_OFF) == 0;
    }


    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        if (on) {
            powerOn();
        }
        else {
            powerOff();
        }
    }

}
