 package org.lsst.ccs.drivers.lambda;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.i2c.I2c;

/**
 *  Routines for controlling a Lambda CPFE1000 power supply
 *
 *  @author  Owen Saxton
 */
public class Cpfe1000 extends I2c {

    /**
     *  Constants and data.
     */
    public static final int
        STATUS_OFF  = 0x01,
        STATUS_DSAB = 0x02,
        STATUS_IOGB = 0x04,
        STATUS_OTW  = 0x08,
        STATUS_OTP  = 0x10,
        STATUS_VBAD = 0x20,
        STATUS_IBAD = 0x40,
        STATUS_MASK = 0x7f;

    private static final int
        REG_STATUS       = 0x07,
        REG_SERIAL_NO    = 0x01,
        REG_FIRMWARE_REV = 0x02,
        REG_PRODUCT_REV  = 0x03,
        REG_UNIT_PART_NO = 0x08,
        REG_MFG_DATE     = 0x09,
        REG_MFG_LOCATION = 0x10,
        REG_VOLTAGE      = 0x04,
        REG_CURRENT      = 0x05,
        REG_TEMPERATURE  = 0x06,
        REG_TURN_ON      = 0x1a,
        REG_TURN_OFF     = 0x0a,
        LENG_STATUS       = 1,
        LENG_SERIAL_NO    = 20,
        LENG_FIRMWARE_REV = 3,
        LENG_PRODUCT_REV  = 4,
        LENG_UNIT_PART_NO = 11,
        LENG_MFG_DATE     = 8,
        LENG_MFG_LOCATION = 3,
        LENG_VOLTAGE      = 2,
        LENG_CURRENT      = 2,
        LENG_TEMPERATURE  = 2,
        LENG_TURN_ON      = 1,
        LENG_TURN_OFF     = 1,
        ADC_BAD_MASK      = 0xfc00;

    private static final int
        MODEL_UNK = -1,
        MODEL_12 = 0,
        MODEL_28 = 1,
        MODEL_48 = 2;
    private static final double[]
        vScale = {0.034, 0.0777, 0.1355},
        iScale = {0.0815, 0.0488, 0.0282};

    private int busAddr, model;


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type
     *  @param  ident     The identification
     *  @param  addr      The I2C bus address
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int addr) throws DriverException
    {
        super.open(connType, ident, 0);
        busAddr = addr;
        model = MODEL_UNK;
        try {
            String mName = getUnitPartNo();
            if (mName.startsWith("CPFE1000F")) {
                String mNum = mName.substring(9);
                model = mNum.equals("12") ? MODEL_12
                          : mNum.equals("28") ? MODEL_28
                          : mNum.equals("48") ? MODEL_48 : MODEL_UNK;
            }
            if (model == MODEL_UNK) {
                throw new DriverException("Unrecognized Lambda PS model: "
                                            + mName);
            }
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNo() throws DriverException
    {
        return readString(REG_SERIAL_NO, LENG_SERIAL_NO);
    }


    /**
     *  Gets the firmware revision.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getFirmwareRev() throws DriverException
    {
        return readString(REG_FIRMWARE_REV, LENG_FIRMWARE_REV);
    }


    /**
     *  Gets the product revision.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getProductRev() throws DriverException
    {
        return readString(REG_PRODUCT_REV, LENG_PRODUCT_REV);
    }


    /**
     *  Gets the unit part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getUnitPartNo() throws DriverException
    {
        return readString(REG_UNIT_PART_NO, LENG_UNIT_PART_NO);
    }


    /**
     *  Gets the manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getManufactureDate() throws DriverException
    {
        return readString(REG_MFG_DATE, LENG_MFG_DATE);
    }


    /**
     *  Gets the manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getManufactureLoc() throws DriverException
    {
        return readString(REG_MFG_LOCATION, LENG_MFG_LOCATION);
    }


    /**
     *  Reads the status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readStatus() throws DriverException
    {
        return readInt(REG_STATUS, LENG_STATUS) & STATUS_MASK;
    }


    /**
     *  Reads the voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readVoltage() throws DriverException
    {
        return fixValue(readInt(REG_VOLTAGE, LENG_VOLTAGE)) * vScale[model];
    }


    /**
     *  Reads the current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readCurrent() throws DriverException
    {
        return fixValue(readInt(REG_CURRENT, LENG_CURRENT)) * iScale[model];
    }


    /**
     *  Reads the baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return (fixValue(readInt(REG_TEMPERATURE, LENG_TEMPERATURE)) - 610) / 1.8 + 25.0;
    }


    /**
     *  Turns on the power.
     *
     *  @throws  DriverException
     */
    public void powerOn() throws DriverException
    {
        write(busAddr, REG_TURN_ON, new byte[LENG_TURN_ON]);
    }


    /**
     *  Turns off the power.
     *
     *  @throws  DriverException
     */
    public void powerOff() throws DriverException
    {
        write(busAddr, REG_TURN_OFF, new byte[LENG_TURN_OFF]);
    }


    /**
     *  Reads an integer.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The integer value
     *  @throws  DriverException
     */
    private int readInt(int reg, int count) throws DriverException
    {
        byte[] data = doRead(reg, count);
        int value = data[0] & 0xff;
        if (count == 2) {
            value = (value << 8) | (data[1] & 0xff);
        }

        return value;
    }


    /**
     *  Reads a string.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The string value
     *  @throws  DriverException
     */
    private String readString(int reg, int count) throws DriverException
    {
        return new String(doRead(reg, count));
    }


    /**
     *  Reads data.
     *
     *  @param  reg    The register number
     *  @param  count  The number of bytes to read
     *  @return  The array of read bytes
     *  @throws  DriverException
     */
    private byte[] doRead(int reg, int count) throws DriverException
    {
        byte[] data = new byte[count];
        int nread = read(busAddr, reg, data);
        if (nread < count) {
            throw new DriverException("Data deficit: " + nread + "/" + count + " bytes read");
        }
        return data;
    }


    /**
     *  Fixes a value.
     *
     *  If the raw value is 2^10 or greater, NaN is returned.
     *
     *  @param  raw    The raw value
     *  @return  The calculated value
     */
    private static double fixValue(int raw)
    {
        return (raw & ADC_BAD_MASK) == 0 ? raw : Double.NaN;
    }

}
