package org.lsst.ccs.drivers.lambda;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/*
 *  Program to test the Lambda CPFE1000 power supply driver
 * 
 *  @author Owen Saxton
 */
public class TestCpfe1000 {

    private static final int MAX_DEVICE = 4;

    protected static enum OnOff { ON, OFF; };

    private final Cpfe1000[] cpfe = new Cpfe1000[MAX_DEVICE];
    private int nOpen = 0;


    public TestCpfe1000()
    {
        for (int j = 0; j < cpfe.length; j++) {
            cpfe[j] = new Cpfe1000();
        }
    }


    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Cpfe1000.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="addrs", description="Bus addresses")
                     int... addrs) throws DriverException
    {
        if (nOpen > 0) {
            throw new DriverException("Device(s) already open");
        }
        if (addrs.length == 0) {
            throw new DriverException("No devices specified");
        }
        if (addrs.length >= MAX_DEVICE) {
            throw new DriverException("Too many devices specified");
        }
        for (int j = 0; j < addrs.length; j++) {
            try {
                cpfe[j].open(connType, ident, addrs[j]);
            }
            catch (DriverException e) {
                for (int k = 0; k < j; k++) {
                    cpfe[k].close();
                }
                throw e;
            }
        }
        nOpen = addrs.length;
    }


    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        checkOpen();
        for (int j = 0; j < nOpen; j++) {
            try {
                cpfe[j].close();
            }
            catch (DriverException e) {
            }
        }
        nOpen = 0;
    }


    @Command(name="showserial", description="Show the serial number")
    public String showSerial(@Argument(name="devs", description="Device indices")
                             int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getSerialNo()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showfirmware", description="Show the firmware revision")
    public String showFirmware(@Argument(name="devs", description="Device indices")
                               int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getFirmwareRev()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showproduct", description="Show the product revision")
    public String showProduct(@Argument(name="devs", description="Device indices")
                              int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getProductRev()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showunitpart", description="Show the unit part number")
    public String showUnitPart(@Argument(name="devs", description="Device indices")
                               int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getUnitPartNo()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showdatemanu", description="Show the manufacture date")
    public String showDateManu(@Argument(name="devs", description="Device indices")
                               int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getManufactureDate()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showlocmanu", description="Show the manufacture location")
    public String showLocManu(@Argument(name="devs", description="Device indices")
                              int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(cpfe[j].getManufactureLoc()).append(" ");
        }
        return text.toString();
    }


    @Command(name="showstatus", description="Show the status byte")
    public String showStatus(@Argument(name="devs", description="Device indices")
                             int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(String.format("%02x", cpfe[j].readStatus())).append(" ");
        }
        return text.toString();
    }


    @Command(name="showvoltage", description="Show the voltage")
    public String showVoltage(@Argument(name="devs", description="Device indices")
                              int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(String.format("%.2f", cpfe[j].readVoltage())).append(" ");
        }
        return text.toString();
    }


    @Command(name="showcurrent", description="Show the current")
    public String showCurrent(@Argument(name="devs", description="Device indices")
                              int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(String.format("%.2f", cpfe[j].readCurrent())).append(" ");
        }
        return text.toString();
    }


    @Command(name="showtemperature", description="Show the temperature")
    public String showTemperature(@Argument(name="devs", description="Device indices")
                                  int... devs) throws DriverException
    {
        checkOpen();
        StringBuilder text = new StringBuilder();
        for (int j : getIndices(devs)) {
            text.append(String.format("%.2f", cpfe[j].readTemperature())).append(" ");
        }
        return text.toString();
    }


    @Command(name="power", description="Turn the power on or off")
    public void power(@Argument(name="action", description="Action: on or off")
                      OnOff action,
                      @Argument(name="devs", description="Device indices")
                      int... devs) throws DriverException
    {
        checkOpen();
        for (int j : getIndices(devs)) {
            if (action == OnOff.ON) {
                cpfe[j].powerOn();
            }
            else {
                cpfe[j].powerOff();
            }
        }
    }


    private void checkOpen() throws DriverException
    {
        if (nOpen == 0) {
            throw new DriverException("No devices open");
        }
    }


    private int[] getIndices(int[] devs) throws DriverException
    {
        if (devs.length == 0) {
            int[] indices = new int[nOpen];
            for (int j = 0; j < indices.length; j++) {
                indices[j] = j;
            }
            return indices;
        }
        else {
            for (int dev : devs) {
                if (dev < 0 || dev >= nOpen) {
                    throw new DriverException("Invalid device index: " + dev);
                }
            }
            return devs;
        }
    }

}
