package org.lsst.ccs.drivers.lambda;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 ** 
 **  Program to test the Lambda CPFE1000 driver
 ** 
 **  @author Owen Saxton
 ** 
 ******************************************************************************
 */
public class TestCpfe1000 {

    protected static enum OnOff { ON, OFF; };

    private final Cpfe1000 cpfe = new Cpfe1000();


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    **  @param  connType  The type of connection: ftdi or serial
    **
    **  @param  ident     The identification: serial no. (ftdi) or port name
    **                    (serial)
    **
    **  @param  addr      The bus address
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Cpfe1000.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="addr", description="Bus address")
                     int addr) throws DriverException
    {
        cpfe.open(connType, ident, addr);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes the connection.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        cpfe.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the serial number.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showserial", description="Show the serial number")
    public String showSerial() throws DriverException
    {
        return cpfe.getSerialNo();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the firmware revision.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showfirmware", description="Show the firmware revision")
    public String showFirmware() throws DriverException
    {
        return cpfe.getFirmwareRev();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the product revision.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showproduct", description="Show the product revision")
    public String showProduct() throws DriverException
    {
        return cpfe.getProductRev();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the unit part number.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showunitpart", description="Show the unit part number")
    public String showUnitPart() throws DriverException
    {
        return cpfe.getUnitPartNo();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the manufacture date.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdatemanu", description="Show the manufacture date")
    public String showDateManu() throws DriverException
    {
        return cpfe.getManufactureDate();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the manufacture location.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showlocmanu", description="Show the manufacture location")
    public String showLocManu() throws DriverException
    {
        return cpfe.getManufactureLoc();
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the status byte.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showstatus", description="Show the status byte")
    public String showStatus() throws DriverException
    {
        return String.format("%02x", cpfe.readStatus());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the voltage.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the voltage")
    public String showVoltage() throws DriverException
    {
        return String.format("%.2f", cpfe.readVoltage());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the current.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the current")
    public String showCurrent() throws DriverException
    {
        return String.format("%.2f", cpfe.readCurrent());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the temperature.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showtemperature", description="Show the temperature")
    public String showTemperature() throws DriverException
    {
        return String.format("%.2f", cpfe.readTemperature());
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the power on or off.
    **
    **  @param  action  On or off
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="power", description="Turn the power on or off")
    public void power(@Argument(name="action", description="Action: on or off")
                      OnOff action) throws DriverException
    {
        if (action == OnOff.ON) {
            cpfe.powerOn();
        }
        else {
            cpfe.powerOff();
        }
    }

}
