package org.lsst.ccs.drivers.lakeshore;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the LakeShore 336 Temperature device.
 * 
 * @author Peter Doherty and Heather Kelly
 * 
 * Manual indicates baud rates of 300, 1200
 * Software timing
 * 1 start, 7 data, 1 parity, 1 stop
 * Odd parity
 * Baud 300 or 1200
 * Terminator: CR(0DH) LF(0AH) 
 *
 * Manual indicates that when using the serial connection, the only
 *  "common" commands available are *IDN? and *RST, see page 4-7
 *
 * The device powers on into LOCal mode.
 */
public class LS330 extends Ascii {

    /**
     * Public constants
     */

    public static final int
        MODE_LOCAL = 0,
        MODE_REMOTE = 1,
        MODE_REMOTEWITHLOCKOUT = 2;

    public enum onOff {

        OFF, ON;
    }

    public enum HeaterRange {
        OFF(0), 
        LOW(1),
        MEDIUM(2),
        HIGH(3);
  
        int value;
        HeaterRange(int value) {
            this.value = value;
        }

        public int getValue() { 
            return value;
        }

    }

  
    public enum Modes {
        LOCAL(MODE_LOCAL),
        REMOTE(MODE_REMOTE),
        REMOTEWITHLOCALLOCKOUT(MODE_REMOTEWITHLOCKOUT);
        
        int value;

        Modes(int value) {
            this.value = value;
        }

        public int getValue() { 
            return value;
        }
    }

    /**
     * Private constants
     */
    private Boolean verbose = false;
    private Boolean tracedebug = false;
    private Boolean fakeout = false;
    private double setPoint = -95.00;
    private String controlUnits = "C";
    private String sampleUnits = "C";
    private String controlChannel = "A";
    private String sampleChannel = "A";
    private final double maxAllowedTempC = 30.;
    private final double minAllowedTempC = -120.;    
    private final double maxAllowedTempK = maxAllowedTempC + 273.15;
    private final double minAllowedTempK = minAllowedTempC + 273.15;    

    /**
     * Constructor
     */
    public LS330()
    {
        setOptions(Option.NO_NET);
    }

    /**
     * Opens a connection.
     *
     * This overrides the Ascii version and so is always called.
     * 
     * @param type The enumerated connection type
     * @param ident The device identifier
     * @param parm1 The first parameter (baud rate) 
     * @param parm2 The second parameter (data characteristics)
     * @throws DriverException
     */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2) throws DriverException
    {
        int asciiParam = makeDataCharacteristics(DataBits.SEVEN, StopBits.ONE, Parity.ODD, FlowCtrl.NONE);
        super.open(type, ident, parm1, asciiParam);
        setCommandTerm(Terminator.CRLF);
        setMode(Modes.REMOTE);
        init();
    }

    /**
     * Initializes device.
     * 
     * Also sets up some standard settings
     * Celcius for temperature
     * Control channel A
     * set point -95.
     * Heater range 3 (high)
     * 
     * @throws DriverException
     */
    public void init() throws DriverException
    {
       if (verbose) {
           System.out.println("init of Lakeshore 336");
       }
       
       /* HMK This should be handled elsewhere, I think as part of the
         configuration  */
       setControlUnits("C");   // Celcius
       setControlChannel("A"); // use channel A for control loop
       setSampleUnits("C");    // Celcius
       setSampleChannel("A");  // use channel A for sampling too
       setSetpoint(-95.0);     // reasonable default
       setHeaterRange(HeaterRange.HIGH.getValue());      // high please
       if (tracedebug) {
           System.out.println("Done with init of Lakeshore 336");
       }
    }    
    
    /**
     * Set Verbosity
     *
     * @param flag Verbosity flag
     */
    public void setVerbose(boolean flag) {
        verbose = flag;
    }

    /**
     * Set Trace/Debug
     * 
     * @param flag Trace/debud flag
     */
    public void setTracedebug(boolean flag) {
        tracedebug = flag;
    }

    /**
     * Set Fakeout
     * 
     * @param flag Fakeout (simulation) flag
     */
    public void setFakeout(boolean flag) {
        fakeout = flag;
    }

    
    /**
     * Get the Lakeshore 330 identity.
     * Communication with instrument:
     *     IDN? Identification Query
     *     Input: *IDN?[term]
     *     Returned: manufacturer,model,serial,firmware_version[term]
     *     Format: aaaa,aaaaaaaa,aaaaaaa,n.n/n.n
     *     manufacture:  Manufacturer ID
     *     model:        Instrument model number
     *     serial:       Serial number
     *     fw version:   Instrument firmware version, main firmware/input firmware.
     *     Example: LSCI,MODEL330,1234567,1.0/1.0
     * 
     * @return   Instrument identity string
     * @throws      DriverException
     */
    public String getIdent() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getIdent");
        }
        String idstr;
        if (fakeout) {
            idstr = "Fake Lakeshore 330";
        } else {
            idstr = readLS330("*IDN?");
        }
        if (verbose) {
            System.out.println("ls330 getIdent: ID = " + idstr);
        }
        return idstr;
    }

    public void reset() throws DriverException {
        writeLS330("*RST");
    }

    public String getMode() throws DriverException {
        return readLS330("MODE?");
    }

    public void setMode(Modes mode) throws DriverException {
        writeLS330("MODE " + mode.getValue());
    }

    /**
     * Get the Lakeshore 330 temperature set point.
     * 
     * Communication with instrument:
     *     SETP? Control Setpoint Query
     *     Command: SETP?[term]
     *     Returned: value[term]
     *     Format: ±nnnnnn
     * 
     * @return      Current setpoint
     * @throws      DriverException
     */
    public double getSetpoint() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getSetpoint");
        }
        double setp;
        if (fakeout) {    
            setp = setPoint;
        } else {
            setp = Double.valueOf(readLS330("SETP?"));
        }
        if (verbose) {
            System.out.println("ls330 getSetpoint: Set Point = " + setp);
        }
        setPoint = setp;
        return setp;
    }    

    /** 
     * Sets the Lakeshore 330 temperature set point.
     * 
     * Communication with instrument:
     *     SETP Control Setpoint Command
     *     Command: SETP value[term]
     *     Format: n,±nnnnnn
     *     value: The value for the setpoint (in whatever units the device is using).
     *     Example: SETP -95.0[term]
     *
     * @param setp  Value (in current units) for temperature set point
     * @throws      DriverException
     */
    public void setSetpoint(double setp) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setSetpoint: " + setp);
        }
        if ("C".equals(controlUnits) && (setp < minAllowedTempC || setp > maxAllowedTempC)) {
            //String errstr = "ls330 setSetpoint: Invalid set point: " + setp;
            //System.out.println(errstr);     
            throw new DriverException("Invalid set point (C): " + setp);
        }   
        if ("K".equals(controlUnits) && (setp < minAllowedTempK || setp > maxAllowedTempK)) {
            //String errstr = "ls330 setSetpoint: Invalid set point: " + setp;
            //System.out.println(errstr);     
            throw new DriverException("Invalid set point (K): " + setp);
        }
        writeLS330("SETP " + setp);
        if (verbose) {
            System.out.println("ls330 setSetpoint: Set Point = " + setp);
        }
        setPoint = setp;
    }    
    
    /**
     * Get the Lakeshore 330 control channel.
     * 
     * Communication with instrument:
     *     CCHN? Control Channel Query.
     *     Input: CCHN?
     *     Returned: A or B
     *     Remarks: Returns the current control channel setting: 
     *              A = channel A, B = channel B.
     * 
     * @return      channel string: A or B
     * @throws      DriverException
     */
    public String getControlChannel() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getControlChannel");
        }
        String chan;
        if (fakeout) {    
            chan = controlChannel;
        } else {
            chan = (readLS330("CCHN?"));
        }
        if (verbose) {
            System.out.println("ls330 getControlChannel: channel = " + chan);
        }
        controlChannel = chan;
        return chan;
    }    
    
    /** 
     * Sets the Lakeshore 330 control channel.
     * 
     * Communication with instrument:
     *     CCHN Set Control Channel to A or B.
     *     Input: CCHN A, or CCHN B
     *     Returned: Nothing
     *     Remarks: Sets control channel to sensor A or B. 
     *     Example: CCHN A[term] changes the control channel to A.
     * 
     * @param chan  The channel to use for control: A or B
     * @throws      DriverException
     */
    public void setControlChannel(String chan) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setControlChannel: " + chan);
        }
        if ((!"A".equals(chan)) && (!"B".equals(chan))) {
            //String errstr;
            //errstr = "ls330 setControlChannel: Invalid channel: " + chan;
            //System.out.println(errstr);     
            throw new DriverException("Invalid channel: " + chan);
        }
        if (!fakeout) {
            writeLS330("CCHN " + chan);
        }
        if (verbose) {
            System.out.println("ls330 setControlChannel: chan = " + chan);
        }
        controlChannel = chan;
    }    
    
    /**
     * Gets the Lakeshore 330 control loop temperature units.
     * 
     * Communication with instrument:
     *     CUNI? Control Units Query.
     *     Input: CUNI?
     *     Returned: K, C, V, R, or M
     *     Remarks: Current control units setting: K = kelvin, C = Celsius, 
     *       V = volts, R = Ohms, M = millivolts. (should not see last three)
     * 
     * @return      units string
     * @throws      DriverException
     */
    public String getControlUnits() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getControlUnits");
        }
        String unitstr;
        if (fakeout) {    
            unitstr = controlUnits;
        } else {
            unitstr = (readLS330("CUNI?"));
        }
        if (verbose) {
            System.out.println("ls330 getControlUnits: units = " + unitstr);
        }
        controlUnits = unitstr;
        return unitstr;
    }    
    
    /** 
     * Sets the Lakeshore 330 units.
     * 
     * Communication with instrument:
     *     CUNI Set Units for the Control Channel.
     *     Input: CUNI K, CUNI C
     *     Returned: Nothing
     *     Remarks: Set control channel units: 
     *              K = kelvin, C = Celsius
     *     Note: This software does not allow for 'sensor' units
     * 
     * @param unitstr  units string: 'K' = kelvin, 'C' = Celsius
     * @throws      DriverException
     */
    public void setControlUnits(String unitstr) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setControlUnits: " + unitstr);
        }
        if ((!"C".equals(unitstr)) && (!"K".equals(unitstr))) {
            //String errstr;
            //errstr = "ls330 setControlUnits: Invalid units: " + unitstr;
            //System.out.println(errstr);     
            throw new DriverException("Invalid units: " + unitstr);
        }
        if (!fakeout) {
            writeLS330("CUNI " + unitstr);
        }
        if (verbose) {
            System.out.println("ls330 setControlUnits: units = " + unitstr);
        }
        controlUnits = unitstr;
    }    
    
    /**
     * Get the Lakeshore 330 control channel data measurement.
     * 
     * Communication with instrument:
     *     CDAT? Sample Sensor Data Query.
     *     Input: CDAT?
     *     Returned: ±000.00
     *     Remarks: Returns sample sensor data. 
     *       The value returned is 7 characters: a sign, 5 digits and a decimal 
     *       point. The last digit may be a null.
     *     Example: -123.40[term] Typical response for degrees Celsius
     *              +234.50[term] Typical response for kelvin or Celsius
     * 
     * @return      measured temperature in current units
     * @throws      DriverException
     */
    public Double getControlData() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getControlData");
        }
        Double data;
        if (fakeout) {    
            data = -95.00;
        } else {
            data = Double.valueOf(readLS330("CDAT?"));
        }
        if (verbose) {
            System.out.println("ls330 getControlData: data = " + data);
        }
        return data;
    }    
    
    /**
     * Get the Lakeshore 330 sample channel.
     * 
     * Communication with instrument:
     *     CCHN? Control Channel Query.
     *     Input: CCHN?
     *     Returned: A or B
     *     Remarks: Returns the current control channel setting: 
     *              A = channel A, B = channel B.
     * 
     * @return      channel string: A or B
     * @throws      DriverException
     */
    public String getSampleChannel() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getSampleChannel");
        }
        String chan;
        if (fakeout) {    
            chan = sampleChannel;
        } else {
            chan = readLS330("SCHN?");
        }
        if (verbose) {
            System.out.println("ls330 getSampleChannel: channel = " + chan);
        }
        sampleChannel = chan;
        return chan;
    }    
    
    /** 
     * Sets the Lakeshore 330 sample channel.
     * 
     * Communication with instrument:
     *     SCHN Set Sample Channel to A or B.
     *     Input: SCHN A, or SCHN B
     *     Returned: Nothing
     *     Remarks: Sets the sample channel to sensor A or B. 
     *     Example: SCHN A[term] changes the sample channel to A.
     * 
     * @param chan  The channel to use for sampling: A or B
     * @throws      DriverException
     */
    public void setSampleChannel(String chan) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setSampleChannel: " + chan);
        }
        if ((!"A".equals(chan)) && (!"B".equals(chan))) {
            //String errstr;
            //errstr = "ls330 setSampleChannel: Invalid channel: " + chan;
            //System.out.println(errstr);     
            throw new DriverException("Invalid channel: " + chan);
        }
        if (!fakeout) {
            writeLS330("SCHN " + chan);
        }
        if (verbose) {
            System.out.println("ls330 setSampleChannel: units = " + chan);
        }
        sampleChannel = chan;
    }    
    
    /**
     * Gets the Lakeshore 330 sample channel temperature units.
     * 
     * Communication with instrument:
     *     SUNI? Sample Units Query.
     *     Input: SUNI?
     *     Returned: K, C, V, R, or M
     *     Remarks: Current control units setting: K = kelvin, C = Celsius, 
     *       V = volts, R = Ohms, M = millivolts. (should not see last three)
     * 
     * @return      units string
     * @throws      DriverException
     */
    public String getSampleUnits() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getSampleUnits");
        }
        String unitstr;
        if (fakeout) {    
            unitstr = sampleUnits;
        } else {
            unitstr = (readLS330("SUNI?"));
        }
        if (verbose) {
            System.out.println("ls330 getSampleUnits: units = " + unitstr);
        }
        sampleUnits = unitstr;
        return unitstr;
    }    
    
    /** 
     * Sets the Lakeshore 330 sample channel units.
     * 
     * Communication with instrument:
     *     SUNI Set Units for the Sample Channel.
     *     Input: SUNI K, SUNI C
     *     Returned: Nothing
     *     Remarks: Set control channel units: 
     *              K = kelvin, C = Celsius
     *     Note: This software does not allow for 'sensor' units
     * 
     * @param unitstr  units string: 'K' = kelvin, 'C' = Celsius
     * @throws      DriverException
     */
    public void setSampleUnits(String unitstr) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setSampleUnits: " + unitstr);
        }
        if ((!"C".equals(unitstr)) && (!"K".equals(unitstr))) {
            //String errstr;
            //errstr = "ls330 setSampleUnits: Invalid units: " + unitstr;
            //System.out.println(errstr);     
            throw new DriverException("Invalid units: " + unitstr);
        }
        if (!fakeout) {
            writeLS330("SUNI " + unitstr);
        }
        if (verbose) {
            System.out.println("ls330 setSampleUnits: units = " + unitstr);
        }
        sampleUnits = unitstr;
    }    
    
    /**
     * Get the Lakeshore 330 sample channel data measurement.
     * 
     * Communication with instrument:
     *     SDAT? Sample Sensor Data Query.
     *     Input: SDAT?
     *     Returned: ±000.00
     *     Remarks: Returns sample sensor data. 
     *       The value returned is 7 characters: a sign, 5 digits and a decimal 
     *       point. The last digit may be a null.
     *     Example: -123.40[term] Typical response for degrees Celsius
     *              +234.50[term] Typical response for kelvin or Celsius
     * 
     * @return      measured temperature in current units
     * @throws      DriverException
     */
    public Double getSampleData() throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 getSampleData");
        }
        Double data;
        if (fakeout) {    
            data = -95.00;
        } else {
            data = Double.valueOf(readLS330("SDAT?"));
        }
        if (verbose) {
            System.out.println("ls330 getSampleData: data = " + data);
        }
        return data;
    }    
    
    /**
     * Gets the Lakeshore 330 heater current.
     * 
     * Communication with instrument:
     *     HEAT? Heater Output Query
     *     Input: HTR? loop[term]
     *     Returned: heater_value[term]
     *     Format: +nnn.n
     *     heater_value: heater output in percent (%) of current or power, 
     *         depending on setting.
     * 
     * @return      the heater power % 
     * @throws      DriverException
     */
    public double getHeater() throws DriverException {
        if (tracedebug) {
            System.out.println("ls330 getHeater: Getting heater status");
        }
        double htrpwr;
        if (fakeout) {    
            htrpwr = 50.00;
        } else {
            htrpwr = Double.valueOf(readLS330("HEAT?"));
        }
        if (verbose) {
            System.out.println("ls330 getHeater: HTR = " + htrpwr);
        }
        return htrpwr;
    }    

    /**
     * Gets the Lakeshore 330 heater range.
     * 
     * RANG? Heater Range Query
     * Input: RANGE? loop[term]
     * Format: n
     * Returned: range[term]
     * Format: n (Refer to command for description)
     * 
     * @return  the heater range: 0 = off, 1 = low, 2 = medium, 3 = high
     * @throws      DriverException
     */
    public int getHeaterRange() throws DriverException {
        if (tracedebug) {
            System.out.println("ls330 getHeaterRange");
        }
        int range;
        if (fakeout) {    
             range = 0;
        } else {  
            range = Integer.valueOf(readLS330("RANG?"));
        }
        if (verbose) {
            System.out.println("ls330 getHeaterRange: RANGE = " + range);
        }
        return range;
    }

    /**
     * Gets whether in control.
     * 
     * Autotuning status query, 
     * 0 = Manual, 1 = P, 2 = PI, and 3 = PID
     *
     * @return      Whether in control
     * @throws      DriverException
     */
    public boolean isInControl() throws DriverException {
        Integer tune;
        if (fakeout) {
            tune = 3;
        } else {
            tune = Integer.valueOf(readLS330("TUNE?"));
        }
        if (verbose) 
            System.out.println("ls330 isInControl = " + tune);
        return tune != 0;
    }

    
    /**
     * Sets the Lakeshore 330 heater range.
     * 
     * Communication with instrument:
     *     RANGE Heater Range Command
     *     Input: RANG range[term]
     *     Format: n,n
     *     range: 0 = Off, 1 = Low, 2 = Medium, 3 = High
     * 
     * @param range Specifies the control mode. Valid entries: 
     *              For loop 1: 0 = Off, 1 = Low (2.5 W), 2 = High (25 W)
     *              For loop 2: 0 = Off, 1 = On
     * @throws      DriverException
     */
    public void setHeaterRange(int range) throws DriverException {
        if (tracedebug) {
            System.out.println("trace: ls330 setHeaterRange: " + range);
        }
        if (!fakeout) {
            writeLS330("RANG " + range);
        }
        if (verbose) {
            System.out.println("ls330 setHeaterRange: Range = " + range);
        }
    }    
    
    /**
     * Writes a command.
     *
     * @param command The command to write, excluding terminator
     * @throws DriverException
     */
    public synchronized void writeLS330(String command) throws DriverException {
        if (fakeout) {
            System.out.println("Fake LS330 Cmd: " + command);
        } else {
            write(command);
        }
    }

    /**
     * Reads a response.
     *
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readLS330() throws DriverException {
        return(read());
    }

    /**
     ** Reads a response after writing a command.
     * 
     * @param command The command to write, excluding terminator
     * @return The command response
     * @throws DriverException
     */
    public synchronized String readLS330(String command) throws DriverException {
        writeLS330(command);
        return readLS330();
    }

}
