package org.lsst.ccs.drivers.lakeshore;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *************************************************************************** 
 ** 
 ** Program to test the LakeShore 330 temperature device driver
 ** 
 ** @author Heather Kelly
 ** 
 *************************************************************************** 
 */
public class TestLS330 {

	protected LS330 mydev;       
        
    @Command(name = "open", description = "Open a device for use")
	public void open(
            @Argument(name = "devId", description = "The device ID of the device to open") String MyDevId
                ) throws DriverException {
		
    	    mydev = new LS330();
			mydev.open(MyDevId,1200);
    }

    @Command(name="openftdi", description="Open FTDI connection to device")
    public void openftdi(@Argument(name="serialname", description="Serial device name") String host,
          @Argument(name="baud", description="baud") int baud) throws DriverException
    {
    	mydev = new LS330();
        mydev.openftdi(host, baud);
    }

    @Command(name="reset", description="Reset device")
    public void reset() throws DriverException {
        mydev.reset();
    }

    @Command(name="readident", description="Return ID of device")
    public String readident() throws DriverException {
        return mydev.getIdent();
    }

    @Command(name="getMode", description="Return current mode [0,2]")
    public String getMode() throws DriverException {
        return mydev.getMode();
    }

    @Command(name="setMode", description="Set the mode [0,2]")
    public void setMode(@Argument(name="mode", description="Mode Setting 0=LOC, 1=REM, 2=REMwithLockout") int mode) throws DriverException {
        
        switch (mode) {
        case 0:
          mydev.setMode(LS330.Modes.LOCAL);
          break;
        case 1:
          mydev.setMode(LS330.Modes.REMOTE);
          break;
        case 2:
          mydev.setMode(LS330.Modes.REMOTEWITHLOCALLOCKOUT);
          break;
        default:
            throw new DriverException("Invalid Mode: " + mode);
        }
    }

    @Command(name="getSetpoint", description="Return the current setpoint")
    public Double getSetpoint() throws DriverException {
        return mydev.getSetpoint();
    }

    @Command(name="setSetpoint", description="set the setpoint")
    public void setSetpoint(@Argument(name="temperature", description="Temperature setting") double temp) throws DriverException {
        mydev.setSetpoint(temp);
    }

    @Command(name="getControlChannel", description="return current control channel")
    public String getControlChannel() throws DriverException {
        return mydev.getControlChannel();
    }

    @Command(name="setControlChannel", description="set the Control channel")
    public void setControlChannel(@Argument(name="channel", description="Channel A or B") String chan) throws DriverException {
        mydev.setControlChannel(chan);
    }

    @Command(name="getControlUnits", description="returns current control units")
    public String getControlUnits() throws DriverException {
        return mydev.getControlUnits();
    }

    @Command(name="setControlUnits", description="set Control Units to K or C")
    public void setControlUnits(@Argument(name="units", description="K or C") String units) throws DriverException {
        mydev.setControlUnits(units);
    }

     @Command(name="getControlData", description="returns sample sensor data")
     public Double getControlData() throws DriverException {
         return mydev.getControlData();
     }

     @Command(name="getSampleChannel", description="returns sample channel: A or B")
     public String getSampleChannel() throws DriverException {
         return mydev.getSampleChannel();
     }

     @Command(name="setSampleChannel", description="set sample channel to A or B")
     public void setSampleChannel(@Argument(name="channel", description="set sample channel to A or B") String chan) throws DriverException {
         mydev.setSampleChannel(chan);
     }

     @Command(name="getSampleUnits", description="returns Sample units: K (kelvin), C (celsius), V (volts), R (Ohms), or M (millivolts) but should not see last three?")
     public String getSampleUnits() throws DriverException {
         return mydev.getSampleUnits();
     }

     @Command(name="setSampleUnits", description="setSampleUnits to K or C")
     public void setSampleUnits(@Argument(name="units", description="units K or C") String units) throws DriverException {
         mydev.setSampleUnits(units);
     }

     @Command(name="getSampleData", description="returns sample temperature measurement")
     public Double getSampleData() throws DriverException {
         return mydev.getSampleData();
     }

     @Command(name="isInControl", description="queries autotune setting")
     public Boolean isInControl() throws DriverException {
         return mydev.isInControl();
     }
}
