package org.lsst.ccs.drivers.lakeshore;

import java.text.DecimalFormat;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 **************************************************************************
 **
 ** General access routines for the LakeShore 330 Temperature device 
 ** @author Peter Doherty and Heather Kelly
 * Manual indicates baud rates of 300,1200
 * Software timing
 * 1 start, 7 data, 1 parity, 1 stop
 * Odd parity
 * Baud 300 or 1200
 * Terminator: CR(0DH) LF(0AH) 
 *
 *  Manual indicates that when using the serial connection, the only
 *  "common" commands available are *IDN? and *RST, see page 4-7
 *
 *  The device powers on into LOCal mode.
 * **************************************************************************
 */
public class LS330 extends Ascii {

    /**
     ** Public constants *
     **/

    public final static int
        MODE_LOCAL = 0,
        MODE_REMOTE = 1,
        MODE_REMOTEWITHLOCKOUT = 2;

    public enum onOff {

        OFF, ON;
    }

    public enum HeaterRange {
        OFF(0), 
        LOW(1),
        MEDIUM(2),
        HIGH(3);
  
        int value;
        HeaterRange(int value) {
            this.value = value;
        }

        public int getValue() { 
            return value;
        }

    }

  
    public enum Modes {
        LOCAL(MODE_LOCAL),
        REMOTE(MODE_REMOTE),
        REMOTEWITHLOCALLOCKOUT(MODE_REMOTEWITHLOCKOUT);
        
        int value;

        Modes(int value) {
            this.value = value;
        }

        public int getValue() { 
            return value;
        }
    }

    /**
     ** Private constants *
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private final String terminator = "\r\n";
    private int timeout = 1000;
    private final int wordLen = 7;
    private final int stopBits = 1;
    private final int parity = 1;
    private Boolean verbose = true;
    private Boolean tracedebug = false;
    private Boolean fakeout = false;
    private double setPoint = -95.00;
    private String controlUnits = "C";
    private String sampleUnits = "C";
    private String controlChannel = "A";
    private String sampleChannel = "A";
    private double maxAllowedTempC = 30.;
    private double minAllowedTempC = -120.;    
    private double maxAllowedTempK = maxAllowedTempC + 273.15;
    private double minAllowedTempK = minAllowedTempC + 273.15;    

    public LS330() throws DriverException {
    }

    /**
     **
     ** Opens a connection. * * @param device The device name of the serial
     * port
     * @param serialname
     * @param baud 
     **/
    public void open(String serialname, int baud) throws DriverException
    {
        int asciiParam = makeDataCharacteristics(Ascii.DataBits.SEVEN, Ascii.StopBits.ONE, Ascii.Parity.ODD, Ascii.FlowCtrl.NONE);
        open(Ascii.CONN_TYPE_SERIAL, serialname, baud, asciiParam);
        init();
    }

    /**
    **  Opens an FTDI connection.
     * @param serialname
     * @param baud
    */
    public void openftdi(String serialname, int baud) throws DriverException
    {
        System.out.println("opening connection to the LakeShore 330 temperature controller");
        int asciiParam = makeDataCharacteristics(Ascii.DataBits.SEVEN, Ascii.StopBits.ONE, Ascii.Parity.ODD, Ascii.FlowCtrl.NONE);
        open(Ascii.CONN_TYPE_FTDI, serialname, baud, asciiParam);
        init();
        if (this.tracedebug)
            System.out.println("Done opening Lakeshore 330");
    
    }

    /**
     **  Initializes device.
     ** Also sets up some standard settings
     ** Celcius for temperature
     ** Control channel A
     ** set point -95.
     ** Heater range 3 (high)
     **/
    public void init() throws DriverException
    {
       if (this.verbose)
           System.out.println("init of Lakeshore 330");
       setTerminator(terminator);
       setMode(Modes.REMOTE);
       
       /* HMK This should be handled elsewhere, I think as part of the
         configuration  */
       setControlUnits("C");   // Celcius
       setControlChannel("A"); // use channel A for control loop
       setSampleUnits("C");    // Celcius
       setSampleChannel("A");  // use channel A for sampling too
       setSetpoint(-95.0);     // reasonable default
       setHeaterRange(HeaterRange.HIGH.getValue());      // high please
       if (this.tracedebug)
           System.out.println("Done with init of Lakeshore 330");
    }    
    
    /**
     ** Closes the connection. * * @throws DriverException *
     **/
    public void close() throws DriverException {
    }

    /** set Verbosity */
    public void setVerbose(Boolean flag) throws DriverException {
        this.verbose = flag;
    }

    /** set TraceDebug true or false */
    public void setTracedebug(Boolean flag) throws DriverException {
        this.tracedebug = flag;
    }

    /** set Fakeout true or false */
    public void setFakeout(Boolean flag) throws DriverException {
        this.fakeout = flag;
    }

    /**
     ** Sets the receive timeout. * 
     * 0 means no timeout. * * @throws DriverException *
     * @param timeout The receive timeout (ms).
     */
    public void setTimeout(int timeout) throws DriverException {
       if (this.verbose)
           System.out.println("setTime Lakeshore 330");
        this.timeout = timeout;
    }
    
    /** Get the Lakeshore 330 identity.
     * Communication with instrument:
     *     IDN? Identification Query
     *     Input: *IDN?[term]
     *     Returned: manufacturer,model,serial,firmware_version[term]
     *     Format: aaaa,aaaaaaaa,aaaaaaa,n.n/n.n
     *     manufacture:  Manufacturer ID
     *     model:        Instrument model number
     *     serial:       Serial number
     *     fw version:   Instrument firmware version, main firmware/input firmware.
     *     Example: LSCI,MODEL330,1234567,1.0/1.0
     * 
     * @return   Instrument identity string
     **/
    public String getIdent() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getIdent");
        try {
            String idstr;
            if (this.fakeout == true) {
                idstr = "Fake Lakeshore 330";
            } else {
                idstr = readLS330("*IDN?");
            }
            if (this.verbose == true) 
                System.out.println("ls330 getIdent: ID = " + idstr);
            return idstr;
        } catch (DriverException e) {
            String errstr = "ls330 getIdent: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }

    public void reset() throws DriverException {
        try {
            writeLS330("*RST");
        } catch (DriverException e) {
            String errstr = "ls330 reset: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);
       }
    }

    public String getMode() throws DriverException {
        try {
            String mode = readLS330("MODE?"); 
            return (mode);
        } catch (DriverException e) {
            String errstr = "ls330 getMode: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);
       }
    }

    public void setMode(Modes mode) throws DriverException {
        try {
            if ((mode.getValue() < Modes.LOCAL.getValue()) && (mode.getValue() > Modes.REMOTEWITHLOCALLOCKOUT.getValue()))
                throw new DriverException("mode out of bounds [0,2], " + mode.getValue());
            writeLS330("MODE " + mode.getValue());
        } catch (DriverException e) {
            String errstr = "ls330 setMode: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }

    /** Get the Lakeshore 330 temperature set point
     * 
     * Communication with instrument:
     *     SETP? Control Setpoint Query
     *     Command: SETP?[term]
     *     Returned: value[term]
     *     Format: ±nnnnnn
     * 
     * @return      Current setpoint
    **/
    public double getSetpoint() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getSetpoint");
        try {
            double setp;
            if (this.fakeout == true) {    
                setp = setPoint;
            } else {
                setp = Double.valueOf(readLS330("SETP?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getSetpoint: Set Point = " + setp);
            setPoint = setp;
            return setp;
        } catch (DriverException e) {
            String errstr = "ls330 getSetpoint: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** 
     * Sets the Lakeshore 330 temperature set point
     * Communication with instrument:
     *     SETP Control Setpoint Command
     *     Command: SETP value[term]
     *     Format: n,±nnnnnn
     *     value: The value for the setpoint (in whatever units the device is using).
     *     Example: SETP -95.0[term]
     *
     * @param setp  Value (in current units) for temperature set point
    **/
    public void setSetpoint(double setp) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setSetpoint: " + setp);
        if ("C".equals(this.controlUnits) && (setp < minAllowedTempC) || (setp > maxAllowedTempC)) {
            String errstr = "ls330 setSetpoint: Invalid set point: " + setp;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }   
        if ("K".equals(this.controlUnits) && (setp < minAllowedTempK) || (setp > maxAllowedTempK)) {
            String errstr = "ls330 setSetpoint: Invalid set point: " + setp;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }   
        try {
            writeLS330("SETP " + setp);
            if (this.verbose == true) 
                System.out.println("ls330 setSetpoint: Set Point = " + setp);
            setPoint = setp;
        } catch (DriverException e) {
            String errstr = "ls330 setSetpoint: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Get the Lakeshore 330 control channel
     * 
     * Communication with instrument:
     *     CCHN? Control Channel Query.
     *     Input: CCHN?
     *     Returned: A or B
     *     Remarks: Returns the current control channel setting: 
     *              A = channel A, B = channel B.
     * 
     * @return      channel string: A or B
    **/
    public String getControlChannel() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getControlChannel");
        try {
            String chan;
            if (this.fakeout == true) {    
                chan = controlChannel;
            } else {
                chan = (readLS330("CCHN?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getControlChannel: channel = " + chan);
            controlChannel = chan;
            return chan;
        } catch (DriverException e) {
            String errstr = "ls330 getControlChannel: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** 
     * Sets the Lakeshore 330 control channel
     * Communication with instrument:
     *     CCHN Set Control Channel to A or B.
     *     Input: CCHN A, or CCHN B
     *     Returned: Nothing
     *     Remarks: Sets control channel to sensor A or B. 
     *     Example: CCHN A[term] changes the control channel to A.
     * 
     * @param chan  The channel to use for control: A or B
     * 
    **/
    public void setControlChannel(String chan) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setControlChannel: " + chan);
        if ((!"A".equals(chan)) && (!"B".equals(chan))) {
            String errstr;
            errstr = "ls330 setControlChannel: Invalid channel: " + chan;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
        try {
            if (this.fakeout != true)
                writeLS330("CCHN " + chan);
            if (this.verbose == true) 
                System.out.println("ls330 setControlChannel: units = " + chan);
            controlChannel = chan;
        } catch (DriverException e) {
            String errstr = "ls330 setControlChannel: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Get the Lakeshore 330 control loop temperature units
     * 
     * Communication with instrument:
     *     CUNI? Control Units Query.
     *     Input: CUNI?
     *     Returned: K, C, V, R, or M
     *     Remarks: Current control units setting: K = kelvin, C = Celsius, 
     *       V = volts, R = Ohms, M = millivolts. (should not see last three)
     * 
     * @return      units string
    **/
    public String getControlUnits() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getControlUnits");
        try {
            String unitstr;
            if (this.fakeout == true) {    
                unitstr = controlUnits;
            } else {
                unitstr = (readLS330("CUNI?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getControlUnits: units = " + unitstr);
            controlUnits = unitstr;
            return unitstr;
        } catch (DriverException e) {
            String errstr = "ls330 getControlUnits: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** 
     * Sets the Lakeshore 330 units
     * Communication with instrument:
     *     CUNI Set Units for the Control Channel.
     *     Input: CUNI K, CUNI C
     *     Returned: Nothing
     *     Remarks: Set control channel units: 
     *              K = kelvin, C = Celsius
     *     Note: This software does not allow for 'sensor' units
     * 
     * @param unitstr  units string: 'K' = kelvin, 'C' = Celsius
    **/
    public void setControlUnits(String unitstr) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setControlUnits: " + unitstr);
        if ((!"C".equals(unitstr)) && (!"K".equals(unitstr))) {
            String errstr;
            errstr = "ls330 setControlUnits: Invalid units: " + unitstr;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
        try {
            if (this.fakeout != true)
                writeLS330("CUNI " + unitstr);
            if (this.verbose == true) 
                System.out.println("ls330 setControlUnits: units = " + unitstr);
            controlUnits = unitstr;
        } catch (DriverException e) {
            String errstr = "ls330 setControlUnits: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Get the Lakeshore 330 control channel data measurement
     * 
     * Communication with instrument:
     *     CDAT? Sample Sensor Data Query.
     *     Input: CDAT?
     *     Returned: ±000.00
     *     Remarks: Returns sample sensor data. 
     *       The value returned is 7 characters: a sign, 5 digits and a decimal 
     *       point. The last digit may be a null.
     *     Example: -123.40[term] Typical response for degrees Celsius
     *              +234.50[term] Typical response for kelvin or Celsius
     * 
     * @return      measured temperature in current units
    **/
    public Double getControlData() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getControlData");
        try {
            Double data;
            if (this.fakeout == true) {    
                data = -95.00;
            } else {
                data = Double.valueOf(readLS330("CDAT?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getControlData: data = " + data);
            return data;
        } catch (DriverException e) {
            String errstr = "ls330 getControlData: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Get the Lakeshore 330 sample channel
     * 
     * Communication with instrument:
     *     CCHN? Control Channel Query.
     *     Input: CCHN?
     *     Returned: A or B
     *     Remarks: Returns the current control channel setting: 
     *              A = channel A, B = channel B.
     * 
     * @return      channel string: A or B
    **/
    public String getSampleChannel() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getSampleChannel");
        try {
            String chan;
            if (this.fakeout == true) {    
                chan = sampleChannel;
            } else {
                chan = (readLS330("SCHN?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getSampleChannel: channel = " + chan);
            sampleChannel = chan;
            return chan;
        } catch (DriverException e) {
            String errstr = "ls330 getSampleChannel: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** 
     * Sets the Lakeshore 330 sample channel
     * Communication with instrument:
     *     SCHN Set Sample Channel to A or B.
     *     Input: SCHN A, or SCHN B
     *     Returned: Nothing
     *     Remarks: Sets the sample channel to sensor A or B. 
     *     Example: SCHN A[term] changes the sample channel to A.
     * 
     * @param chan  The channel to use for sampling: A or B
     * 
    **/
    public void setSampleChannel(String chan) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setSampleChannel: " + chan);
        if ((!"A".equals(chan)) && (!"B".equals(chan))) {
            String errstr;
            errstr = "ls330 setSampleChannel: Invalid channel: " + chan;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
        try {
            if (this.fakeout != true)
                writeLS330("SCHN " + chan);
            if (this.verbose == true) 
                System.out.println("ls330 setSampleChannel: units = " + chan);
            sampleChannel = chan;
        } catch (DriverException e) {
            String errstr = "ls330 setSampleChannel: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Gets the Lakeshore 330 sample channel temperature units
     * 
     * Communication with instrument:
     *     SUNI? Sample Units Query.
     *     Input: SUNI?
     *     Returned: K, C, V, R, or M
     *     Remarks: Current control units setting: K = kelvin, C = Celsius, 
     *       V = volts, R = Ohms, M = millivolts. (should not see last three)
     * 
     * @return      units string
    **/
    public String getSampleUnits() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getSampleUnits");
        try {
            String unitstr;
            if (this.fakeout == true) {    
                unitstr = sampleUnits;
            } else {
                unitstr = (readLS330("SUNI?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getSampleUnits: units = " + unitstr);
            sampleUnits = unitstr;
            return unitstr;
        } catch (DriverException e) {
            String errstr = "ls330 getSampleUnits: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** 
     * Sets the Lakeshore 330 sample channel units
     * Communication with instrument:
     *     SUNI Set Units for the Sample Channel.
     *     Input: SUNI K, SUNI C
     *     Returned: Nothing
     *     Remarks: Set control channel units: 
     *              K = kelvin, C = Celsius
     *     Note: This software does not allow for 'sensor' units
     * 
     * @param unitstr  units string: 'K' = kelvin, 'C' = Celsius
    **/
    public void setSampleUnits(String unitstr) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setSampleUnits: " + unitstr);
        if ((!"C".equals(unitstr)) && (!"K".equals(unitstr))) {
            String errstr;
            errstr = "ls330 setSampleUnits: Invalid units: " + unitstr;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
        try {
            if (this.fakeout != true)
                writeLS330("SUNI " + unitstr);
            if (this.verbose == true) 
                System.out.println("ls330 setSampleUnits: units = " + unitstr);
            sampleUnits = unitstr;
            
        } catch (DriverException e) {
            String errstr = "ls330 setControlUnits: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Get the Lakeshore 330 sample channel data measurement
     * 
     * Communication with instrument:
     *     SDAT? Sample Sensor Data Query.
     *     Input: SDAT?
     *     Returned: ±000.00
     *     Remarks: Returns sample sensor data. 
     *       The value returned is 7 characters: a sign, 5 digits and a decimal 
     *       point. The last digit may be a null.
     *     Example: -123.40[term] Typical response for degrees Celsius
     *              +234.50[term] Typical response for kelvin or Celsius
     * 
     * @return      measured temperature in current units
    **/
    public Double getSampleData() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 getSampleData");
        try {
            Double data;
            if (this.fakeout == true) {    
                data = -95.00;
            } else {
                data = Double.valueOf(readLS330("SDAT?"));
            }
            if (this.verbose == true)
                System.out.println("ls330 getSampleData: data = " + data);
            return data;
        } catch (DriverException e) {
            String errstr = "ls330 getSampleData: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
    /** Gets the Lakeshore 330 heater current
     * Communication with instrument:
     *     HEAT? Heater Output Query
     *     Input: HTR? loop[term]
     *     Returned: heater_value[term]
     *     Format: +nnn.n
     *     heater_value: heater output in percent (%) of current or power, 
     *         depending on setting.
     * 
     * @return      the heater power % 
    **/
    public double getHeater() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("ls330 getHeater: Getting heater status");
        try {
            double htrpwr;
            if (this.fakeout == true) {    
                htrpwr = 50.00;
            } else {
                htrpwr = Double.valueOf(readLS330("HEAT?"));
            }
            if (this.verbose == true) 
                System.out.println("ls330 getHeater: HTR = " + htrpwr);
            return htrpwr;
        } catch (DriverException e) {
            String errstr = "ls330 getHeater: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    

    /** Gets the Lakeshore 330 heater range
     * RANG? Heater Range Query
     * Input: RANGE? loop[term]
     * Format: n
     * Returned: range[term]
     * Format: n (Refer to command for description)
     * 
     * @return  the heater range: 0 = off, 1 = low, 2 = medium, 3 = high
    **/
    public int getHeaterRange() throws DriverException {
        if (this.tracedebug == true)
            System.out.println("ls330 getHeaterRange");
        try {
            int range;
            if (this.fakeout == true) {    
                 range = 0;
            } else {  
                range = Integer.valueOf(readLS330("RANG?"));
            }
            if (this.verbose == true) 
                System.out.println("ls330 getHeaterRange: RANGE = " + range);
            return range;
        } catch (DriverException e) {
            String errstr = "ls330 getHeaterRange: " + e;
            System.err.println(errstr);     
            throw new DriverException(errstr);
        }
    }
    
    /** Sets the Lakeshore 330 heater range
     * Communication with instrument:
     *     RANGE Heater Range Command
     *     Input: RANG range[term]
     *     Format: n,n
     *     range: 0 = Off, 1 = Low, 2 = Medium, 3 = High
     * 
     * @param range Specifies the control mode. Valid entries: 
     *              For loop 1: 0 = Off, 1 = Low (2.5 W), 2 = High (25 W)
     *              For loop 2: 0 = Off, 1 = On
    **/
    public void setHeaterRange(int range) throws DriverException {
        if (this.tracedebug == true)
            System.out.println("trace: ls330 setHeaterRange: " + range);
        try {
            if (this.fakeout != true) {
                writeLS330("RANG " + range);
            }
            if (this.verbose == true) 
                System.out.println("ls330 setHeaterRange: Range = " + range);
        } catch (DriverException e) {
            String errstr = "ls330 setHeaterRange: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }    
    
     /**
     ** Writes a command. * * @param command The command to write, excluding
     * terminator * * @throws DriverE 
     * @param command   a command string
     **/
    public synchronized void writeLS330(String command) throws DriverException {
       // write(command + terminator);
        if (this.fakeout == true) {
                System.out.println("Fake LS330 Cmd: " + command);
            } else {
                write(command);
            }
    }

    /**
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     * @return  the response string
     */
    public synchronized String readLS330() throws DriverException {
        return(read());
    }

    /**
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     * @param command   command string
     * @return          response string
     */
    public synchronized String readLS330(String command) throws DriverException {
        writeLS330(command);
        return readLS330();
    }

}
