package org.lsst.ccs.drivers.keyence;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * General access the KeyenceG5001 device using its ASCII command interface. The lower-level methods
 * defined by the {@link Ascii} class are available in addition to the methods defined here.
 *
 * @author saxton
 * @author tether
 * @author Homer Neal
 */
public class KeyenceG5001 extends Ascii {

    private static final int timeout = 1000; // In milliseconds.
    public static final int DEFAULT_BAUD = 38400;
    public static final String DEFAULT_DEV = "/dev/ttyS1"; // default serial device

    /**
     * Map of error codes to messages.
     */
    private static final Map<String, String> errors;
    static {
        Map<String, String> tmp = new HashMap<>();
        tmp.put("50", "Command error");
        tmp.put("51", "Status error");
        tmp.put("60", "Command length error");
        tmp.put("61", "Parameter count error");
        tmp.put("62", "Parameter range error");
        tmp.put("63", "Parameter range error (OUT calculation count)");
        tmp.put("64", "Parameter range error (OUT or head number)");
        tmp.put("65", "Parameter range error (OUT for veloc/accn)");
        tmp.put("66", "Parameter range error (OUT recursive)");
        tmp.put("67", "Parameter range error (cycle too fast)");
        tmp.put("68", "Parameter range error (scaling)");
        tmp.put("69", "Parameter range error (analog output scaling)");
        tmp.put("70", "Parameter range error (number of data points)");
        tmp.put("71", "Parameter range error (OUT too large)");
        tmp.put("88", "Timeout error");
        tmp.put("99", "Other error");
        errors = Collections.unmodifiableMap(tmp);
    }

    /**
     * Constructor. Disallows using the device over the network.
     */
    public KeyenceG5001() {
        setOptions(Option.NO_NET);
        setDefaultBaud(DEFAULT_BAUD);
    }

    /**
     *  Opens a connection to the device; device state is unchanged. The device is left closed()
     *  if an exception is thrown.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier
     *  @param  baud   The baud rate: use 0 for the default of 38400.
     *  @param  dChar  The encoded data characteristics: always set to 0
     *  @throws  DriverException upon I/O error.
     */
    @Override
    public synchronized void open(ConnType type, String ident, int baud, int dChar) throws DriverException {
        super.open(type, ident, baud, 0);
        setTimeout(timeout);
        setTerminator(Terminator.CR);
    }

    /**
     * Sets the KeyenceG5001 laser state by switching to general mode (lasers on) or comm mode (off).
     *
     * @param laserstate Whether laser is turned on
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void laser(boolean laserstate) throws DriverException {
        if (laserstate) {
            genmode();
        } else {
            commmode();
        }
    }

    /**
     * Sets communications mode.
     *
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void commmode() throws DriverException {
        try {
            writeKey("Q0");
        } catch (DriverException e) {
            if (!e.getMessage().contains("51")) {  // Already in comm mode
                throw e;
            }
        }
    }

    /**
     * Sets general mode.
     *
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void genmode() throws DriverException {
        try {
            writeKey("R0");
        } catch (DriverException e) {
            if (!e.getMessage().contains("51")) {  // Already in gen mode
                throw e;
            }
        }
    }

    /**
     * Sets the number of samples to be averaged (as implied by cycle time).
     *
     * @param cycle Cycle time: 0: 2.55 us, 1: 5 us, 2: 10 us, 3: 20 us,
     *                          4: 50 us, 5: 100 us, 6:200 us, 7: 500 us, 8: 1000 us
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void setnsamps(int cycle) throws DriverException {
        writeKey("SW,CA," + cycle);
    }

    /**
     * Gets the number of samples to be averaged.
     *
     * @return The sample count.
     * @throws DriverException on I/O error or an error response from the device.
     */
    public int getnsamps() throws DriverException {
        return readIntKey("SR,CA");
    }

    /**
     * Sets the measurement mode for a sensor.
     *
     * @param ikey The sensor number
     * @param mode The mode: 0: Normal, 1: Translucent object, 2: Transparent object,
     *                       3: Transparent object 2, 4: Semi opaque
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void setmeasmode(int ikey, int mode) throws DriverException {
        writeKey("SW,HB,M", ikey, String.valueOf(mode));
    }

    /**
     * Gets the measurement mode for a sensor.
     *
     * @param ikey The sensor number
     * @return The measurement mode
     * @throws DriverException on I/O error or an error response from the device.
     */
    public int getmeasmode(int ikey) throws DriverException {
        return readIntKey("SR,HB,M", ikey);
    }

    /**
     * Sets the minimum display units for a sensor, that is, the unit of measure and the precision
     * displayed for that unit.
     * 
     * @param ikey The sensor number
     * @param dspu The units: 0: 0.01 mm, 1: 0.001 mm, 2: 0.0001 mm,
     *                        3: 0.00001 mm, 4: 0.1 um, 5: 0.01 um, 6: 0.001 um
     * @throws DriverException on I/O error or an error response from the device.
     */
    public void setmindispunit(int ikey, int dspu) throws DriverException {
        writeKey("SW,OG", ikey, String.valueOf(dspu));
    }

    /**
     * Gets minimum display units for a sensor.
     * 
     * @param ikey The sensor number
     * @return The display units.
     * @throws DriverException on I/O error or an error response from the device.
     * @see #setmindispunit(int, int)
     */
    public int getmindispunit(int ikey) throws DriverException {
        return readIntKey("SR,OG", ikey);
    }

    /**
     * Sets minimum display units for both sensors.
     * 
     * @param dspu The display units.
     * @throws DriverException on I/O error or an error response from the device.
     * @see #setmindispunit(int, int) 
     */
    public void setmindispunit(int dspu) throws DriverException {
        setmindispunit(1, dspu);
        setmindispunit(2, dspu);
    }

    /**
     * Gets the KeyenceG5001 measurement for a head.
     *
     * @param ikey The head number
     * @return The measured value
     * @throws DriverException on I/O error or an error response from the device.
     */
    public double readDistance(int ikey) throws DriverException {
        return readDoubleKey("MS", ikey)[0];
    }

    /**
     * Gets the KeyenceG5001 measurement for all heads.
     *
     * @return The array of values.
     * @throws DriverException on I/O error or an error response from the device.
     */
    public double[] readDistance() throws DriverException {
        return readDoubleKey("MA");
    }

    /**
     * Writes a command, reads the response and checks whether it's an error response.
     *
     * @param command The command to write.
     * @throws DriverException on I/O error or an error response from the device.
     */
    public synchronized void writeKey(String command) throws DriverException {

        String rply = read(command);
        if (!rply.startsWith(command.split(",")[0].toUpperCase())) {
            reportCommandError(rply);
        }
    }

    /**
     * Like {@link #writeKey(java.lang.String)} but returns the response stripped of the
     * the echoed command and comma at the beginning.
     *
     * @param command The command to write.
     * @return The command response string.
     * @throws DriverException on I/O error or an error response from the device.
     */
    public synchronized String readKey(String command) throws DriverException {

        String rply = read(command);
        if (rply.startsWith(command.toUpperCase())) {
            return rply.substring(command.length() + 1);
        } else {
            reportCommandError(rply);
            return null;  // Can't happen
        }
    }

    /**
     * Like {@link #writeKey(java.lang.String)} but assumes that the command
     * has the form CMD,NN,ARG where CMD is the command, NN is the two-digit sensor number
     * and ARG is the command argument.
     * 
     * @param command The command to send.
     * @param ikey The sensor number.
     * @param arg The command argument.
     * @throws DriverException on I/O error or an error response from the device.
     */
    private void writeKey(String command, int ikey, String arg) throws DriverException {
        writeKey(String.format("%s,%02d,%s", command, ikey, arg));
    }

    /**
     * Like {@link #readKey(java.lang.String)} but assumes that the command
     * has the form CMD,NN where CMD is the command and NN is the two-digit sensor number.
     * 
     * @param command The command to send.
     * @param ikey The sensor number.
     * @return The response.
     * @throws DriverException on I/O error or an error response from the device.
     */
    private String readKey(String command, int ikey) throws DriverException {
        return readKey(String.format("%s,%02d", command, ikey));
    }

    /**
     * Like {@link #readKey(java.lang.String, int)} but looks for and returns an integer
     * value at the end of the response.
     * 
     * @param command The command to send.
     * @param ikey The sensor number.
     * @return The integer value.
     * @throws DriverException on I/O error or an error response from the device.
     */
    private int readIntKey(String command, int ikey) throws DriverException {
        return getInt(readKey(command, ikey));
    }

    /**
     * Like {@link #readKey(java.lang.String)} but looks for and returns an integer
     * value at the end of the response.
     * 
     * @param command The command to send.
     * @return The integer value.
     * @throws DriverException on I/O error or an error response from the device.
     */
    private int readIntKey(String command) throws DriverException {
        return getInt(readKey(command));
    }

    /**
     * Gets an integer from a command response that has had the echoed command and comma
     * stripped from the beginning.
     * 
     * @param resp The stripped command response.
     * @return The integer value.
     * @throws DriverException .
     */
    private static int getInt(String resp) throws DriverException {
        try {
            return Integer.decode(resp);
        } catch (NumberFormatException ex) {
            throw new DriverException("Command response is not an integer: " + resp);
        }
    }

    /**
     * Like {@link #readKey(java.lang.String, int)} but looks for and returns a series
     * of double values at the end of the command.
     * 
     * @param command The command to send.
     * @param ikey The sensor number.
     * @return An array of double values. A NaN value represents a response value of +FFFFFFF, -FFFFFFF or
     * XXXXXXXX.
     * @throws DriverException on I/O error, error response or response that can't be parsed.
     */
    private double[] readDoubleKey(String command, int ikey) throws DriverException {
        return getDoubles(readKey(command, ikey));
    }

    /**
     * Like {@link #readKey(java.lang.String)} but looks for and returns a series
     * of double values at the end of the command.
     * 
     * @param command The command to send.
     * @return An array of double values. A NaN value represents a response value of +FFFFFFF, -FFFFFFF or
     * XXXXXXXX.
     * @throws DriverException on I/O error, error response or response that can't be parsed.
     */
    private double[] readDoubleKey(String command) throws DriverException {
        return getDoubles(readKey(command));
    }

    /**
     * Gets an array of doubles from a command response that's been stripped of the leading command-and-comma.
     * 
     * @param resp The stripped command response.
     * @return The array of double values. A NaN value represents a response value of +FFFFFFF, -FFFFFFF or
     * XXXXXXXX.
     * @throws DriverException if response is can't be parsed.
     */
    private static double[] getDoubles(String resp) throws DriverException {
        String[] words = resp.split(",");
        double[] value = new double[words.length];
        for (int i = 0; i < words.length; i++) {
            try {
                value[i] = Double.parseDouble(words[i]);
            } catch (NumberFormatException ex) {
                if (words[i].equals("-FFFFFFF") || words[i].equals("+FFFFFFF") || words[i].equals("XXXXXXXX")) {
                    value[i] = Double.NaN;
                } else {
                    throw new DriverException("Unparseable response: " + resp);
                }
            }
        }
        return value;
    }

    /**
     * Reports a command error.
     * 
     * @param  reply  The reply.
     * @throws  DriverException always.
     */
    private void reportCommandError(String reply) throws DriverException {
        String[] words = reply.split(",");
        if (words[0].equals("ER")) {
            String errnum = words[words.length - 1];
            String errmsg = errors.get(errnum);
            if (errmsg != null) {
                throw new DriverException("Command execution error: " + errnum + " ("+ errmsg + ")");
            } else {
                throw new DriverException("Command execution error: " + errnum);
            }
        }
        else {
            throw new DriverException("Unrecognized command response: " + reply);
        }
    }

}
