package org.lsst.ccs.drivers.keithley;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the Keithley N6487 device driver
 *     (based on Owen Saxton's Agilent driver tester)
 *
 *  @author Homer Neal
 */
public class TestN6487 extends TestScpi {

   /*
    *  Inner class for detecting console input while monitoring.
    */
    class ConsThread extends Thread {

        private final BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);
        private boolean[] consDone;
        private Thread mainThread;

        ConsThread() {
            super();
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                awaitStart();
                awaitTerminal();
                consDone[0] = true;
                if (mainThread != null) {
                    mainThread.interrupt();
                }
            }
        }

        public void start(boolean[] done, boolean wake) {
            if (getState() == Thread.State.NEW) {
                start();
            }
            consDone = done;
            consDone[0] = false;
            mainThread = wake ? Thread.currentThread() : null;
            consQueue.offer(0);
        }

        public void start(boolean[] done) {
            start(done, false);
        }

        private void awaitStart() {
            while (true) {
                try {
                    consQueue.take();
                    return;
                }
                catch (InterruptedException e) {
                }
            }
        }

        private void awaitTerminal() {
            while (true) {
                try {
                    if (System.in.available() > 0) {
                        break;
                    }
                    try {
                        Thread.sleep(50);
                    }
                    catch (InterruptedException e) {
                    }
                }
                catch (IOException e) {
                    break;
                }
            }
            try {
                while (System.in.available() > 0) {
                    System.in.read();
                }
            }
            catch (IOException e) {
            }
        }

    }

    /*
     * Fields
     */
    public enum onOff {
        OFF, ON;
    }

    private final N6487 n64;
    private final ConsThread consThread = new ConsThread();


    /*+
     *  Constructor
     */
    public TestN6487()
    {
        super(new N6487());
        n64 = (N6487)scpi;
    }


    /*+
     *  Shows the line frequency
     */
    @Command(name = "showlinefreq", description = "Show the line frequency")
    public String showLineFreq() throws DriverException
    {
        return "Line frequency = " + n64.getLineFreq() + " Hz";
    }


    /*+
     *  Shows the maximum buffer size
     */
    @Command(name = "showmaxbuffsize", description = "Show the maximum buffer size")
    public String showMaxBuffSize()
    {
        return "Maximum buffer size = " + n64.getMaxBufferSize();
    }


    /*+
     *  Sets the rate
     */
    @Command(name="setrate", description="Set reading rate")
    public void setRate(@Argument(name="value", description="number of power cycles")
                        double value) throws DriverException
    {
        n64.setRate(value);
    }


    /*+
     *  Shows the rate
     */
    @Command(name="showrate", description="Show reading rate")
    public String showRate() throws DriverException
    {
        return "Reading rate = " + n64.getRate();
    }


    /*+
     *  Sets ARM count
     */
    @Command(name="setarmcount", description="Set the ARM count")
    public void setArmCount(@Argument(name="value", description="ARM count")
                           int value) throws DriverException
    {
        n64.setArmCount(value);
    }


    /*+
     *  Sets TRIGger count
     */
    @Command(name="settrigcount", description="Set the trigger count")
    public void setTrigCount(@Argument(name="value", description="trigger count")
                           int value) throws DriverException
    {
        n64.setTrigCount(value);
    }


    /*+
     *  Clear buffer
     */
    @Command(name="clrbuff", description="Clear the read buffer")
    public void clearBuffer() throws DriverException
    {
        n64.clrBuff();
    }
    

    /*+
     * Start the accumulation of readings into the device's internal buffer
     */
    @Command(name = "accumbuff", description = "start the buffer accumulation")
    public void accumBuffer(int nreads, double nplc) throws DriverException
    {
        n64.accumBuffer(nreads, nplc);
    }


    /*+
     * Wait for buffer accumulation to complete
     */
    @Command(name = "waitaccum", description = "wait for accumulation completion")
    public void waitAccum(double timeout) throws DriverException
    {
        n64.waitAccum(timeout);
    }


    /*+
     *  Read buffer
     */
    @Command(name="readdblbuff", description="read the buffer")
    public String readBuffer() throws DriverException
    {
        double data[][] = n64.readBuffer();
        return "Data  = " + getString(data[0]) + "\nTimes = " + getString(data[1]);
    }


    /*+
     * Reads a set of currents
     */
    @Command(name = "readcurrents", description = "read a set of currents")
    public String readCurrents(int nreads, double nplc) throws DriverException
    {
        double data[][] = n64.readCurrents(nreads, nplc);
        return "Data  = " + getString(data[0]) + "\nTimes = " + getString(data[1]);
    }


    /*+
     *  Reset device to a useable state
     */
    @Override
    @Command(name="reset", description="reset the device to a useable state")
    public void reset() throws DriverException
    {
        n64.reset();
    }


    /*+
    *  Set buffer size
    */
    @Command(name="setbuffsize", description="Set the buffer size.")
    public void setBuffSize(@Argument(name="value", description="buffer size")
                           int value) throws DriverException
    {
        n64.setBuffSize(value);
    }


    /*+
     *  Sets the voltage range
     */
    @Command(name="setvoltagerange", description="Set the voltage range. Can be 10V, 50V or 500V")
    public void setVoltageRange(@Argument(name="value", description="Voltage Range to set")
                           double value) throws DriverException
    {
        n64.setVoltageRange(value);
    }


    /*+
     *  Sets the voltage.
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        n64.setVoltage(value);
    }


    /*+
     *  Ramps to the desired voltage.
     */
    @Command(name="rampvolts", description="Set the voltage")
    public void rampVolts(@Argument(name="duration", description="number of second for the ramp")
                           double duration,
                           @Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        n64.rampVolts(duration, value);
    }


    /*+
     *  Shows the set voltage.
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public String getVoltage() throws DriverException
    {
        return "Voltage = " + n64.getVoltage();
    }


    /*+
     *  Reads the actual voltage.
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + n64.readVoltage();
    }


    /*+
     *  Sets the current range
     */
    @Command(name="setcurrentrange", description="Set the current range")
    public void setCurrentRange(@Argument(name="value", description="Current Range to set")
                           double value) throws DriverException
    {
        n64.setCurrentRange(value);
    }


    /*+
     *  Zero correct the current
     */
    @Command(name="zerocorrectcurrent", description="Zero correct the current")
    public void zeroCorrectCurrent() throws DriverException
    {
        n64.zeroCorrectCurrent();
    }


    /*+
     *  Reads the actual current.
     */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + n64.readCurrent();
    }


    /*+
     *  Times reading the actual current.
     */
    @Command(name="timereadcurrent", description="Time reading the actual current")
    public String timeReadCurrent() throws DriverException
    {
        long startTime = System.currentTimeMillis();
        double value = n64.readCurrent();
        int time = (int)(System.currentTimeMillis() - startTime);
        return "Current = " + value + " (" + time + " ms)";
    }


    /*+
     *  Sets the soft current limit.
     */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value",
                                          description="Current limit to set")
                                double value) throws DriverException
    {
        n64.setCurrentLimit(value);
    }


    /*+
     *  Shows the soft current limit.
     */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public String getCurrentLimit() throws DriverException
    {
        return "Current limit = " + n64.getCurrentLimit();
    }


    /*+
     *  Turns the output on or off.
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off")
                          onOff state) throws DriverException
    {
        n64.setOutput(state == onOff.ON);
    }


    /*+
     *  Shows the output state.
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(n64.getOutput());
    }


    /*+
     *  Monitors the current.
     */
    @Command(name="moncurrent", description="Monitor the current")
    public void monCurrent(@Argument(name="interval", description="Monitoring interval (secs)")
                           double intvl) throws DriverException
    {
        System.out.println("Monitoring current: press any key to terminate");
        long millis = (long)(1000 * intvl);
        boolean done[] = {false};
        consThread.start(done, true);
        while (!done[0]) {
            System.out.print(n64.readCurrent() + " ");
            try {
                Thread.sleep(millis);
            }
            catch (InterruptedException e) {
            }
        }
        System.out.println();
    }


    /*+
     *  Times an arbitrary read command.
     */
    @Command(name="timereadstring", description="Time reading arbitrary command data")
    public String timeReadString(@Argument(name="command", description="Command to execute")
                                 String command) throws DriverException
    {
        long startTime = System.currentTimeMillis();
        String value = n64.readString(command);
        int time = (int)(System.currentTimeMillis() - startTime);
        return value + " (" + time + " ms)";
    }


    /*+
     *  Times an arbitrary write command.
     */
    @Command(name="timewritecommand", description="Time writing arbitrary command")
    public String timeWriteCommand(@Argument(name="command", description="Command to execute")
                                   String command) throws DriverException
    {
        long startTime = System.currentTimeMillis();
        n64.writeCommand(command);
        int time = (int)(System.currentTimeMillis() - startTime);
        return "(" + time + " ms)";
    }


    /*+
     *  Converts an array of numbers to a string.
     */
    private StringBuilder getString(double[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }


    /*+
     *  Converts a boolean to on/off.
     */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }
    
}
