package org.lsst.ccs.drivers.keithley;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 * Class for controlling the Keithley 6487 picoammeter and voltage source
 *
 * @author Homer Neal
 */
public class N6487 extends Scpi implements PowerSupplyDriver {

    public static final int MAX_BUFF_SIZE = 3000;
    public static final int DEFAULT_BAUD = 19200;         // Keithley default baud

    @Deprecated
    private static final String DEFAULT_DEV = "/dev/ttyUSB0"; // default serial device
    private static final double MINBIAS = -75.; // CCD protection values
    private static final double MAXBIAS = 5.0;
    private static final double MAXCURRENT = 0.001;

    private boolean debug = true;
    private boolean okToTalk = true;          // Blocking flag used during long operations
    private boolean accumInProgress = false;  // Possibly redundant
    //private boolean dataReady = false;        // Redundant
    private boolean trip = false;  // Not used
    private boolean abort = false; // Not used
    private int lineFreq;
    private boolean singleRead;
    
    private boolean do_avg = false;
    private int nAvg = 2;

    /**
     *  Constructor.
     */
    public N6487() {
        setOptions(Option.NO_NET);   // Disallow network connections
        setDefaultParm(DEFAULT_BAUD);
    }

    /**
     *  Opens a connection.
     *
     *  This gets called by all other open methods
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate, or 0 for the default (9600)
     *  @param  commParm  The communications parameters
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int commParm)
        throws DriverException
    {
        super.open(connType, ident, baudRate, commParm);
        try {
            setTimeout(10.0); // set SCPI timeout
            checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH, "MODEL 648", Scpi.CHECK_STARTS_WITH);
            writeCommand("SYST:LFR:AUTO ON");
            lineFreq = readInteger("SYST:LFR?");
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }

    /**
     * Returns a formatted device identification string.
     * 
     * @return The string
     * @throws DriverException 
     */
    public synchronized String printdevid() throws DriverException {
        checkOkToTalk();
        String[] id = getIdentification();
        return "Successfully connected to:"
                + "\nManufacturer:   " + id[Scpi.IDENT_MANUFACTURER]
                + "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME]
                + "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER]
                + "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }

    /**
     * Sets the debug state
     * 
     * @param setOn True to set debug state, false to clear it
     */
    public void setDebug(boolean setOn) {
        debug = setOn;
    }

    /**
     * Reset
     *
     * @throws DriverException
     */
    @Override
    public void reset() throws DriverException {
        okToTalk = true; // clear any incomplete previous state
        writeCommand("*RST");
        setArmCount(1);
        writeCommand("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeCommand("SOUR:VOLT:SWE:ABOR"); // stop any sweep any progress
    }

    /**
     * Soft reset
     *
     * @throws DriverException
     */
    public void softReset() throws DriverException {
        okToTalk = true; // clear any incomplete previous state
        setTrigCount(1);
        setArmCount(1);
        writeCommand("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeCommand("SOUR:VOLT:SWE:ABOR"); // stop any sweep any progress
    }

    /**
     * Clears status
     *
     * @throws DriverException
     */
    public synchronized void clearStat() throws DriverException {
        checkOkToTalk();
        clearStatus();
    }

    /**
     * Sets the display on(true)/off(false)
     *
     * @param dstate
     * @throws DriverException
     */
    public synchronized void setDisplay(boolean dstate) throws DriverException {
        checkOkToTalk();
        writeCommand("DISP:ENAB " + (dstate ? "ON" : "OFF"));
    }

    /**
     * Sets the integration rate as the number of power line cycles
     *
     * @param value The integration rate
     * @throws DriverException
     */
    public synchronized void setRate(double value) throws DriverException {
        checkOkToTalk();
        writeCommand("NPLC " + value);
    }

    /**
     * Gets the integration rate as the number of power line cycles
     *
     * @return The integration rate
     * @throws DriverException
     */
    public synchronized double getRate() throws DriverException {
        checkOkToTalk();
        return readDouble("NPLC?");
    }

    /**
     ** Sets the ARM count
     *
     * @param value The ARM count
     * @throws DriverException
     */
    public synchronized void setArmCount(int value) throws DriverException {
        checkOkToTalk();
        writeCommand("ARM:COUN " + value);
        singleRead = false;
    }

    /**
     * Sets the trigger count
     *
     * @param value The trigger count
     * @throws DriverException
     */
    public synchronized void setTrigCount(int value) throws DriverException {
        checkOkToTalk();
        writeCommand("TRIG:COUN " + value);
        singleRead = false;
    }

    /**
     * Sets the trigger delay
     *
     * @param value The trigger delay
     * @throws DriverException
     */
    public synchronized void setTrigDelay(double value) throws DriverException {
        checkOkToTalk();
        writeCommand("TRIG:DEL " + value);
        singleRead = false;
    }

    /**
     * Sets the buffer size (#readings up to 3000 for the 6487)
     *
     * @param value The buffer size
     * @throws DriverException
     */
    public synchronized void setBuffSize(int value) throws DriverException {
        checkOkToTalk();
        writeCommand("TRAC:POIN " + value);
    }

    /**
     * Clears the buffer
     *
     * @throws DriverException
     */
    public synchronized void clrBuff() throws DriverException {
        checkOkToTalk();
        writeCommand("TRAC:CLE");
    }

    /**
     * Sets the current range.
     *
     * @param value The current range
     * @throws DriverException
     */
    public synchronized void setCurrentRange(double value) throws DriverException {
        checkOkToTalk();
        writeCommand("FUNC 'CURR'");
        writeCommand("RANG " + value);
    }

    /**
     * Zero corrects the current.
     *
     * @throws DriverException
     */
    public synchronized void zeroCorrectCurrent() throws DriverException {
        checkOkToTalk();
        /* the following is just for the zero correction */
        writeCommand("SYST:ZCH ON");    // enable zero checking
        writeCommand("FUNC 'CURR'");
        writeCommand("INIT");           // trigger reading for zero correction
        writeCommand("SYST:ZCOR:ACQ");  // set zero correction value
        writeCommand("SYST:ZCOR ON");   // turn zero correction on
        writeCommand("SYST:ZCH OFF"); // important to turn this off before doing the read
    }

    /**
     * Reads the voltage & current.
     *
     * @return A two-element array containing the current [0] and voltage [1]
     * @throws DriverException
     */
    public synchronized double[] readPower() throws DriverException {
        checkOkToTalk();
        if (!singleRead) {
            setArmCount(1);
            setTrigCount(1);
            setTrigDelay(0.0);
            writeCommand("SYST:ZCH OFF"); // important to turn this off before doing the read
            writeCommand("FORM:ELEM VSO,READ");  // return the current & voltage (not the timestamp etc...)
            singleRead = true;
        }
        return readDoubleArray("READ?");  // read and return the result
    }

    /**
     * Reads the current.
     *
     * @return The read current
     * @throws DriverException
     */
    public double readCurrent() throws DriverException {
        return readPower()[0];
    }

    /**
     * Makes a set of current readings.
     *
     * @param nreads The number of readings
     * @param intvl The sample interval (in power line cycles)
     * @return The read current
     * @throws DriverException
     */
    public synchronized double[][] readCurrents(int nreads, double intvl) throws DriverException {
        checkOkToTalk();
        setArmCount(nreads);
        setTrigCount(1);
        double nplc = getRate();
        double delay = 0.0;
        if (intvl < nplc) {
            setRate(intvl);
        }
        else {
            delay = (intvl - nplc) / lineFreq;
        }
        setTrigDelay(delay);
        writeCommand("FORM:ELEM READ,TIME");  // read back current and time
        writeCommand("SYST:ZCH OFF");         // Turn zero check off
        writeCommand("SYST:AZER:STAT OFF");   // Turn auto-zero off
        writeCommand("SYST:TIME:RESET");      // Reset the timestamp
        String[] buff = readStringArray("READ?"); // read the result
        if (buff.length != 2 * nreads) {
            throw new DriverException("Bad data amount: expected " + (2 * nreads) + " items, got " + buff.length
                                        + ". First item = " + buff[0]);
        }
        return getDoublePairArray(buff); // extract value and time arrays
    }

    /**
     * Starts the accumulation of current readings in the buffer.
     *
     * @param nreads The number of readings
     * @param intvl The sample interval (in power line cycles)
     * @throws DriverException
     */
    public synchronized void accumBuffer(int nreads, double intvl) throws DriverException {
        //if (nreads > MAX_BUFF_SIZE) {
        //    throw new DriverException("Request read count (" + nreads + ") exceeds " + MAX_BUFF_SIZE);
        //}
        checkOkToTalk();
        setArmCount(nreads);
        setTrigCount(1);
        double nplc = getRate();
        double delay = 0.0;
        if (intvl < nplc) {
            setRate(intvl);
        }
        else {
            delay = (intvl - nplc) / lineFreq;
        }
        setTrigDelay(delay);
        setBuffSize(nreads);
        writeCommand("SYST:ZCH OFF");         // Turn zero check off
        writeCommand("SYST:AZER:STAT OFF");   // Turn auto-zero off
        writeCommand("TRAC:TST:FORM ABS");    // select DELT vs. ABS time format
        if (isDo_avg()) {
            writeCommand("AVER:COUNT " + nAvg );
            writeCommand("AVER:TCON REP");
            writeCommand("AVER ON");
        } else {
            writeCommand("AVER OFF");
        }
        writeCommand("TRAC:FEED SENS");       // store raw input readings
        writeCommand("TRAC:FEED:CONT NEXT");  // set buffer control mode and start reading
        debugPrint(String.format("Sending PD accumulation trigger at %14.3f", System.currentTimeMillis() / 1000.));
        writeCommand("INIT");           // trigger readings
        write("*OPC?");  // Plant completion test seed
        okToTalk = false;
        accumInProgress = true;
    }

    /**
     * Waits for the buffer accumulation to complete.
     * 
     * The command to check for completion has already been issued, and will not
     * be executed until completion actually occurs.  At that point the result
     * (1) will be sent.  Until then, reading will time out.
     * 
     * @param timeout The timeout (secs)
     * @throws DriverException
     */
    public synchronized void waitAccum(double timeout) throws DriverException {
        if (!accumInProgress) {
            debugPrint("No buffer accumulation in progress");
            return;
        }
        long endTime = System.currentTimeMillis() + (long)(1000 * timeout);
        DriverTimeoutException te = null;
        do {
            try {
                String resp = read();
                if (!resp.equals("1")) {
                    debugPrint("Unexpected response while waiting for buffer accumulation: " + resp);
                }
                te = null;
                okToTalk = true;
                accumInProgress = false;
                break;
            }
            catch (DriverTimeoutException e) {
                te = e;
            }
        } while (System.currentTimeMillis() <= endTime);
        if (te != null) {
            throw te;
        }
    }

    /**
     * Reads the accumulated buffer data.
     *
     * @return The array of buffered values
     * @throws DriverException
     */
    public synchronized double[][] readBuffer() throws DriverException {
        checkOkToTalk();
        int numread = readInteger("TRAC:POIN:ACT?");
        debugPrint("Number of elements read = " + numread);
        writeCommand("FORM:ELEM READ,TIME");  // read back current and time (add ,VSO for voltage)
        String[] buff = readStringArray("TRAC:DATA?");
        if (buff.length != 2 * numread) {
            throw new DriverException("Bad data amount: expected " + (2 * numread) + " items, got " + buff.length
                                        + ". First item = " + buff[0]);
        }
        return getDoublePairArray(buff); // extract data and time arrays
    }

    /**
     * Gets the maximum buffer size
     * 
     * @return The maximum allowed buffer size
     */
    public int getMaxBufferSize() {
        return MAX_BUFF_SIZE;
    }

    public boolean isDo_avg() {
        return do_avg;
    }

    public void setDo_avg(boolean do_avg) {
        this.do_avg = do_avg;
    }

    public int getnAvg() {
        return nAvg;
    }

    public void setnAvg(int nAvg) {
        this.nAvg = nAvg;
    }

    /**
     * Gets the line frequency
     * 
     * @return The frequency (50 or 60)
     * @throws DriverException
     */
    public int getLineFreq() throws DriverException {
        return lineFreq;
    }

    /**
     * Gets the blocking flag
     * 
     * @return The blocking flag
     */
    public boolean isOkToTalk() {
        return okToTalk;
    }

    /**
     * Gets whether data is ready
     * 
     * @return Whether data is ready
     */
    public boolean isDataReady() {
        return !accumInProgress;  //dataReady;
    }

    /**
     * Gets whether data accumulation is in progress
     * 
     * @return Whether data accumulation is in progress
     */
    public boolean isAccumInProgress() {
        return accumInProgress;
    }

    /**
     * Gets the minimum allowed bias value
     * 
     * @return The minimum value
     */
    public static double getMinBias() {
        return MINBIAS;
    }

    /**
     * Gets the maximum allowed bias value
     * 
     * @return The maximum value
     */
    public static double getMaxBias() {
        return MAXBIAS;
    }

    /**
     * Turns output on or off.
     *
     * @param on Whether to turn on
     * @throws DriverException
     */
    public synchronized void setOutput(boolean on) throws DriverException {
        checkOkToTalk();
        writeCommand("SOUR:VOLT:STAT " + (on ? "ON" : "OFF"));
    }

    /**
     * Gets the output state.
     *
     * @return Whether output is on
     * @throws DriverException
     */
    public synchronized boolean getOutput() throws DriverException {
        checkOkToTalk();
        return readInteger("SOUR:VOLT:STAT?") != 0;
    }

    /**
     * Gets the (unused) trip state.
     *
     * @return Whether trip is true
     */
    public boolean isTrip() {
        return trip;
    }

    /**
     * Sets the (unused) trip state.
     *
     * @param trip Whether trip is true
     */
    public void setTrip(boolean trip) {
        this.trip = trip;
    }

    /**
     * Gets the (unused) abort state.
     *
     * @return Whether abort is true
     */
    public boolean isAbort() {
        return abort;
    }

    /**
     * Sets the (unused) abort state.
     *
     * @param abort Whether abort is true
     */
    public void setAbort(boolean abort) {
        this.abort = abort;
    }

    /**
     * Sets the voltage range.
     *
     * @param value The voltage range
     * @throws DriverException
     */
    public synchronized void setVoltageRange(double value) throws DriverException {
        checkOkToTalk();
        writeCommand("SOUR:VOLT:RANG " + value);
    }

    /**
     * Sets the voltage.
     *
     * @param value The voltage to set
     * @throws DriverException
     */
    public synchronized void setVoltage(double value) throws DriverException {
        double usedvalue = 0.0;
        if (value < MINBIAS || value > MAXBIAS) {
            debugPrint("Only values between -75.V and 5.V are allowed in order to protect the CCD!");
            throw new DriverException();
        } else {
            usedvalue = value; // I know there is a more direct way but I'm just being extra cautious
        }
        checkOkToTalk();
        writeCommand("SOUR:VOLT " + usedvalue);
    }

    /**
     * Ramps the voltage to a desired level over a given duration.
     *
     * @param duration The time duration
     * @param value The voltage value
     * @param nsteps The number of steps
     * @throws DriverException
     */
    public synchronized void rampVoltsSweep(double duration, double value, int nsteps) throws DriverException {
        checkOkToTalk();
        double vnow = readVoltage();
        double vstep = Math.abs(value - vnow) / (double) nsteps;
        double delta = duration / (double) nsteps;
        debugPrint("starting ramp from " + Double.toString(vnow)
                + " to " + Double.toString(value)
                + " with vstep = " + Double.toString(vstep)
                + " with tstep = " + Double.toString(delta));

        writeCommand("SOUR:VOLT " + value);
        writeCommand("SOUR:VOLT:SWE:STAR " + vnow);
        writeCommand("SOUR:VOLT:SWE:STOP " + value);
        writeCommand("SOUR:VOLT:SWE:STEP " + vstep);
        writeCommand("SOUR:VOLT:SWE:DEL " + delta);
        setArmCount(nsteps + 1);
        writeCommand("FORM:ELEM READ");  // just read back the values (add ,VSO for voltage)
        writeCommand("SOUR:VOLT:SWE:INIT");
        writeCommand("SYST:ZCH OFF");

        debugPrint("started ramp");
        double[] buff = readDoubleArray("READ?"); // another approach
        debugPrint("ramp completed");
        /*
         write("INIT");
         debugPrint("started");
         int iwaitsecs = 0;
         boolean sweeping = false;
         debugPrint("waiting");
         while (iwaitsecs < timeout) {
         sweeping = read("SOUR:VOLT:SWE:STAT?").matches("1");
         debugPrint("sweeping = " + sweeping);
         if (!sweeping) {
         break;
         }
         debugPrint("waiting a second");

         try {
         debugPrint("trying to wait");
         Thread.currentThread().sleep(1000);//sleep for 1000 ms
         debugPrint("done waiting");
         } catch (Exception ie) {
         }
         debugPrint("Waiting for sweep to complete.");
         iwaitsecs++;
         }
         // if still sweeping then abort
         if (sweeping) writeCommand("SOUR:VOLT:SWE:ABOR");
         setArmCount(1);
         */
    }

    /**
     * Checks whether current is within bounds
     * 
     * @return Whether current is okay
     * @throws DriverException 
     */
    public boolean isCurrentOk() throws DriverException {
        return readCurrent() < MAXCURRENT;
    }

    /**
     * Ramps the voltage to a desired level over a given duration.
     *
     * @param duration The time duration
     * @param value The voltage value
     * @param nsteps The number of steps
     * @throws DriverException
     */
    public synchronized void rampVolts(double duration, double value, int nsteps) throws DriverException {
        checkOkToTalk();
        double vnow = readVoltage();
        double vstep = (value - vnow) / (double) nsteps;
        int delta = (int) (1000. * duration / (double) nsteps); // delta in msecs
        debugPrint("ramp from " + Double.toString(vnow)
                + " to " + Double.toString(value)
                + " with vstep = " + Double.toString(vstep)
                + " with tstep = " + Integer.toString(delta));

        debugPrint("starting ramp");

        double v = vnow;
        boolean trip0 = isTrip();
        //okToTalk = true; // override
        for (int istep = 0; istep < nsteps + 1; istep++) {
            debugPrint("V = " + v);

            if ((!trip0 && isTrip()) || isAbort()) {
                debugPrint("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                break;
            } else if (isCurrentOk()) {
                setVoltage(v);
            } else {
                debugPrint("driver aborting ramp due to high current!");
                this.setAbort(true);
            }

            try {
                Thread.sleep(delta);//sleep for delta ms
            } catch (Exception ie) {
            }
            v += vstep;
        }
        debugPrint("ramp completed");

    }

    /**
     * Ramps the voltage to a desired level over a given duration using a
     * default of 10 steps
     *
     * @param duration The time duration
     * @param value The voltage level
     * @throws DriverException
     */
    public void rampVolts(double duration, double value) throws DriverException {
        int nsteps = 10; // default to this. We'll see later if this should be an argument.
        rampVolts(duration, value, nsteps);
    }

    /**
     * Gets the set voltage.
     *
     * @return The set voltage
     * @throws DriverException
     */
    public synchronized double getVoltage() throws DriverException {
        checkOkToTalk();
        return readDouble("SOUR:VOLT?");
    }

    /**
     * Zero correct the voltage.
     *
     * @throws DriverException
     */
    public synchronized void zeroCorrectVoltage() throws DriverException {
        checkOkToTalk();
        writeCommand("SYST:ZCH ON"); // enable zero check
        writeCommand("SYST:GUAR ON"); // enable guard
        writeCommand("FUNC 'VOLT'");
        writeCommand("SYST:ZCOR ON"); // perform zero correction
        writeCommand("SYST:ZCH OFF"); // important to turn this off before doing the read
    }

    /**
     * Reads the voltage.
     *
     * @return The read voltage
     * @throws DriverException
     */
    public double readVoltage() throws DriverException {
        return readPower()[1];
    }

    /**
     * Sets the current limit.
     *
     * @param maxima The current limit
     * @throws DriverException
     */
    public synchronized void setCurrentLimit(double maxima) throws DriverException {
        checkOkToTalk();
        writeCommand("SOUR:VOLT:ILIM " + maxima);
    }

    /**
     * Gets the set current limit.
     *
     * @return The current limit
     * @throws DriverException
     */
    public synchronized double getCurrentLimit() throws DriverException {
        checkOkToTalk();
        return readDouble("SOUR:VOLT:ILIM?");
    }

    /**
     * Gets a double pair array from a string array.
     *
     * @param buff The string array
     * @return The returned array of double float values
     * @throws DriverException
     */
    private double[][] getDoublePairArray(String[] buff) throws DriverException {
        try {
            double[][] data = new double[2][buff.length / 2];
            int ridx = 0;
            for (int j = 0; j < buff.length; j += 2) {
                data[0][ridx] = Double.valueOf(buff[j]);
                data[1][ridx] = Double.valueOf(buff[j + 1]);
                ridx++;
            }
            return data;
        } catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }

    /**
     * Prints debug output.
     * 
     * @param  text  The text to print
     */
    private void debugPrint(String text) {
        if (debug) {
            System.out.println(text);
        }
    }

    /**
     * Checks whether it's okay to talk (i.e. not in buffered read)
     */
    private void checkOkToTalk() throws DriverException {
        if (!okToTalk) {
            throw new DriverException("Buffered read in progress");
        }
    }

    /**
     * Writes a command.
     *
     * @param instr The command to write, excluding terminator
     * @throws DriverException
     */
    public synchronized void writeKthly(String instr) throws DriverException {
        checkOkToTalk();
        writeCommand(instr);
    }

    /**
     * Writes a command and reads the (string) result.
     *
     * @param instr The command to write, excluding terminator
     * @return The returned string
     * @throws DriverException
     */
    public synchronized String readStringKthly(String instr) throws DriverException {
        checkOkToTalk();
        return readString(instr);
    }

    /*
     * Power supply driver interface routines.
     */
    @Override
    public double readVoltage(int chan) throws DriverException {
        return readVoltage();
    }

    @Override
    public double readCurrent(int chan) throws DriverException {
        return readCurrent();
    }

    @Override
    public void setVoltage(double value, int chan) throws DriverException {
        setVoltageRange(value);
        setVoltage(value);
    }

    @Override
    public double getVoltage(int chan) throws DriverException {
        return getVoltage();
    }

    @Override
    public void setCurrent(double value, int chan) throws DriverException {
        setCurrentLimit(value);
    }

    @Override
    public double getCurrent(int chan) throws DriverException {
        return getCurrentLimit();
    }

    @Override
    public void setOutput(boolean on, int chan) throws DriverException {
        setOutput(on);
    }

    @Override
    public boolean getOutput(int chan) throws DriverException {
        return getOutput();
    }

    @Override
    public void setOnDelay(double value, int chan) {
    }

    @Override
    public void setOffDelay(double value, int chan) {
    }

    /*
     * Deprecated routines
     */
    @Deprecated   // Use openSerial(devname, 0)
    public void open(String devname) throws DriverException {
        openSerial(devname, 0);
    }

    @Deprecated   // Default where???
    public void open() throws DriverException {
        open(DEFAULT_DEV, 0);
    }

    @Deprecated   // Use openSerial
    public void open(String devname, int baud) throws DriverException {
        openSerial(devname, baud);
    }

    @Deprecated  // Use openFtdi
    public void openftdi(String serial, int baud) throws DriverException {
        openFtdi(serial, baud);
    }

    @Deprecated
    public synchronized void accumBuffer(int nreads, double nplc, boolean wait) throws DriverException {
        accumBuffer(nreads, nplc);
    }

    @Deprecated
    public boolean isACCUM_IN_PROGRESS() {
        return isAccumInProgress();
    }

    @Deprecated
    public boolean isDATAREADY() {
        return isDataReady();
    }

    @Deprecated
    public boolean isOKTOTALK() {
        return isOkToTalk();
    }

    @Deprecated
    public void setOKTOTALK(boolean okToTalk) {
        this.okToTalk = okToTalk;
    }

    @Deprecated
    public boolean current_OK() throws DriverException {
        return isCurrentOk();
    }

    @Deprecated
    public void rampVolts_sweep(double duration, double value, int nsteps) throws DriverException {
        rampVoltsSweep(duration, value, nsteps);
    }

    @Deprecated
    public static double getMINBIAS() {
        return getMinBias();
    }

    @Deprecated
    public static double getMAXBIAS() {
        return getMaxBias();
    }

}
