package org.lsst.ccs.drivers.keithley;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the Keithley model 2231A device driver
 * 
 *  @author Owen Saxton
 */
public class Test2231A extends TestScpi {

    private final Model2231A ps;
    private int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


    /**
     *  Constructor
     */
    public Test2231A()
    {
        super(new Model2231A());
        ps = (Model2231A)scpi;
    }


    /**
     *  Sets the channel.
     * 
     *  @param  chan  The channel number
     */
    @Command(name="setchannel", description="Set the current channel")
    public void setChannel(@Argument(description="Channel number")
                           int chan)
    {
        chanNum = chan;
    }


    /**
     *  Shows the set channel.
     * 
     *  @return  The result string
     */
    @Command(name="showchannel", description="Show the set channel")
    public String showChannel()
    {
        return "Channel = " + chanNum;
    }


    /**
     *  Sets the voltage.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(description="Voltage to set")
                           double value) throws DriverException
    {
        ps.setVoltage(value, chanNum);
    }


    /**
     *  Sets the voltage.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(description="Voltage to set")
                           double value,
                           @Argument(description="Channel number")
                           int chan) throws DriverException
    {
        ps.setVoltage(value, chan);
    }


    /**
     *  Sets all the voltages.
     * 
     *  @param  values  The values to set
     *  @throws  DriverException
     */
    @Command(name="setvoltages", description="Set all the voltages")
    public void setVoltages(@Argument(description="Voltages to set")
                            double... values) throws DriverException
    {
        ps.setVoltage(values);
    }


    /**
     *  Sets the voltage to the maximum.
     * 
     *  @throws  DriverException
     */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage() throws DriverException
    {
        ps.setVoltage(Model2231A.MAX_VALUE, chanNum);
    }


    /**
     *  Sets the voltage to the maximum.
     * 
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        ps.setVoltage(Model2231A.MAX_VALUE, chan);
    }


    /**
     *  Shows the maximum voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage() throws DriverException
    {
        return "Maximum voltage = " + ps.getMaximumVoltage(chanNum);
    }


    /**
     *  Shows the maximum voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage(@Argument(description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum voltage = " + ps.getMaximumVoltage(chan);
    }


    /**
     *  Shows the set voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage() throws DriverException
    {
        return "Voltage = " + ps.getVoltage(chanNum);
    }


    /**
     *  Shows the set voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + ps.getVoltage(chan);
    }


    /**
     *  Shows all the set voltage(s).
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltages", description="Show all the set voltages")
    public String showVoltages() throws DriverException
    {
        return "Voltages = " + getString(ps.getVoltage());
    }


    /**
     *  Reads the actual voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + ps.readVoltage(chanNum);
    }


    /**
     *  Reads the actual voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + ps.readVoltage(chan);
    }


   /**
     *  Sets the soft voltage limit.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(description="Voltage limit to set")
                                double value) throws DriverException
    {
        ps.setVoltageLimit(value, chanNum);
    }


    /**
     *  Sets the soft voltage limit.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(description="Voltage limit to set")
                                double value,
                                @Argument(description="Channel number")
                                int chan) throws DriverException
    {
        ps.setVoltageLimit(value, chan);
    }


    /**
     *  Shows the soft voltage limit.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + ps.getVoltageLimit(chanNum);
    }


    /**
     *  Shows the soft voltage limit.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit(@Argument(description="Channel number")
                                   int chan) throws DriverException
    {
        return "Voltage limit = " + ps.getVoltageLimit(chan);
    }


    /**
     *  Sets the current.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(description="Current to set")
                           double value) throws DriverException
    {
        ps.setCurrent(value, chanNum);
    }


    /**
     *  Sets the current.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(description="Current to set")
                           double value,
                           @Argument(description="Channel number")
                           int chan) throws DriverException
    {
        ps.setCurrent(value, chan);
    }


    /**
     *  Sets all the currents.
     * 
     *  @param  values  The values to set
     *  @throws  DriverException
     */
    @Command(name="setcurrents", description="Set all the currents")
    public void setCurrents(@Argument(description="Currents to set")
                            double... values) throws DriverException
    {
        ps.setCurrent(values);
    }


    /**
     *  Sets the current to the maximum.
     * 
     *  @throws  DriverException
     */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent() throws DriverException
    {
        ps.setVoltage(Model2231A.MAX_VALUE, chanNum);
    }


    /**
     *  Sets the current to the maximum.
     * 
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        ps.setVoltage(Model2231A.MAX_VALUE, chan);
    }


    /**
     *  Shows the maximum current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent() throws DriverException
    {
        return "Maximum current = " + ps.getMaximumCurrent(chanNum);
    }


    /**
     *  Shows the maximum current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent(@Argument(description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum current = " + ps.getMaximumCurrent(chan);
    }


    /**
     *  Shows the set current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent() throws DriverException
    {
        return "Current = " + ps.getCurrent(chanNum);
    }


    /**
     *  Shows the set current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + ps.getCurrent(chan);
    }


    /**
     *  Shows all the set currents.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showcurrents", description="Show the set current")
    public String showCurrents() throws DriverException
    {
        return "Currents = " + getString(ps.getCurrent());
    }


    /**
     *  Reads the actual current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + ps.readCurrent(chanNum);
    }


    /**
     *  Reads the actual current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + ps.readCurrent(chan);
    }


    /**
     *  Sets the output on delay.
     * 
     *  @param  delay  The delay before turning the output on (sec)
     *  @throws  DriverException
     */
    @Command(name="setondelay", description="Set output on delay")
    public void setOnDelay(@Argument(description="Output on delay")
                           double delay) throws DriverException
    {
        ps.setOnDelay(delay, chanNum);
    }


    /**
     *  Sets the output on delay.
     * 
     *  @param  delay  The delay before turning the output on (sec)
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setondelay", description="Set output on delay")
    public void setOnDelay(@Argument(description="Output on delay")
                           double delay,
                           @Argument(description="Channel number")
                           int chan) throws DriverException
    {
        ps.setOnDelay(delay, chan);
    }


    /**
     *  Sets the output off delay.
     * 
     *  @param  delay  The delay before turning the output off (sec)
     *  @throws  DriverException
     */
    @Command(name="setoffdelay", description="Set output off delay")
    public void setOffDelay(@Argument(description="Output off delay")
                            double delay) throws DriverException
    {
        ps.setOffDelay(delay, chanNum);
    }


    /**
     *  Sets the output off delay.
     * 
     *  @param  delay  The delay before turning the output off (sec)
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setoffdelay", description="Set output off delay")
    public void setOffDelay(@Argument(description="Output off delay")
                            double delay,
                            @Argument(description="Channel number")
                            int chan) throws DriverException
    {
        ps.setOffDelay(delay, chan);
    }


    /**
     *  Turns the output on or off.
     * 
     *  @param  state  The state to set: on or off
     *  @throws  DriverException
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        ps.setOutput(state == OnOff.ON, chanNum);
    }


    /**
     *  Turns the output on or off.
     * 
     *  @param  state  The state to set: on or off
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(description="Output state: on or off")
                          OnOff state,
                          @Argument(description="Channel number")
                          int chan) throws DriverException
    {
        ps.setOutput(state == OnOff.ON, chan);
    }


   /**
     *  Shows the output state.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(ps.getOutput(chanNum));
    }


    /**
     *  Shows the output state.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput(@Argument(description="Channel number")
                             int chan) throws DriverException
    {
        return "Output state = " + getOnOff(ps.getOutput(chan));
    }


    /**
     *  Locks the front panel.
     * 
     *  @throws  DriverException
     */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        ps.lockPanel(true);
    }



    /**
     *  Unlocks the front panel.
     * 
     *  @throws  DriverException
     */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        ps.lockPanel(false);
    }


    /**
     *  Performs timing measurements.
     *
     *  @param  chan  The channel to use
     *  @param  count  The number of times to read the voltage
     *  @throws  DriverException
     */
    @Command(name="timeread", description="Time voltage reading")
    public void timeRead(@Argument(description="Channel number")
                         int chan,
                         @Argument(description="Read count")
                         int count) throws DriverException
    {
        for (int j = 0; j < 4; j++) {
            double volts = 0.5 * j;
            ps.setVoltage(volts, chan);
            long startTime = System.currentTimeMillis();
            System.out.println("Voltage set: " + volts);
            for (int k = 0; k < count; k++) {
                double value = ps.readVoltage(chan);
                System.out.println("Voltage read at " + (System.currentTimeMillis() - startTime) + " ms: " + value);
            }
        }
    }

    /**
     *  Converts an array of numbers to a string.
     */
    private static StringBuilder getString(double[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }


    /**
     *  Converts an array of booleans to a string.
     */
    private static StringBuilder getString(boolean[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0] ? "on" : "off");
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j] ? "on" : "off");
        }

        return text;
    }


    /**
     *  Converts a boolean to on/off.
     */
    private static String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
