package org.lsst.ccs.drivers.keithley;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ******************************************************************************
 **
 **  Routines for controlling a Keithley N6517 series modular power supply
 **        (based on Owen Saxton's Agilent code)
 ** 
 **  @author Homer Neal
 **
 ******************************************************************************
 */

public class N6517 extends Scpi {

    public static double timeout = 60.; 
    public static final int
        DEFAULT_BAUD     = 19200;         // Keithley default baud
    public static final String
        DEFAULT_DEV     = "/dev/ttyUSB0"; // default serial device
    
    private static final double MINBIAS = -75.; // CCD protection values
    private static final double MAXBIAS = 5.0;
        


//    private int    VOLTAGE_RANGE = 10; // can be 10V, 50V or 500V
//    private double CURRENT_RANGE = 2.e-9;


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String devname) throws DriverException
    {
        open(devname, DEFAULT_BAUD);
        reset();
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection.using all default parameters
    **
    ***************************************************************************
    */
    public void open() throws DriverException
    {
        open(DEFAULT_DEV, DEFAULT_BAUD);
        setTimeout(timeout); // set SCPI timeout
        reset();
    }

   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String devname, int port) throws DriverException
    {
        open(Scpi.ConnType.SERIAL, devname, port);
        checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH,
                            "MODEL 648", Scpi.CHECK_STARTS_WITH);
        setTimeout(timeout); // set SCPI timeout
        reset();
    }

   /**
    ***************************************************************************
    **
    **  Opens an FTDI connection.
    **
    ***************************************************************************
    */
    public void openftdi(String serialname, int port) throws DriverException
    {
        open(Scpi.ConnType.FTDI, serialname, port);
        checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH,
                            "MODEL 651", Scpi.CHECK_STARTS_WITH);
        setTimeout(timeout); // set SCPI timeout
        reset();
    }
    
    public String printdevid() throws DriverException
    {
        String[] id = getIdentification();
        return "Successfully connected to:" +
               "\nManufacturer:   " + id[Scpi.IDENT_MANUFACTURER] +
               "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME] +
               "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER] +
               "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }
   /**
    ***************************************************************************
    **
    **  Reset
    **
    ***************************************************************************
    */
    public void reset() throws DriverException
    {
        writeKthly("*RST");
        writeKthly("ARM:COUN 1");
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeKthly("SOUR:VOLT:SWE:ABOR"); // stop any sweep any progress

    }


   /**
    ***************************************************************************
    **
    **  Turns output on or off.
    **
    ***************************************************************************
    */
    public void setOutput(boolean on) throws DriverException
    {
        writeKthly("SOUR:VOLT:STAT " + (on ? "ON" : "OFF"));
    }



   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    ***************************************************************************
    */
    public boolean getOutput() throws DriverException
    {
        return readIntegerKthly("SOUR:VOLT:STAT?") != 0;
    }



   /**
    ***************************************************************************
    **
    **  Sets the integration rate in the number of power line cycles
    **
    ***************************************************************************
    */
    public void setRate(int value) throws DriverException
    {
        writeKthly("CURR:NPLC " + value);
    }


   /**
    ***************************************************************************
    **
    **  Sets the ARM count
    **
    ***************************************************************************
    */
    public void setArmCount(int value) throws DriverException
    {
        //        System.out.println("sac");
        writeKthly("ARM:COUN " + value);
    }

   /**
    ***************************************************************************
    **
    **  Sets the trigger count
    **
    ***************************************************************************
    */
    public void setTrigCount(int value) throws DriverException
    {
        writeKthly("TRIG:COUN " + value);
    }

  /**
    ***************************************************************************
    **
    **  Sets the buffer size  (#readings up to 3000 for the 6517)
    **
    ***************************************************************************
    */
    public void setBuffSize(int value) throws DriverException
    {
        writeKthly("TRAC:POIN " + value);
    }
    
  /**
    ***************************************************************************
    **
    **  Clear the buffer
    **
    ***************************************************************************
    */
    public void clrBuff() throws DriverException
    {
        writeKthly("TRAC:CLE");
    }

   /**
    ***************************************************************************
    **
    **  Sets the voltage range.
    **
    ***************************************************************************
    */
    public void setVoltageRange(double value) throws DriverException
    {
        writeKthly("SOUR:VOLT:RANG " + value);
    }


    /**
    ***************************************************************************
    **
    **  Sets the voltage.
    **
    ***************************************************************************
    */
    public void setVoltage(double value) throws DriverException
    {
        double usedvalue = 0.0;
        if (value < MINBIAS || value > MAXBIAS) {
            System.out.println("Only values between -75.V and 5.V are allowed in order to protect the CCD!");
            throw new DriverException();
        } else {
            usedvalue = value; // I know there is a more direct way but I'm just being extra cautious
        }
        writeKthly("SOUR:VOLT " + usedvalue);
    }


    /**
    ***************************************************************************
    **
    **  Ramps the voltage to a desired level over a given duration.
    **
    ***************************************************************************
    */
    public void rampVolts_sweep(double duration, double value, int nsteps) throws DriverException
    {
        double vnow = readVoltage();
        double vstep = Math.abs(value - vnow) / (double)nsteps;
        double delta = duration / (double)nsteps;
        System.out.println("starting ramp from "+Double.toString(vnow) +
                " to "+Double.toString(value) +
                " with vstep = " + Double.toString(vstep) +
                " with tstep = " + Double.toString(delta));

        writeKthly("SOUR:VOLT " + value);
        writeKthly("SOUR:VOLT:SWE:STAR " + vnow);
        writeKthly("SOUR:VOLT:SWE:STOP " + value);
        writeKthly("SOUR:VOLT:SWE:STEP " + vstep);
        writeKthly("SOUR:VOLT:SWE:DEL " + delta);
        setArmCount(nsteps+1);
        writeKthly("FORM:ELEM READ");  // just read back the values (add ,VSO for voltage)
        writeKthly("SOUR:VOLT:SWE:INIT");
        writeKthly("SYST:ZCH OFF");

        System.out.println("started ramp");
        double[] buff = readDoubleArray("READ?"); // another approach
        System.out.println("ramp completed");
/*
        write("INIT");
        System.out.println("started");
        int iwaitsecs = 0;
        boolean sweeping = false;
        System.out.println("waiting");
        while (iwaitsecs < timeout) {
            sweeping = read("SOUR:VOLT:SWE:STAT?").matches("1");
            System.out.println("sweeping = " + sweeping);
            if (!sweeping) {
                break;
            }
            System.out.println("waiting a second");

            try {
                System.out.println("trying to wait");
                Thread.currentThread().sleep(1000);//sleep for 1000 ms
                System.out.println("done waiting");
            } catch (Exception ie) {
            }
            System.out.println("Waiting for sweep to complete.");
            iwaitsecs++;
        }
        // if still sweeping then abort
        if (sweeping) writeKthly("SOUR:VOLT:SWE:ABOR");
        setArmCount(1);
  */
   }

    /**
    ***************************************************************************
    **
    **  Ramps the voltage to a desired level over a given duration.
    **
    ***************************************************************************
    */
    public void rampVolts(double duration, double value, int nsteps) throws DriverException
    {
        double vnow = readVoltage();
        double vstep = (value - vnow) / (double)nsteps;
        int delta = (int)(1000. * duration / (double)nsteps); // delta in msecs
        System.out.println("ramp from "+Double.toString(vnow) +
                " to "+Double.toString(value) +
                " with vstep = " + Double.toString(vstep) +
                " with tstep = " + Integer.toString(delta));

        System.out.println("starting ramp");

        double v = vnow;
        for (int istep=0;istep<nsteps+1;istep++) {
            System.out.println("V = " + v);
            
            setVoltage(v);

            try {
                Thread.currentThread().sleep(delta);//sleep for delta ms
            } catch (Exception ie) {
            }
            v += vstep;
        }
        System.out.println("ramp completed");

    }

    /**
    ***************************************************************************
    **
    **  Ramps the voltage to a desired level over a given duration.
    **      using a default of 10 steps
    ***************************************************************************
    */
    public void rampVolts(double duration, double value) throws DriverException
    {
        int    nsteps = 10; // default to this. We'll see later if this should be an argument.
        rampVolts(duration, value, nsteps);
    }
    
   /**
    ***************************************************************************
    **
    **  Gets the set voltage.
    **
    ***************************************************************************
    */
    public double getVoltage() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT?");
    }


   /**
    ***************************************************************************
    **
    **  zero correct the voltage.
    **
    ***************************************************************************
    */
    public void zeroCorrectVoltage() throws DriverException
    {

        writeKthly("SYST:ZCH ON"); // enable zero check
        writeKthly("SYST:GUAR ON"); // enable guard
        writeKthly("FUNC 'VOLT'");
        writeKthly("SYST:ZCOR ON"); // perform zero correction
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    ***************************************************************************
    */
    public double readVoltage() throws DriverException
    {
        setArmCount(1);
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
        writeKthly("FORM:ELEM VSO");    // only return the current reading (not the timestamp etc...)
        return readDoubleKthly("READ?"); // read and return the result

    }



   /**
    ***************************************************************************
    **
    **  Sets the current range.
    **
    ***************************************************************************
    */
    public void setCurrentRange(double value) throws DriverException
    {
        writeKthly("FUNC 'CURR'");
        writeKthly("RANG " + value);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current.
    **
    ***************************************************************************
    */
    public void setCurrent(double value) throws DriverException
    {
        writeKthly("SOUR:CURR " + value);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current.
    **
    ***************************************************************************
    */
    public double getCurrent() throws DriverException
    {
        return readDoubleKthly("CURR?");
    }


   /**
    ***************************************************************************
    **
    **  zero correct the current.
    **
    ***************************************************************************
    */
    public void zeroCorrectCurrent() throws DriverException
    {

        /* the following is just for the zero correction */
        writeKthly("SYST:ZCH ON");    // enable zero checking
        writeKthly("FUNC 'CURR'");
        writeKthly("INIT");           // trigger reading for zero correction
        writeKthly("SYST:ZCOR:ACQ");  // set zero correction value
        writeKthly("SYST:ZCOR ON");   // turn zero correction on
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    ***************************************************************************
    */
    public double readCurrent() throws DriverException
    {
        setArmCount(1);
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
        writeKthly("FORM:ELEM READ");    // only return the current reading (not the timestamp etc...)
        return readDoubleKthly("READ?"); // read and return the result
    }





   /**
    ***************************************************************************
    **
    **  Sets the voltage limit.
    **
    ***************************************************************************
    */
    public void setVoltageLimit(double maxima) throws DriverException
    {
        
        writeKthly("SOUR:VOLT:LIM " + maxima);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set voltage limit.
    **
    ***************************************************************************
    */
    public double getVoltageLimit() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT:LIM?");
    }


    /**
    ***************************************************************************
    **
    **  Sets the current limit.
    **
    ***************************************************************************
    */
    public void setCurrentLimit(double maxima) throws DriverException
    {
        writeKthly("SOUR:VOLT:ILIM " + maxima);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current limit.
    **
    ***************************************************************************
    */
    public double getCurrentLimit() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT:ILIM?");
    }

    /* *******************************************************
    FORM:ELEM READ,TIME ' Select reading and timestamp.
' Store and Recall Readings:
TRIG:COUN 20 ' Set trigger model to take to 20
readings.
TRAC:POIN 20 ' Set buffer size to 20.
TRAC:FEED SENS ' Store raw input readings.
TRAC:FEED:CONT NEXT ' Start storing readings.
SYST:ZCH OFF ' Turn off zero check.
INIT ' Trigger readings.
TRAC:DATA? ' Request all stored readings.
    ********************************************************** */
/**
    ***************************************************************************
    **
    **  Reads the buffer.
    **
    ***************************************************************************
    */
    public double[] readBuffer() throws DriverException
    {
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeKthly("FORM:ELEM READ");  // just read back the values (add ,VSO for voltage)
        writeKthly("TRAC:FEED SENS"); // store raw input readings
        writeKthly("TRAC:FEED:CONT NEXT"); // set buffer control mode and start reading
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
        writeKthly("INIT");           // trigger readings
        double[] buff = readDoubleArray("TRAC:DATA?"); // request all stored readings
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        return buff;
    }


   /**
    ***************************************************************************
    **
    **  Writes a command.
    **
    ***************************************************************************
    */
    private void writeKthly(String instr) throws DriverException
    {
        writeCommand(instr);
    }



   /**
    ***************************************************************************
    **
    **  Writes a command and reads the (string) result.
    **
    ***************************************************************************
    */
    private String readStringKthly(String instr) throws DriverException
    {
        return readString(instr);
    }



   /**
    ***************************************************************************
    **
    **  Writes a command and reads the double result.
    **
    ***************************************************************************
    */
    private double readDoubleKthly(String instr) throws DriverException
    {
        return readDouble(instr);
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the integer result.
    **
    ***************************************************************************
    */
    private int readIntegerKthly(String instr) throws DriverException
    {
        return readInteger(instr);
    }
}