package org.lsst.ccs.drivers.keithley;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ******************************************************************************
 **
 ** Class for controlling the Keithley 6487 picoammeter and voltage source
 *
 *
 * @author Homer Neal *
 * *****************************************************************************
 */
public class N6487 extends Scpi {

    private double timeout = 10.;
    private static final int DEFAULT_BAUD = 19200;         // Keithley default baud
    private static final String DEFAULT_DEV = "/dev/ttyUSB0"; // default serial device

    private static final double MINBIAS = -75.; // CCD protection values
    private static final double MAXBIAS = 5.0;
    private static final double MAXCURRENT = 0.001;

    private static final boolean debug = true;
    private boolean OKTOTALK = true;            // blocking flag used during long operations
    private boolean DATAREADY = false;
    private boolean trip = false;
    private boolean abort = false;
    private boolean ACCUM_IN_PROGRESS;

    /**
     ***************************************************************************
     **
     ** Opens a connection.
     *
     * @param devname
     * **************************************************************************
     */
    public void open(String devname) throws DriverException {
        open(devname, DEFAULT_BAUD);
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens a connection.using all default parameters
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void open() throws DriverException {
        open(DEFAULT_DEV, DEFAULT_BAUD);
        setTimeout(timeout); // set SCPI timeout
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens a connection.
     *
     * **************************************************************************
     */
    public void open(String devname, int port) throws DriverException {
        open(Scpi.CONN_TYPE_SERIAL, devname, port);
        checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH,
                "MODEL 648", Scpi.CHECK_STARTS_WITH);
        setTimeout(timeout); // set SCPI timeout
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens an FTDI connection.
     *
     * @param serialname
     * @param port
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void openftdi(String serialname, int port) throws DriverException {
        open(Scpi.CONN_TYPE_FTDI, serialname, port);
        checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH,
                "MODEL 648", Scpi.CHECK_STARTS_WITH);
        setTimeout(timeout); // set SCPI timeout
//        reset();
    }

    public String printdevid() throws DriverException {
        String[] id = getIdentification();
        return "Successfully connected to:"
                + "\nManufacturer:   " + id[Scpi.IDENT_MANUFACTURER]
                + "\nModel name:     " + id[Scpi.IDENT_MODEL_NAME]
                + "\nSerial number:  " + id[Scpi.IDENT_SERIAL_NUMBER]
                + "\nF/W version:    " + id[Scpi.IDENT_FW_VERSION];
    }

    /**
     ***************************************************************************
     **
     ** Reset *
     * **************************************************************************
     */
    public void reset() throws DriverException {
        OKTOTALK = true; // clear any incomplete previous state
        writeKthly("*RST");
        writeKthly("ARM:COUN 1");
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeKthly("SOUR:VOLT:SWE:ABOR"); // stop any sweep any progress

    }

    public void softReset() throws DriverException {
        OKTOTALK = true; // clear any incomplete previous state
        writeKthly("ARM:COUN 1");
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        writeKthly("SOUR:VOLT:SWE:ABOR"); // stop any sweep any progress

    }

    /**
     ***************************************************************************
     **
     ** Clear Status *
     * **************************************************************************
     */
    public void clearStat() throws DriverException {
        writeKthly("*CLS");
    }

    /**
     ***************************************************************************
     **
     ** Turns output on or off. *
     * **************************************************************************
     */
    public void setOutput(boolean on) throws DriverException {
        writeKthly("SOUR:VOLT:STAT " + (on ? "ON" : "OFF"));
    }

    /**
     ***************************************************************************
     **
     ** Gets the output state. *
     * **************************************************************************
     */
    public boolean getOutput() throws DriverException {
        return readIntegerKthly("SOUR:VOLT:STAT?") != 0;
    }

    public boolean isTrip() {
        return trip;
    }

    public void setTrip(boolean trip) {
        this.trip = trip;
    }

    public boolean isAbort() {
        return abort;
    }

    public void setAbort(boolean abort) {
        this.abort = abort;
    }

    /**
     ***************************************************************************
     **
     ** Sets the integration rate in the number of power line cycles *
     * **************************************************************************
     */
    public void setRate(double value) throws DriverException {
        writeKthly("NPLC " + value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the ARM count *
     * **************************************************************************
     */
    public void setArmCount(int value) throws DriverException {
        String line = "ARM:COUN " + value;
//        System.out.println("SetArmCount line being sent is:(" + line + ")");
        writeKthly(line);
    }

    /**
     ***************************************************************************
     **
     ** Sets the trigger count *
     * **************************************************************************
     */
    public void setTrigCount(int value) throws DriverException {
        writeKthly("TRIG:COUN " + value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the trigger delay *
     * **************************************************************************
     */
    public void setTrigDelay(int value) throws DriverException {
        writeKthly("TRIG:DEL " + value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the buffer size (#readings up to 3000 for the 6487) *
     * **************************************************************************
     */
    public void setBuffSize(int value) throws DriverException {
        writeKthly("TRAC:POIN " + value);
    }

    /**
     ***************************************************************************
     **
     ** Clear the buffer *
     * **************************************************************************
     */
    public void clrBuff() throws DriverException {
        writeKthly("TRAC:CLE");
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage range. *
     * **************************************************************************
     */
    public void setVoltageRange(double value) throws DriverException {
        writeKthly("SOUR:VOLT:RANG " + value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage. *
     * **************************************************************************
     */
    public void setVoltage(double value) throws DriverException {
        double usedvalue = 0.0;
        if (value < MINBIAS || value > MAXBIAS) {
            System.out.println("Only values between -75.V and 5.V are allowed in order to protect the CCD!");
            throw new DriverException();
        } else {
            usedvalue = value; // I know there is a more direct way but I'm just being extra cautious
        }
        writeKthly("SOUR:VOLT " + usedvalue);
    }

    /**
     ***************************************************************************
     **
     ** Ramps the voltage to a desired level over a given duration. *
     * **************************************************************************
     */
    public void rampVolts_sweep(double duration, double value, int nsteps) throws DriverException {
        double vnow = readVoltage();
        double vstep = Math.abs(value - vnow) / (double) nsteps;
        double delta = duration / (double) nsteps;
        System.out.println("starting ramp from " + Double.toString(vnow)
                + " to " + Double.toString(value)
                + " with vstep = " + Double.toString(vstep)
                + " with tstep = " + Double.toString(delta));

        writeKthly("SOUR:VOLT " + value);
        writeKthly("SOUR:VOLT:SWE:STAR " + vnow);
        writeKthly("SOUR:VOLT:SWE:STOP " + value);
        writeKthly("SOUR:VOLT:SWE:STEP " + vstep);
        writeKthly("SOUR:VOLT:SWE:DEL " + delta);
        setArmCount(nsteps + 1);
        writeKthly("FORM:ELEM READ");  // just read back the values (add ,VSO for voltage)
        writeKthly("SOUR:VOLT:SWE:INIT");
        writeKthly("SYST:ZCH OFF");

        System.out.println("started ramp");
        double[] buff = readDoubleArray("READ?"); // another approach
        System.out.println("ramp completed");
        /*
         write("INIT");
         System.out.println("started");
         int iwaitsecs = 0;
         boolean sweeping = false;
         System.out.println("waiting");
         while (iwaitsecs < timeout) {
         sweeping = read("SOUR:VOLT:SWE:STAT?").matches("1");
         System.out.println("sweeping = " + sweeping);
         if (!sweeping) {
         break;
         }
         System.out.println("waiting a second");

         try {
         System.out.println("trying to wait");
         Thread.currentThread().sleep(1000);//sleep for 1000 ms
         System.out.println("done waiting");
         } catch (Exception ie) {
         }
         System.out.println("Waiting for sweep to complete.");
         iwaitsecs++;
         }
         // if still sweeping then abort
         if (sweeping) writeKthly("SOUR:VOLT:SWE:ABOR");
         setArmCount(1);
         */
    }

    public boolean current_OK() throws DriverException {
        return (this.readCurrent() < MAXCURRENT);
    }

    /**
     ***************************************************************************
     **
     ** Ramps the voltage to a desired level over a given duration. *
     * **************************************************************************
     */
    public void rampVolts(double duration, double value, int nsteps) throws DriverException {
        double vnow = readVoltage();
        double vstep = (value - vnow) / (double) nsteps;
        int delta = (int) (1000. * duration / (double) nsteps); // delta in msecs
        System.out.println("ramp from " + Double.toString(vnow)
                + " to " + Double.toString(value)
                + " with vstep = " + Double.toString(vstep)
                + " with tstep = " + Integer.toString(delta));

        System.out.println("starting ramp");

        double v = vnow;
        boolean trip0 = isTrip();
        OKTOTALK = true; // override
        for (int istep = 0; istep < nsteps + 1; istep++) {
            System.out.println("V = " + v);

            if ((!trip0 && isTrip()) || isAbort()) {
                System.out.println("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                break;
            } else {
                if (current_OK()) {
                    setVoltage(v);
                } else {
                    System.out.println("driver aborting ramp due to high current!");
                    this.setAbort(true);
                }
            }

            try {
                Thread.currentThread().sleep(delta);//sleep for delta ms
            } catch (Exception ie) {
            }
            v += vstep;
        }
        System.out.println("ramp completed");

    }

    /**
     ***************************************************************************
     **
     ** Ramps the voltage to a desired level over a given duration. using a
     * default of 10 steps
     *
     * @param duration
     * @param value
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void rampVolts(double duration, double value) throws DriverException {
        int nsteps = 10; // default to this. We'll see later if this should be an argument.
        rampVolts(duration, value, nsteps);
    }

    /**
     ***************************************************************************
     **
     ** Gets the set voltage.
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public double getVoltage() throws DriverException {
        return readDoubleKthly("SOUR:VOLT?");
    }

    /**
     ***************************************************************************
     **
     ** zero correct the voltage. *
     * **************************************************************************
     */
    public void zeroCorrectVoltage() throws DriverException {

        writeKthly("SYST:ZCH ON"); // enable zero check
        writeKthly("SYST:GUAR ON"); // enable guard
        writeKthly("FUNC 'VOLT'");
        writeKthly("SYST:ZCOR ON"); // perform zero correction
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }

    /**
     ***************************************************************************
     **
     ** Reads the voltage. *
     * **************************************************************************
     */
    public double readVoltage() throws DriverException {
        if (OKTOTALK) {
            setArmCount(1);
            writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
            writeKthly("FORM:ELEM VSO");    // only return the current reading (not the timestamp etc...)
            return readDoubleKthly("READ?"); // read and return the result
        } else {
            throw new DriverException("Can't read voltage because buffered read is in progress.");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the current range. *
     * **************************************************************************
     */
    public void setCurrentRange(double value) throws DriverException {
        writeKthly("FUNC 'CURR'");
        writeKthly("RANG " + value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the current. *
     * **************************************************************************
     */
    public void setCurrent(double value) throws DriverException {
        writeKthly("SOUR:CURR " + value);
    }

    /**
     ***************************************************************************
     **
     ** Gets the set current. *
     * **************************************************************************
     */
    public double getCurrent() throws DriverException {
        if (OKTOTALK) {
            return readDoubleKthly("CURR?");
        } else {
            throw new DriverException("Can't read set current because buffered read is in progress.");
        }
    }

    /**
     ***************************************************************************
     **
     ** zero correct the current. *
     * **************************************************************************
     */
    public void zeroCorrectCurrent() throws DriverException {

        /* the following is just for the zero correction */
        writeKthly("SYST:ZCH ON");    // enable zero checking
        writeKthly("FUNC 'CURR'");
        writeKthly("INIT");           // trigger reading for zero correction
        writeKthly("SYST:ZCOR:ACQ");  // set zero correction value
        writeKthly("SYST:ZCOR ON");   // turn zero correction on
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }

    /**
     ***************************************************************************
     **
     ** Reads the current. *
     * **************************************************************************
     */
    public double readCurrent() throws DriverException {
        if (OKTOTALK) {
            setArmCount(1);
            writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
            writeKthly("FORM:ELEM READ");    // only return the current reading (not the timestamp etc...)
            return readDoubleKthly("READ?"); // read and return the result
        } else {
            throw new DriverException("Can't read current because buffered read is in progress.");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage limit. *
     * **************************************************************************
     */
    public void setVoltageLimit(double maxima) throws DriverException {

        writeKthly("SOUR:VOLT:LIM " + maxima);
    }

    /**
     ***************************************************************************
     **
     ** Gets the set voltage limit. *
     * **************************************************************************
     */
    public double getVoltageLimit() throws DriverException {
        return readDoubleKthly("SOUR:VOLT:LIM?");
    }

    /**
     ***************************************************************************
     **
     ** Sets the current limit. *
     * **************************************************************************
     */
    public void setCurrentLimit(double maxima) throws DriverException {
        writeKthly("SOUR:VOLT:ILIM " + maxima);
    }

    /**
     ***************************************************************************
     **
     ** Gets the set current limit. *
     * **************************************************************************
     */
    public double getCurrentLimit() throws DriverException {
        return readDoubleKthly("SOUR:VOLT:ILIM?");
    }

    /**
     ***************************************************************************
     **
     ** Sets the display on(true)/off(false)
     *
     *
     * @param dstate
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void setDisplay(boolean dstate) throws DriverException {
        writeKthly("DISP:ENAB " + (dstate ? "ON" : "OFF"));
    }

    /**
     ***************************************************************************
     **
     ** starts the accumulation of readings in the buffer.
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void accumBuffer(int nreads, double nplc, boolean wait) throws DriverException {
        OKTOTALK = false;
        DATAREADY = false;
        ACCUM_IN_PROGRESS = false;

        try {
            Thread.currentThread().sleep(1000);//sleep for 1000 ms
        } catch (InterruptedException ex) {
            System.out.println("Cannot sleep because of exception: " + ex);
        }
        System.out.println("Waiting for any previous operations to complete.");

        int eachnreads = nreads;
        if (nreads > 2048) {
            if (wait) {
                eachnreads = eachnreads / 2;
                System.out.println("Requested nreads over 2048, splitting into two reads of " + eachnreads);
            } else {
                eachnreads = 2048;
                System.out.println("Requested nreads over 2048, and wait mode not specified, thus nreads set to " + eachnreads);
            }
        }
        setArmCount(eachnreads);

        setRate(nplc);

        readIntegerKthly("*OPC?");  // any other operations to complete
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        clrBuff();
        writeKthly("FORM:ELEM READ,TIME");  // read back current and time (add ,VSO for voltage)
        clearStat();
        writeKthly("SYST:ZCH OFF");         // Turn zero check off
        writeKthly("SYST:AZER:STAT OFF");   // Turn auto-zero off
        writeKthly("TRAC:TST:FORM ABS");    // select DELT vs. ABS time format
        writeKthly("TRAC:FEED SENS");       // store raw input readings
        writeKthly("TRAC:FEED:CONT NEXT");  // set buffer control mode and start reading
        writeKthly("STAT:MEAS:ENAB 512");   // Enable buffer full event
        writeKthly("*SRE 1");               // Enable SRQ on buffer full event
//        writeKthly("DISP:ENAB OFF");
        readIntegerKthly("*OPC?");  // wait for setup to complete
        if (wait) { // we're going asynch and we can do more this way
            if (nreads <= 2048) { // simple case
                System.out.format("Sending PD accumulation trigger at %14.3f", System.currentTimeMillis() / 1000.);
                ACCUM_IN_PROGRESS = true;
                writeKthly("INIT");           // trigger readings
            } else {
                System.out.format("Sending first PD accumulation trigger at %14.3f", System.currentTimeMillis() / 1000.);
                ACCUM_IN_PROGRESS = true;
                writeKthly("INIT"); // trigger the rest
//            int numread = readIntegerKthly("TRAC:POIN:ACT?");
//            System.out.println("AccumBuff:Number of elements read = " + numread);
                System.out.format("Sending second PD accumulation trigger at %14.3f", System.currentTimeMillis() / 1000.);
                writeKthly("INIT"); // trigger the rest
            }
        } else { // the following will not wait for the readings to be ready
            System.out.format("Sending PD accumulation trigger at %14.3f", System.currentTimeMillis() / 1000.);
            ACCUM_IN_PROGRESS = true;
            write("INIT");           // trigger readings
        }
        DATAREADY = true;

    }

    /**
     ***************************************************************************
     **
     ** Reads the buffer. *
     * **************************************************************************
     */
    public double[][] readBuffer() throws DriverException {
        int krdy = 0;
        System.out.println("Beginning of readBuffer: DATAREADY = " + DATAREADY + " krdy = " + krdy);
        while (DATAREADY == false || krdy != 1) { // this should have been set false by accumbuffer
            System.out.println("In loop of readBuffer: DATAREADY = " + DATAREADY + " krdy = " + krdy);
            try {
                krdy = readIntegerKthly("*OPC?");  // wait for read operations to complete
            } catch (Exception ex) {
                krdy = 0;                         // obviously we are not ready yet
            }
            System.out.println("krdy = " + krdy);
            if (DATAREADY == false || krdy != 1) {
                System.out.println("sleep for a second then recheck.");
                try {
                    Thread.currentThread().sleep(1000);//sleep for 1000 ms
                } catch (InterruptedException ex) {
                    System.out.println("Cannot sleep because of exception: " + ex);
                }
                System.out.println("Waiting for data to become available. This is not an expected situation.");
            }
            // in case conditions are never satisfied, count on the timeout to free us
        }
        ACCUM_IN_PROGRESS = false;

        int numread = readIntegerKthly("TRAC:POIN:ACT?");
        System.out.println("Number of elements read = " + numread);
        if (debug) {
            System.out.println("About to do double pair array read.");
        }

//        writeKthly("DISP:ENAB ON");
//        if (debug) {
//            System.out.println("Calling io flush");
//        }
//        flush();
        if (debug) {
            System.out.println("Calling read dbl pair array");
        }
        double[][] buff = readDoublePairArray("TRAC:DATA?"); // request all stored readings
        if (debug) {
            System.out.println("Stop accumulation");
        }
        writeKthly("TRAC:FEED:CONT NEVER"); // stop accumulation and avoid the error -- 800,Illegal with storage active
        if (debug) {
            System.out.println("Reenable display");
        }
//        writeKthly("DISP:ENAB ON");
        if (debug) {
            System.out.println("Return buff");
        }
        OKTOTALK = true;
        DATAREADY = false;
        return buff;
    }

    public static double getMINBIAS() {
        return MINBIAS;
    }

    public static double getMAXBIAS() {
        return MAXBIAS;
    }

    public void setOKTOTALK(boolean OKTOTALK) {
        this.OKTOTALK = OKTOTALK;
    }

    public boolean isOKTOTALK() {
        return OKTOTALK;
    }

    public boolean isDATAREADY() {
        return DATAREADY;
    }

    public boolean isACCUM_IN_PROGRESS() {
        return ACCUM_IN_PROGRESS;
    }

    
    
    /**
     **************************************************************************
     **
     ** Reads a double pair array with SCPI error checking after writing a
     * command. * * @param command The command to write, excluding terminator
     *
     *
     * * @return The returned array of double float values * * @throws
     * DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public double[][] readDoublePairArray(String command) throws DriverException {
        try {
            if (debug) {
                System.out.println("Doing String Array read.");
            }
            String[] reply = readStringArray(command);
            if (debug) {
                /*
                 for (String str : reply) {
                 System.out.println("reply string = " + str);
                 }
                 */
                System.out.println("Retrieved string of values from buffer. length = " + reply.length);
            }
            double[][] dReply = new double[2][reply.length / 2];
            int ridx = 0;
            for (int j = 0; j < reply.length; j += 2) {
//                System.out.println("reply = (" + reply[j] + ")");
                dReply[0][ridx] = Double.valueOf(reply[j]);
                dReply[1][ridx] = Double.valueOf(reply[j + 1]);
                ridx++;
            }
            if (debug) {
                System.out.println("Binary array prepared.");
            }
            OKTOTALK = true;
            return dReply;
        } catch (NumberFormatException e) {
            throw new DriverException(e);
        }

    }

    /**
     ***************************************************************************
     **
     ** Writes a command. *
     * **************************************************************************
     */
    public void writeKthly(String instr) throws DriverException {
        if (!OKTOTALK) {
            System.out.println("writeK:Sending command (" + instr + ") while OKTOTALK = " + OKTOTALK);
        }
        writeCommand(instr);
    }

    /**
     ***************************************************************************
     **
     ** Writes a command and reads the (string) result. *
     * **************************************************************************
     */
    public String readStringKthly(String instr) throws DriverException {
        if (!OKTOTALK) {
            System.out.println("readStringK: Sending command (" + instr + ") while OKTOTALK = " + OKTOTALK);
        }
        return readString(instr);
    }

    /**
     ***************************************************************************
     **
     ** Writes a command and reads the double result. *
     * **************************************************************************
     */
    private double readDoubleKthly(String instr) throws DriverException {
        if (!OKTOTALK) {
            System.out.println("readDblK:Sending command (" + instr + ") while OKTOTALK = " + OKTOTALK);
        }
        return readDouble(instr);
    }

    /**
     ***************************************************************************
     **
     ** Writes a command and reads the integer result. *
     * **************************************************************************
     */
    private int readIntegerKthly(String instr) throws DriverException {
        if (!OKTOTALK) {
            System.out.println("readIK:Sending command (" + instr + ") while OKTOTALK = " + OKTOTALK);
        }
        return readInteger(instr);
    }
}
