package org.lsst.ccs.drivers.keithley;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ******************************************************************************
 **
 **  Routines for controlling a Keithley N6487 series modular power supply
 **        (based on Owen Saxton's Agilent code)
 ** 
 **  @author Homer Neal
 **
 ******************************************************************************
 */

public class N6517 extends Scpi {

    public static final int
        DEFAULT_BAUD     = 9600;          // Keithley default baud
    public static final String
        DEFAULT_DEV     = "/dev/ttyUSB0"; // default serial device

//    private int    VOLTAGE_RANGE = 10; // can be 10V, 50V or 500V
//    private double CURRENT_RANGE = 2.e-9;


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String devname) throws DriverException
    {
        open(devname, DEFAULT_BAUD);
        writeKthly("ARM:COUN 1");
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection.using all default parameters
    **
    ***************************************************************************
    */
    public void open() throws DriverException
    {
        open(DEFAULT_DEV, DEFAULT_BAUD);
        writeKthly("ARM:COUN 1");
    }

   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String devname, int port) throws DriverException
    {
        open(Scpi.CONN_TYPE_SERIAL, devname, port);
        checkIdentification("KEITHLEY", Scpi.CHECK_STARTS_WITH,
                            "MODEL 648", Scpi.CHECK_STARTS_WITH);
        writeKthly("ARM:COUN 1");
    }

   /**
    ***************************************************************************
    **
    **  Reset
    **
    ***************************************************************************
    */
    public void reset() throws DriverException
    {
        writeKthly("RST*");
        writeKthly("ARM:COUN 1");
    }


   /**
    ***************************************************************************
    **
    **  Turns output on or off.
    **
    ***************************************************************************
    */
    public void setOutput(boolean on) throws DriverException
    {
        writeKthly("SOUR:VOLT:STAT " + (on ? "ON" : "OFF"));
    }



   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    ***************************************************************************
    */
    public boolean getOutput() throws DriverException
    {
        return readIntegerKthly("SOUR:VOLT:STAT?") != 0;
    }



    
    

   /**
    ***************************************************************************
    **
    **  Sets the voltage range.
    **
    ***************************************************************************
    */
    public void setVoltageRange(double value) throws DriverException
    {
        writeKthly("SOUR:VOLT:RANG " + value);
    }


    /**
    ***************************************************************************
    **
    **  Sets the voltage.
    **
    ***************************************************************************
    */
    public void setVoltage(double value) throws DriverException
    {
        writeKthly("SOUR:VOLT " + value);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set voltage.
    **
    ***************************************************************************
    */
    public double getVoltage() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT?");
    }


   /**
    ***************************************************************************
    **
    **  zero correct the voltage.
    **
    ***************************************************************************
    */
    public void zeroCorrectVoltage() throws DriverException
    {

        writeKthly("SYST:ZCH ON"); // enable zero check
        writeKthly("SYST:GUAR ON"); // enable guard
        writeKthly("FUNC 'VOLT'");
        writeKthly("SYST:ZCOR ON"); // perform zero correction
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    ***************************************************************************
    */
    public double readVoltage() throws DriverException
    {

        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
        writeKthly("FORM:ELEM READ");    // only return the current reading (not the timestamp etc...)
        return readDoubleKthly("READ?"); // read and return the result

    }



   /**
    ***************************************************************************
    **
    **  Sets the current range.
    **
    ***************************************************************************
    */
    public void setCurrentRange(double value) throws DriverException
    {
        writeKthly("FUNC 'CURR'");
        writeKthly("RANG " + value);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current.
    **
    ***************************************************************************
    */
    public void setCurrent(double value) throws DriverException
    {
        writeKthly("SOUR:CURR " + value);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current.
    **
    ***************************************************************************
    */
    public double getCurrent() throws DriverException
    {
        return readDoubleKthly("CURR?");
    }


   /**
    ***************************************************************************
    **
    **  zero correct the current.
    **
    ***************************************************************************
    */
    public void zeroCorrectCurrent() throws DriverException
    {

        /* the following is just for the zero correction */
        writeKthly("SYST:ZCH ON");    // enable zero checking
        writeKthly("FUNC 'CURR'");
        writeKthly("INIT");           // trigger reading for zero correction
        writeKthly("SYST:ZCOR:ACQ");  // set zero correction value
        writeKthly("SYST:ZCOR ON");   // turn zero correction on
        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    ***************************************************************************
    */
    public double readCurrent() throws DriverException
    {

        writeKthly("SYST:ZCH OFF"); // important to turn this off before doing the read
        writeKthly("FORM:ELEM READ");    // only return the current reading (not the timestamp etc...)
        return readDoubleKthly("READ?"); // read and return the result
    }





   /**
    ***************************************************************************
    **
    **  Sets the voltage limit.
    **
    ***************************************************************************
    */
    public void setVoltageLimit(double maxima) throws DriverException
    {
        
        writeKthly("SOUR:VOLT:LIM " + maxima);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set voltage limit.
    **
    ***************************************************************************
    */
    public double getVoltageLimit() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT:LIM?");
    }


    /**
    ***************************************************************************
    **
    **  Sets the current limit.
    **
    ***************************************************************************
    */
    public void setCurrentLimit(double maxima) throws DriverException
    {
        writeKthly("SOUR:VOLT:ILIM " + maxima);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current limit.
    **
    ***************************************************************************
    */
    public double getCurrentLimit() throws DriverException
    {
        return readDoubleKthly("SOUR:VOLT:ILIM?");
    }


   /**
    ***************************************************************************
    **
    **  Writes a command.
    **
    ***************************************************************************
    */
    private void writeKthly(String instr) throws DriverException
    {
        writeCommand(instr);
    }



   /**
    ***************************************************************************
    **
    **  Writes a command and reads the (string) result.
    **
    ***************************************************************************
    */
    private String readStringKthly(String instr) throws DriverException
    {
        return readString(instr);
    }



   /**
    ***************************************************************************
    **
    **  Writes a command and reads the double result.
    **
    ***************************************************************************
    */
    private double readDoubleKthly(String instr) throws DriverException
    {
        return readDouble(instr);
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the integer result.
    **
    ***************************************************************************
    */
    private int readIntegerKthly(String instr) throws DriverException
    {
        return readInteger(instr);
    }
}