/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.drivers.iocard;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StreamTokenizer;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.PrimitiveIterator;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;

public final class PciCard {
    private final List<Resource> resources;
    private final int irq;
    private static final String PCI_DEVICES = "/sys/bus/pci/devices/";

    private PciCard(int irq, Stream<Resource> res) {
        this.irq = irq;
        this.resources = res.collect(Collectors.toList());
    }

    public List<Resource> getResources() {
        return Collections.unmodifiableList(this.resources);
    }

    public int getIrq() {
        return this.irq;
    }

    private static Stream<Resource> readResources(Reader rdr) {
        StreamTokenizer tkr = new StreamTokenizer(rdr);
        tkr.resetSyntax();
        tkr.eolIsSignificant(false);
        tkr.whitespaceChars(32, 32);
        tkr.whitespaceChars(10, 10);
        tkr.whitespaceChars(9, 9);
        tkr.wordChars(48, 57);
        tkr.wordChars(97, 102);
        tkr.wordChars(65, 70);
        tkr.wordChars(120, 120);
        tkr.wordChars(88, 88);
        LongStream.Builder lng = LongStream.builder();
        try {
            int tok = tkr.nextToken();
            while (tok != -1) {
                if (tok != -3) {
                    throw new Error("Badly formatted PCI resource description", null);
                }
                long val = Long.decode(tkr.sval);
                lng.accept(val);
                tok = tkr.nextToken();
            }
        }
        catch (IOException exc) {
            throw new Error("I/O error while reading PCI resource descriptions", exc);
        }
        Stream.Builder<Resource> res = Stream.builder();
        PrimitiveIterator.OfLong itr = lng.build().iterator();
        while (itr.hasNext()) {
            long bottom = (Long)itr.next();
            long top = (Long)itr.next();
            long flags = (Long)itr.next();
            if (flags == 0L) continue;
            res.accept(new Resource(bottom, top, flags));
        }
        return res.build();
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static Optional<PciCard> fromGeographicAddress(String geo) {
        try (FileReader irqrdr = new FileReader(PCI_DEVICES + geo + "/irq");){
            Optional<PciCard> optional;
            try (FileReader resrdr = new FileReader(PCI_DEVICES + geo + "/resource");){
                int irq = Integer.decode(new BufferedReader(irqrdr).readLine());
                Stream<Resource> res = PciCard.readResources(resrdr);
                optional = Optional.of(new PciCard(irq, res));
            }
            return optional;
        }
        catch (FileNotFoundException exc) {
            return Optional.empty();
        }
        catch (IOException exc) {
            throw new Error("Error reading description for " + geo, exc);
        }
        catch (NumberFormatException exc) {
            throw new Error("Error parsing IRQ number for " + geo, exc);
        }
    }

    public static Optional<PciCard> fromVendorId(int vid) {
        Optional<Path> pth = PciCard.pciDevices(dev -> PciCard.isFromVendor(dev, vid)).findFirst();
        if (pth.isPresent()) {
            return PciCard.fromGeographicAddress(PciCard.geographicAddress(pth.get()));
        }
        return Optional.empty();
    }

    private static Stream<Path> pciDevices(Predicate<Path> isWanted) {
        try {
            return Files.list(FileSystems.getDefault().getPath(PCI_DEVICES, new String[0])).filter(isWanted);
        }
        catch (IOException exc) {
            throw new Error("Error while scanning for PCI devices", exc);
        }
    }

    private static String geographicAddress(Path dev) {
        return dev.getName(dev.getNameCount() - 1).toString();
    }

    private static boolean isFromVendor(Path dev, int vid) {
        boolean bl;
        String geo = PciCard.geographicAddress(dev);
        FileReader vidrdr = new FileReader(dev.toString() + "/vendor");
        try {
            bl = vid == Integer.decode(new BufferedReader(vidrdr).readLine());
        }
        catch (Throwable throwable) {
            try {
                try {
                    vidrdr.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException exc) {
                throw new Error("Error reading vendor ID for " + geo, exc);
            }
            catch (NumberFormatException exc) {
                throw new Error("Error parsing vendor ID for " + geo, exc);
            }
        }
        vidrdr.close();
        return bl;
    }

    public String toString() {
        return Stream.concat(Stream.of(String.format("IRQ %d", this.irq)), this.resources.stream().map(Resource::toString)).collect(Collectors.joining("\n"));
    }

    public static void main(String[] arg) {
        Optional card1 = PciCard.fromGeographicAddress(arg[0]).flatMap(c -> Optional.of(c.toString()));
        Optional card2 = PciCard.fromVendorId(Integer.decode(arg[1])).flatMap(c -> Optional.of(c.toString()));
        System.out.println(card1.orElseGet(() -> "No PCI card at geo address " + arg[0]));
        System.out.println();
        System.out.println(card2.orElseGet(() -> "No PCI card from vendor " + arg[1]));
    }

    public static class Error
    extends RuntimeException {
        Error(String msg, Throwable originalCause) {
            super(msg, originalCause);
        }
    }

    public static final class Resource {
        private final long bottom;
        private final long top;
        private final long flags;

        public Resource(long bottom, long top, long flags) {
            this.bottom = bottom;
            this.top = top;
            this.flags = flags;
        }

        public long start() {
            return this.bottom;
        }

        public long size() {
            return this.top - this.bottom + 1L;
        }

        public boolean isIOSpace() {
            return 0L != (this.flags & 1L);
        }

        public String toString() {
            return String.format("%s space at %#10x, size %8d", this.isIOSpace() ? "I/O   " : "Memory", this.start(), this.size());
        }
    }
}

