package org.lsst.ccs.drivers.iocard.utility;

import java.time.Instant;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedTransferQueue;
import org.lsst.ccs.drivers.iocard.AccesDio;
import static org.lsst.ccs.drivers.iocard.AccesDio.INPUT_A;
import static org.lsst.ccs.drivers.iocard.AccesDio.INPUT_B;
import static org.lsst.ccs.drivers.iocard.AccesDio.INPUT_C_HIGH;
import static org.lsst.ccs.drivers.iocard.AccesDio.INPUT_C_LOW;
import static org.lsst.ccs.drivers.iocard.AccesDio.INTERRUPT_A;
import static org.lsst.ccs.drivers.iocard.AccesDio.INTERRUPT_B;
import static org.lsst.ccs.drivers.iocard.AccesDio.INTERRUPT_C;

/**
 * Prints to stdout each state change of an Access digital I/O card with a time stamp. It assumes
 * that only one PCI DIO card has been installed.
 * @author tether
 */
public final class MonitorDIO {

    private final static class RegisterData {
        public final int a, b, c;
	public final boolean poison;

        public RegisterData(int a, int b, int c, boolean poison) {
            this.a = a;
            this.b = b;
            this.c = c;
	    this.poison = poison;
        }
    }

    private final AccesDio dioCard;
    private int oldA, oldB, oldC;
    private final BlockingQueue<RegisterData> dataQueue;

    private MonitorDIO() {
        this.oldA = 0;
        this.oldB = 0;
        this.oldC = 0;
        this.dataQueue = new LinkedTransferQueue<>();
        this.dioCard = new AccesDio(-1, -1); // Auto-detect the card.
    }

    public void logStateChange(final int regA, final int regB, final int regC) {
        final String current = String.format("A %02x B %02x C %02x", regA, regB, regC);
        final String changes = String.format("A %02x B %02x C %02x",
                                             regA ^ oldA, regB ^ oldB, regC ^ oldC);
        System.out.format("State at %s %s changes %s%n", Instant.now(), current, changes);
        oldA = regA;
        oldB = regB;
        oldC = regC;
    }

    // Required signature void <name>(int xxx, Object yyy)
    private void stateChangeListener(int dummy1, Object dummy2) {
        // Logging is slow so hand off the data ASAP to the main thread in order to
        // avoid missing any changes.
        dataQueue.add(new RegisterData(
            dioCard.dioInp(AccesDio.PORTA_REG),
            dioCard.dioInp(AccesDio.PORTB_REG),
            dioCard.dioInp(AccesDio.PORTA_REG),
	    false));
    }

    public void attachToDIO() {
	// All registers will be inputs.
	dioCard.dioConfig(INPUT_A | INPUT_B | INPUT_C_LOW | INPUT_C_HIGH);
        // Initialize the "old" register state.
        oldA = dioCard.dioInp(AccesDio.PORTA_REG);
        oldB = dioCard.dioInp(AccesDio.PORTB_REG);
        oldC = dioCard.dioInp(AccesDio.PORTC_REG);
        // Handle state changes (interrupts).
        dioCard.attachInt(
	    INTERRUPT_A | INTERRUPT_B | INTERRUPT_C,
            this,                  // Call a method in this object.
            "stateChangeListener", // The method to call.
            null);                 // The method requires no argument.
        System.err.format("Configuration reg = %02x%n", dioCard.readB(AccesDio.CONF_REG));
    }

    public void detachFromDIO() {
        dioCard.detachInt();
    }

    public BlockingQueue<RegisterData> getDataQueue() {
        return dataQueue;
    }


    public static void main(String[] args) {
        final MonitorDIO monitor = new MonitorDIO();
        monitor.attachToDIO();

	// Handle ctrl-C using a shutdown hook.
        Runtime.getRuntime().addShutdownHook(new Thread() {
	    @Override
	    public void run() {
		monitor.getDataQueue().add(new RegisterData(0, 0, 0, true));
		// The JVM will terminate once all shutdown hooks finish,
		// so give the main thread time to clean up.
		try {
                    synchronized(monitor){monitor.wait();}
		}
		catch(InterruptedException exc)
		{}
	    }
	});

        System.err.println("Ready.");
        try {
            while (true) {
                final RegisterData data = monitor.getDataQueue().take();
		if (data.poison) break;
                monitor.logStateChange(data.a, data.b, data.c);
            }
        }
	catch(InterruptedException exc) {
        }
        finally {
            monitor.detachFromDIO();
	    System.err.println("\nDone.");
	    synchronized(monitor){monitor.notifyAll();}; // Let the shutdown hook exit if it's running.
        }
    }
}
