package org.lsst.ccs.drivers.iocard;

import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test the Java drive for digital I/O cards made by Acces I/O.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAccesDio implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_INIT        = 0,
        CMD_DIOCONF     = 1,
        CMD_DIOIN       = 2,
        CMD_DIOOUT      = 3,
        CMD_DIOINBIT    = 4,
        CMD_DIOOUTBIT   = 5,
        CMD_DIOSETBIT   = 6,
        CMD_DIOCLRBIT   = 7,
        CMD_DIOSHOW     = 8,
        CMD_INTATTACH   = 9,
        CMD_INTDETACH   = 10,
        CMD_INTSTART    = 11,
        CMD_INTSTOP     = 12,
        CMD_INTSHOW     = 13,
        CMD_SHOW        = 14,
        CMD_REGSHOW     = 15,
        NUM_CMDS        = 16;

    /*
    **  Command help
    */
    private final static String[] helpInit = {
        "Initialize Acces DIO card. Omit the parameters to scan the PCI device tree.",
        "init <addr> <irq>",
        "addr   The base address of the card (0x340 for ISA)",
        "irq    The IRQ level of the card (6 for ISA)",
    };

    private final static String[] helpShow = {
        "Show initialization parameters",
        "show",
    };

    private final static String[] helpRegshow = {
        "Show register contents",
        "regshow",
    };

    private final static String[] helpDioconf = {
        "Set digital I/O configuration register (set ",
        "dioconf <mask>",
        "mask    which are inputs: 1=C low, 2=B, 8=C high, 16=A",
    };

    private final static String[] helpDioin = {
        "Read and display a byte from a digital I/O port",
        "dioin <port>",
        "port    The digital I/O port to read (0=A, 1=B, 2=C)",
    };

    private final static String[] helpDioout = {
        "Write a byte to a digital I/O port",
        "dioout <port> <value>",
        "port    The digital I/O port to write to (0=A, 1=B, 2=C)",
        "value   The value to write",
    };

    private final static String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read (0=A, 1=B, 2=C)",
        "bitnum  The number of the bit to read",
    };

    private final static String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to (0=A, 1=B, 2=C)",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private final static String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to (0=A, 1=B, 2=C)",
        "bitnum  The number of the bit to set",
    };

    private final static String[] helpDioclrbit = {
        "Clear a bit in a digital I/O port",
        "dioclrbit <port> <bitnum>",
        "port    The digital I/O port to write to (0=A, 1=B, 2=C)",
        "bitnum  The number of the bit to clear",
    };

    private final static String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private final static String[] helpIntattach = {
        "Attach to interrupt processing",
        "intattach <pmask>",
        "pmask  A bit mask of the ports to trigger interrupts (1=A, 2=B, 4=C) (inverted on the card)",
    };

    private final static String[] helpIntdetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("init",       CMD_INIT,       helpInit,       "ii");
        cmnd.add("show",       CMD_SHOW,       helpShow,       "");
        cmnd.add("regshow",    CMD_REGSHOW,    helpRegshow,    "");
        cmnd.add("dioconf",    CMD_DIOCONF,    helpDioconf,    "i");
        cmnd.add("dioin",      CMD_DIOIN,      helpDioin,      "i");
        cmnd.add("dioout",     CMD_DIOOUT,     helpDioout,     "ii");
        cmnd.add("dioinbit",   CMD_DIOINBIT,   helpDioinbit,   "ii");
        cmnd.add("diooutbit",  CMD_DIOOUTBIT,  helpDiooutbit,  "iii");
        cmnd.add("diosetbit",  CMD_DIOSETBIT,  helpDiosetbit,  "ii");
        cmnd.add("dioclrbit",  CMD_DIOCLRBIT,  helpDioclrbit,  "ii");
        cmnd.add("dioshow",    CMD_DIOSHOW,    helpDioshow,    "");
        cmnd.add("intattach",  CMD_INTATTACH,  helpIntattach,  "i");
        cmnd.add("intdetach",  CMD_INTDETACH,  helpIntdetach,  "");
    }

    /*
    **  Private fields
    */
    private final Output out;
    private final AccesDio acc;
    private final CmndProcess proc;
    private final TestIocard tioc;

    private int   baseAddr = -1,  // Negative value prompts a PCI scan.
                  irqLevel = -1,
                  dioPort = 0,
                  dioConf = 0,
                  dioValue = 0,
                  dioBit = 0,
                  intPort = 0,
                  intMask = 0x07;



   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    private TestAccesDio(Output out, CmndProcess proc, TestIocard tioc, AccesDio acc)
    {
        this.out = out;
        this.proc = proc;
        this.tioc = tioc;
        this.acc = acc;
    }

    public static TestAccesDio newInstance(Output out) {
        final Output lout = (out != null) ? out : new ConsOut();
        final CmndProcess proc = new CmndProcess();
        final AccesDio acc = new AccesDio();
        final TestIocard tioc = TestIocard.newInstance(acc, proc, out);
        final TestAccesDio obj = new TestAccesDio(lout, proc, tioc, acc);
        return obj;
    }
    
    public void addAsDispatcher() {
        // This object must be added before the tioc object in order to override
        // commands of the latter.
        proc.add(this, cmnd);
        tioc.addAsDispatcher();
    }
    
    public static TestAccesDio newInstance() {return newInstance(null);}

   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        final TestAccesDio acc = newInstance();
        acc.addAsDispatcher();
        acc.tioc.run();
        System.exit(0); // Get around a bug in the command interpreter.
    }


   /**
    ***************************************************************************
    **
    **  Routine to dispatch a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        switch (code) {
        case CMD_INIT:
            procInit(found, args); break;
        case CMD_SHOW:
            procShow(found, args); break;
        case CMD_REGSHOW:
            procRegshow(found, args); break;
        case CMD_DIOCONF:
            procDioconf(found, args); break;
        case CMD_DIOIN:
            procDioin(found, args); break;
        case CMD_DIOOUT:
            procDioout(found, args); break;
        case CMD_DIOINBIT:
            procDioinbit(found, args); break;
        case CMD_DIOOUTBIT:
            procDiooutbit(found, args); break;
        case CMD_DIOSETBIT:
            procDiosetbit(found, args); break;
        case CMD_DIOCLRBIT:
            procDioclrbit(found, args); break;
        case CMD_DIOSHOW:
            procDioshow(found, args); break;
        case CMD_INTATTACH:
            procIntattach(found, args); break;
        case CMD_INTDETACH:
            procIntdetach(found, args); break;
        default:
            out.println("Command not fully implemented");
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the INIT command
    **
    ***************************************************************************
    */
    private void procInit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) baseAddr = (Integer)args[0]; else baseAddr = -1;
        if ((found & 0x02) != 0) irqLevel = (Integer)args[1]; else irqLevel = -1;
        acc.init(baseAddr, irqLevel);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOW command
    **
    ***************************************************************************
    */
    private void procShow(int found, Object[] args)
    {
        try {
            out.println("Card parameters:");
            out.format("  Base address     = %#6x\n", acc.getBase());
            out.format("  Card size        = %#6x\n", acc.getNregs());
            out.format("  IRQ level        = %6d\n",  acc.getIrq());
        }
        catch (NullPointerException exc) {out.println("Card not initialized!");}
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGSHOW command
    **
    ***************************************************************************
    */
    private final static String[] regDesc = {
        "| A7 | A6 | A5 | A4 | A3 | A2 | A1 | A0 | port bits",
        "| B7 | B6 | B5 | B4 | B3 | B2 | B1 | B0 | port bits",
        "| C7 | C6 | C5 | C4 | C3 | C2 | C1 | C0 | port bits",
        "|    |    |    |  A |C hi|    |  B |C lo| direction, 1 for input",
    };

    private void procRegshow(int found, Object[] args)
    {
        out.format("  %2s: %02x  %s\n", AccesDio.PORTA_REG, acc.readB(AccesDio.PORTA_REG), regDesc[0]);
        out.format("  %2s: %02x  %s\n", AccesDio.PORTB_REG, acc.readB(AccesDio.PORTB_REG), regDesc[1]);
        out.format("  %2s: %02x  %s\n", AccesDio.PORTC_REG, acc.readB(AccesDio.PORTC_REG), regDesc[2]);
        out.format("  %2s: %02x  %s\n", AccesDio.CONF_REG, acc.readB(AccesDio.CONF_REG), regDesc[3]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOCONF command
    **
    ***************************************************************************
    */
    private void procDioconf(int found, Object[] args)
    {
        final int enableDirectionSetting = 0x80;
        if ((found & 0x01) != 0) dioConf = (Integer)args[0];
        acc.dioConfig(dioConf | enableDirectionSetting);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOIN command
    **
    ***************************************************************************
    */
    private void procDioin(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        int value = acc.dioInp(dioPort);
        out.format("Value = 0x%02x\n", value);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOOUT command
    **
    ***************************************************************************
    */
    private void procDioout(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioValue = (Integer)args[1];
        acc.dioOut(dioPort, dioValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOINBIT command
    **
    ***************************************************************************
    */
    private void procDioinbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        int value = acc.dioInpBit(dioPort, dioBit);
        out.println("Value = " + value);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOOUTBIT command
    **
    ***************************************************************************
    */
    private void procDiooutbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        if ((found & 0x04) != 0) dioValue = (Integer)args[2];
        acc.dioOutBit(dioPort, dioBit, dioValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOSETBIT command
    **
    ***************************************************************************
    */
    private void procDiosetbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        acc.dioSetBit(dioPort, dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOCLRBIT command
    **
    ***************************************************************************
    */
    private void procDioclrbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        acc.dioClrBit(dioPort, dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOSHOW command
    **
    ***************************************************************************
    */
    private void procDioshow(int found, Object[] args)
    {
        out.println("Digital I/O parameters:");
        out.format("  Configuration = %02x\n", dioConf);
        out.println("  Port          = " + dioPort);
        out.format("  Output value  = %02x\n", dioValue);
        out.println("  Bit number    = " + dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTATTACH command
    **
    ***************************************************************************
    */
    private void procIntattach(int found, Object[] args)
    {
        if ((found & 0x01) != 0) intMask = (Integer)args[0];
        acc.attachInt(intMask, this, "reportInt", acc);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTDETACH command
    **
    ***************************************************************************
    */
    private void procIntdetach(int found, Object[] args)
    {
        acc.detachInt();
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to report interrupts
    **
    ***************************************************************************
    */
    private void reportInt(int value, Object parm)
    {
        AccesDio dio = (AccesDio)parm;

        out.format("Interrupt! A B C = %03x %03x %03x\n",
                   dio.readB(AccesDio.PORTA_REG),
                   dio.readB(AccesDio.PORTB_REG),
                   dio.readB(AccesDio.PORTC_REG));
    }

}
