package org.lsst.ccs.drivers.iocard;

/**
 ***************************************************************************
 **
 **
 **  Generic Java interface to digital I/O cards on an ISA or PCI/PCIe/mPCIe bus.
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
public class Iocard {

   /**
    ***************************************************************************
    **
    **  Various constants
    **
    ***************************************************************************
    */
    public final static int
        OPT_CHECK  = 0x01,   // Check if card actually caused interrupt.
        OPT_WRITER = 0x02,   // Clear interrupts by writing to a register.
        OPT_READV  = 0x04;   // Read a value from the card in the interrupt handler.

   /**
    ***************************************************************************
    **
    **  Private data
    **
    ***************************************************************************
    */
    private long handle = 0, intDesc = 0;


   /**
    ***************************************************************************
    **
    **  Static initializer
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Iocard");
        initSys();
    }


   /**
    ***************************************************************************
    **
    **  Initialize system access
    **
    ***************************************************************************
    */
    private native static void initSys();


   /**
    ***************************************************************************
    **
    **  Initiate access to a card
    **
    ***************************************************************************
    */
    public native void init(int baseAddr, int size, int irq)
        throws IocardException;


   /**
    ***************************************************************************
    **
    **  Terminate access to a card
    **
    ***************************************************************************
    */
    public native void term() throws IocardException;


   /**
    ***************************************************************************
    **
    **  Read a byte from a register
    **
    ***************************************************************************
    */
    public native int readB(int offs) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Read a short integer from a register
    **
    ***************************************************************************
    */
    public native int readW(int offs) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Read an integer from a register
    **
    ***************************************************************************
    */
    public native int readL(int offs) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Write a byte to a register
    **
    ***************************************************************************
    */
    public native void writeB(int offs, int value) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Write a short integer to a register
    **
    ***************************************************************************
    */
    public native void writeW(int offs, int value) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Write an integer to a register
    **
    ***************************************************************************
    */
    public native void writeL(int offs, int value) throws IocardException;


   /**
    ***************************************************************************
    **
    **  Update a byte in a register
    **
    ***************************************************************************
    */
    public native void updateB(int offs, int mask, int value)
                         throws IocardException;


   /**
    ***************************************************************************
    **
    **  Update a short integer in a register
    **
    ***************************************************************************
    */
    public native void updateW(int offs, int mask, int value)
                         throws IocardException;


   /**
    ***************************************************************************
    **
    **  Update an integer in a register
    **
    ***************************************************************************
    */
    public native void updateL(int offs, int mask, int value)
                         throws IocardException;


   /**
    ***************************************************************************
    **
    **  Attach to interrupts
    **
    ***************************************************************************
    */
    public native void attachInt(int id, int optns, int cOffset, int cMask,
                                 int rOffset, int rValue, int vOffset,
                                 Object cbObj, String cbMeth, Object cbParm)
                         throws IocardException;


   /**
    ***************************************************************************
    **
    **  Detach from interrupts
    **
    ***************************************************************************
    */
    public native void detachInt(int id) throws IocardException;
}
