package org.lsst.ccs.drivers.iocard;

import java.util.Optional;


/**
 * Provides an interface to a digital I/O card made by Acces I/O. There are three eight-bit port registers
 * which contain the state of the I/O lines: A, B and C. One can configure the direction of the
 * lines, input or output, independently for each of four groups of bits: all bits of A, all bits of B,
 * the low-order four bits of C and the high-order four bits of C. One can enable
 * change-of-line-state interrupts independently for three groups: all of A, all of B and all of C.
 * <p>
 * Note that values you write to the ports will read back correctly only in the bits
 * that are set as outputs. The interrupt setup register will always read back as zero.
 * <p>
 * Not all cards in this family allow you to read an interrupt status, so this feature isn't supported.
 * To track changes you have to save the old port values and compare them to the new ones just
 * after the interrupt.
 * <p>
 * Some cards offer two banks of port registers. For such cards we support only the first bank.
 * <p>
 * For the PCI/PCIe/mPCIe cards we can search for the first card we find that comes from Acces I/O.
 * For a particular card among several or for an ISA card you'll have to give the I/O base address
 * and the assigned IRQ.
 * <p>
 * This driver includes a Linux kernel module {@code iocardk.ko} which must be loaded in order
 * to operate the DIO card. The script {@code load.sh} can be run at boot time to load the module
 * and set the permissions for the device {@code /dev/iocard}. There is also a JNI library
 * {@code libIocard.so} which allows you to use the card from Java.
 * <p>
 * In order to handle interrupts, you register a Java method as a callback with the JNI library.
 * The library acknowledges each interrupt so your callback doesn't need to.
 * @author saxton
 * @author tether
 */
public final class AccesDio extends Iocard {

    /**
     * The number of registers in the old ISA cards.
     */
    public final static int
        ISA_N_REGS = 25;

    /**
     *  The number of registers in the PCI/PCIe/mPCIe cards
     */
    public final static int    PCI_N_REGS = 16;

    /**
     * The offset to the A register.
     */
    public final static int    PORTA_REG  = 0;

    /**
     * The offset to the B register.
     */
    public final static int    PORTB_REG  = 1;

    /**
     * The offset to the C register.
     */
    public final static int    PORTC_REG  = 2;

    /**
     * The offset to the I/O direction configuration register.
     */
    public final static int    CONF_REG   = 3;

    /**
     * The offset to the interrupt setup register.
     */
    public final static int    SETUP_REG  = 11;

    /**
     * The offset to the interrupt reset register.
     */
    public final static int    RESET_REG  = 15;

    /**
     * The offset to the interrupt status (read) and reset (write) register. ISA cards only.
     */
    public final static int    ISA_STATUS_AND_RESET_REG  = 24;

    /**
     * The vendor ID for PCI cards from Acces I/O.
     */
    public final static int    PCI_VENDOR_ID = 0x494f;

    /**
     * The minimum I/O port address assigned to PCI cards by the Linux kernel.
     */
    public final static int    PCI_MIN_IO_ADDR = 0xc000;

    /**
     * Indicates that the A register pins are inputs.
     * @see #dioConfig(int)
     */
    public final static int INPUT_A = 1;

    /**
     * Indicates that the B register pins are inputs.
     * @see #dioConfig(int)
     */
    public final static int INPUT_B = 2;

    /**
     * Indicates that low four C register pins are inputs.
     * @see #dioConfig(int)
     */
    public final static int INPUT_C_LOW = 4;

    /**
     * Indicates that the high four C register pins are inputs.
     * @see #dioConfig(int)
     */
    public final static int INPUT_C_HIGH = 8;

    /**
     * Indicates that register A pins are to generate change-of-state interrupts.
     * @see #attachInt(int, Object, String, Object)
     */
    public final static int INTERRUPT_A = 1;

    /**
     * Indicates that register B pins are to generate change-of-state interrupts.
     * @see #attachInt(int, Object, String, Object)
     */
    public final static int INTERRUPT_B = 2;

    /**
     * Indicates that register C pins are to generate change-of-state interrupts.
     * @see #attachInt(int, Object, String, Object)
     */
    public final static int INTERRUPT_C = 4;

    /**
     * Finds and initializes a digital I/O card.
     * @param base The I/O base address of the card. If the value is negative
     * and a PCI/PCIe/mPCIe bus is present then the bus is searched for
     * the first card from Acces I/O.
     * @param irq The IRQ line no. assigned to the card. Ignored if
     * a negative base address was given.
     */
    public AccesDio(final int base, final int irq)
    {
        this.card = new CardInfo(base, irq);
        super.init(card.base, card.nregs, card.irq);
    }

    /**
     * Constructs and instance but doesn't find or initialize a DIO card.
     * @deprecated
     */
    @Deprecated
    public AccesDio() {this.card = null;}

    private CardInfo card;

    /**
     * Finds and initializes a DIO card.
     */
    private final static class CardInfo {
        final int irq;   // IRQ no.
        final int base;  // I/O base address.
        final int nregs; // No. of I/O registers (all eight-bit).
        final boolean isPCI; // Is this a PCI/PCIe/mPCIe card?

        CardInfo(final int base, final int irq) {
            if (base >= 0) {
                // User specified the IRQ and base address.
                this.irq = irq;
                this.base = base;
                this.isPCI =  base >= PCI_MIN_IO_ADDR;
                this.nregs = isPCI ? PCI_N_REGS : ISA_N_REGS;
            }
            else {
                // Probe the PCI bus.
                Optional<PciCard> card = PciCard.fromVendorId(PCI_VENDOR_ID);
                if (card.isPresent()) {
                    // Found a PCI card. For Acces DIO cards the third resource has the port address.
                    this.irq = card.get().getIrq();
                    this.base = (int)card.get().getResources().get(2).start();
                    this.nregs = PCI_N_REGS;
                    this.isPCI = true;
                }
                else {
                    throw new PciCard.Error("Can't find an Acces PCI card. Provide I/O addr and IRQ", null);
                }
            }
        }
    }

    /**
     * Finds and initializes a DIO card.
     * @see #AccesDio(int, int)
     * @param base
     * @param irq
     * @deprecated
     */
    @Deprecated
    public void init(final int base, final int irq) {
        card = new CardInfo(base, irq);
        super.init(card.base, card.nregs, card.irq);
    }

    /**
     * Gets the IRQ line no. assigned to the card.
     * @return The IRQ assignment.
     */
    public int getIrq() {return card.irq;}

    /**
     * Gets the I/O space base address of the card.
     * @return The I/O address.
     */
    public int getBase() {return card.base;}

    /**
     * Gets the number of I/O space addresses assigned to the card.
     * @return The number of eight-bit registers.
     */
    public int getNregs() {return card.nregs;}

    /**
     * Specify which of the DIO card I/O lines are inputs. Any lines not
     * mentioned are configured as outputs.
     * @param inputLineMask The inclusive OR of any combination of 
     * {@link #INPUT_A}, {@link #INPUT_B}, {@link #INPUT_C_LOW} or {@link #INPUT_C_HIGH}.
     */
    public void dioConfig(final int inputLineMask)
    {
        writeB(CONF_REG, dioConf[inputLineMask & 0x0f]);
    }

    private final static int[] dioConf = {0x80, 0x90, 0x82, 0x92,
                                          0x81, 0x91, 0x83, 0x93,
                                          0x88, 0x98, 0x8a, 0x9a,
                                          0x89, 0x99, 0x8b, 0x9b};

    /**
     * Reads all bits from one of the data ports on the DIO card.
     * @param port One of {@link #PORTA_REG}, {@link #PORTB_REG} or {@link #PORTC_REG}.
     * @return The current value of the port register.
     * @exception AccesDioException if the port number is invalid.
     */
    public int dioInp(final int port)
    {
        if (port < 0 || port > 2) {throwInvPort();}
        return readB(port);
    }

    /**
     * Writes all bits to one of the data ports on the DIO card.
     * @param port One of {@link #PORTA_REG}, {@link #PORTB_REG} or {@link #PORTC_REG}.
     * @param value The value to be written, {@code 0 <= value <= 255}.
     * @exception AccesDioException if the port number is invalid.
     */
    public void dioOut(final int port, final int value)
    {
        if (port < 0 || port > 2) {throwInvPort();}
        writeB(port, value);
    }

    /**
     * Reads one of the data ports on the DIO card and returns the value of the selected bit.
     * @param port One of {@link #PORTA_REG}, {@link #PORTB_REG} or {@link #PORTC_REG}.
     * @param bit The bit index, {@code 0 <= bit <= 7}.
     * @return {@code (regValue >> bit) & 1}
     * @exception AccesDioException if the port number is invalid.
     */
    public int dioInpBit(final int port, final int bit)
    {
        if (port < 0 || port > 2) {throwInvPort();}
        return (readB(port) >> bit) & 1;
    }

   /**
     * Uses read-modify-write to update the value of the selected bit in a data port.
     * Afterward {@link #dioInpBit(int, int)} with the same port and bit numbers
     * will return the new bit value.
     * @param port One of {@link #PORTA_REG}, {@link #PORTB_REG} or {@link #PORTC_REG}.
     * @param bit The bit index, {@code 0 <= bit <= 7}.
     * @param value The new bit value, 0 or 1.
     * @exception AccesDioException if the port number is invalid.
     */
    public void dioOutBit(final int port, int bit, final int value)
    {
        if (port < 0 || port > 2) {throwInvPort();}
        updateB(port, 1 << bit, value << bit);
    }

    /**
     * Equivalent to {@code dioOutBit(port, bit, 1)}.
     * @param port
     * @param bit 
     * @exception AccesDioException if the port number is invalid.
     * @see #dioOutBit(int, int, int) 
     */
    public void dioSetBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        int value = 1 << bit;
        updateB(port, value, value);
    }

    /**
     * Equivalent to {@code dioOutBit(port, bit, 0)}.
     * @param port
     * @param bit
     * @exception AccesDioException if the port number is invalid.
     * @see #dioOutBit(int, int, int)
     */
    public void dioClrBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        updateB(port, 1 << bit, 0);
    }

   /**
    ***************************************************************************
    **
    **  Register a Java callback for interrupt handling. Only a single callback
    *   may be registered at any given time.
    **  @param mask Specifies which ports are to be sources of interrupts. The logical
    **  OR of any combination of {@link #INTERRUPT_A}, {@link #INTERRUPT_B} or {@link #INTERRUPT_C}.
    **  @param cbObj The object containing the method to be called.
    **  @param cbMeth The name of the method to call on {@code cbObj} when an interrupt occurs.
    **  @param cbParm The argument to be given to {@code cbObj.cbMeth()}
    **
    **
    ***************************************************************************
    */
    public void attachInt(int mask, Object cbObj, String cbMeth, Object cbParm)
    {
        // Clearing interrupts:
        // ISA: Read from ISA_STATUS_AND_RESET_REG to get and reset interrupt status.
        // PCI: Write anything to RESET_REG.
        if (card.isPCI) {
            super.attachInt
                (0,                        // A unique ID no. for this attachment.
                 OPT_WRITER,               // Write to a register to reset interrupt status.
                 0,                        // Unused.
                 0,                        // Unused.
                 RESET_REG,                // The register to write to reset interrupt status.
                 0,                        // The value to write.
                 0,                        // Unused.
                 cbObj, cbMeth, cbParm);
        }
        else {
            // The iocardk kernal module's interrupt handler won't read a register
            // twice if it's both the check and read-to-reset register,
            super.attachInt
                (0,                        // A unique ID no. for this attachment.
                 OPT_CHECK,                // Check card interrupt status in the iocard handler.
                 ISA_STATUS_AND_RESET_REG, // The register to read for the check.
                 0x0f,                     // The mask to apply to the register value before checking.
                 ISA_STATUS_AND_RESET_REG, // The register to read to reset interrupt status.
                 0,                        // Unused.
                 0,                        // Unused.
                 cbObj, cbMeth, cbParm);
        }
        // In the line below we always make sure that the second register group, if present,
        // is never a source of interrupts (bits 3, 4 and 5 are all 1).
        writeB(SETUP_REG, 0x3f ^ (mask & 0x7));
    }

    /**
     * Unregisters the current Java interrupt callback, if any, and disables all
     * interrupts.
     */
    public void detachInt()
    {
        writeB(SETUP_REG, 0x3f);
        super.detachInt(0);
    }


    private void throwInvPort()
    {
        throw new AccesDioException("Invalid port number");
    }

}
