package org.lsst.ccs.drivers.iocard;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StreamTokenizer;
import java.io.StringReader;

import java.nio.file.Files;
import java.nio.file.FileSystems;
import java.nio.file.Path;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import java.util.function.Predicate;

import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;


/** Represents an I/O card on the PCI/PCIe/MPCIe bus. */
public final class PciCard {

    private final List<Resource> resources;

    private final int irq;

    /** Constructor.
        @param irq The IRQ number assigned to the card.
        @param res A stream of the bus resources assigned to the card.
    */
    private PciCard(int irq, Stream<Resource> res) {
	this.irq = irq;
	this.resources = res.collect(Collectors.toList());
    }



    /** Represents errors that occur when searching for a card or discovering
        the resources assigned to it.
    */
    // This is an unchecked exception since it may need to be thrown or passed up
    // from a lambda expression.
    public static class Error extends RuntimeException {
        Error(String msg, Throwable originalCause) {super(msg, originalCause);}
    }



    /** Represents a bus resource allocated to a card. */
    public static final class Resource {
        /** Constructor.
            @param bottom The lowest address assigned.
            @param top The highest address assigned.
            @param flags The resource flags from the device description under /sys/bus/pci/devices.
        */
        public Resource(long bottom, long top, long flags) {
            this.bottom = bottom;
            this.top = top;
            this.flags = flags;
        }

        /** Gets the base address.
            @return The address.
        */
        public long start() {return bottom;}

        /** Gets the no. of bytes assigned.
            @return The size.
        */
        public long size() {return top - bottom + 1;}

        /** Is this an I/O space resource?
            @return true or false
        */
        public boolean isIOSpace() {return 0 != (flags & 1);}

        /** Gets a string representation of the resource.
            @return A string of the form: ("I/O   " | "Memory") "space at " %#10x ", size " %8d
        */
        public String toString() {
            return String
                .format("%s space at %#10x, size %8d",
                        isIOSpace() ? "I/O   " : "Memory",
                        start(),
                        size());
        }
        private final long bottom, top, flags;
    };


    /* Gets the bus resources assigned to the card.
       @return An unmodifiable List.
    */
    public List<Resource> getResources() {
        return Collections.unmodifiableList(resources);
    }

    /** Gets the IRQ assigned to the card.
        @returns The IRQ number.
    */
    public int getIrq() {return irq;}


    /** Reads a set of bus resource descriptions from a character stream. The format
        should be like that of a "resource "file for a device under /sys/bus/pci/devices, although
        the numbers are allowed to be decimal as well as hex prefixed by "0x" or "0X".
        @param rdr A Reader for the open stream.
        @return A Stream of Resource objects.
    */
    private static Stream<Resource> readResources(Reader rdr)  {
        StreamTokenizer tkr = new StreamTokenizer(rdr);
        // We have to reset the syntax because there is no other way to turn
        // off number parsing completely. The built-in number parsing converts
        // numbers to double and we want longs with full precision.
        tkr.resetSyntax();
        tkr.eolIsSignificant(false);
        tkr.whitespaceChars(' ', ' ');
        tkr.whitespaceChars('\n', '\n');
        tkr.whitespaceChars('\t', '\t');
        tkr.wordChars('0', '9');
        tkr.wordChars('a', 'f');
        tkr.wordChars('A', 'F');
        tkr.wordChars('x', 'x');
        tkr.wordChars('X', 'X');
        final LongStream.Builder lng = LongStream.builder();
        try {
            int tok = tkr.nextToken();
            while (tok != StreamTokenizer.TT_EOF) {
                if (tok != StreamTokenizer.TT_WORD)
                    throw new Error("Badly formatted PCI resource description", null);
                final long val = Long.decode(tkr.sval);
                lng.accept(val);
                tok = tkr.nextToken();
            }
        } catch(IOException exc) {
            throw new Error("I/O error while reading PCI resource descriptions", exc);
        }
	Stream.Builder<Resource> res = Stream.builder();
        Iterator<Long> itr = lng.build().iterator();
        while (itr.hasNext()) {
            final long bottom = itr.next();
            final long top    = itr.next();
            final long flags  = itr.next();
            if (flags != 0) res.accept(new Resource(bottom, top, flags));
        }
        return res.build();
    }

    private static final String PCI_DEVICES = "/sys/bus/pci/devices/";

    /** Reads the "irq" and "resource" files from a directory under /sys/bus/pci/devices. Each
        directory name is a geographic address domain:bus:device.function, e.g., 0000:02:05.0.
        @param geo A geographic address that must exactly match the name of a device directory.
        @return An Optional which if nonempty contains the device info. If empty then
        the requested device was not found.
    */
    public static Optional<PciCard> fromGeographicAddress(String geo)  {
        try (FileReader irqrdr = new FileReader(PCI_DEVICES + geo + "/irq");
             FileReader resrdr = new FileReader(PCI_DEVICES + geo + "/resource"))
         {
            final int irq = Integer.decode(new BufferedReader(irqrdr).readLine());
            final Stream<Resource> res = readResources(resrdr);
            return Optional.of(new PciCard(irq, res));
        }
        catch (FileNotFoundException exc) {return Optional.empty();}
        catch (IOException exc) {throw new Error("Error reading description for " + geo, exc);}
        catch (NumberFormatException exc) {throw new Error("Error parsing IRQ number for " + geo, exc);}
    }

    /** Searches the directories under PCI_DEVICES for any one which has a "vendor" file containing
        the desired vendor ID, then calls fromGeographic(). There is no guaranteed search order.
        @return An Optional which if nonempty contains the device info. If empty then
        the no matching device.
    */
    public static Optional<PciCard> fromVendorId(int vid)  {
        final Optional<Path> pth = pciDevices(dev -> isFromVendor(dev, vid)).findFirst();
        if (pth.isPresent()) return fromGeographicAddress(geographicAddress(pth.get()));
        return Optional.empty();
    }

    // Returns a stream of Paths, one for each directory under PCI_DEVICES, filtered according
    // to the given predicate.
    private static Stream<Path> pciDevices(Predicate<Path> isWanted)  {
        try {
            return Files.list(FileSystems.getDefault().getPath(PCI_DEVICES)).filter(isWanted);
        }
        catch (IOException exc) {throw new Error("Error while scanning for PCI devices", exc);}
    }

    // Extracts the geographic address from a Path found by pciDevices().
    private static String geographicAddress(Path dev) {
        return dev.getName(dev.getNameCount() - 1).toString();
    }

    // A predicate for pciDevices(), matching devices from a given vendor.
    private static boolean isFromVendor(Path dev, int vid) {
        final String geo = geographicAddress(dev);
        try (FileReader vidrdr = new FileReader(dev.toString() + "/vendor")) {
            return vid == Integer.decode(new BufferedReader(vidrdr).readLine());
        }
        catch (IOException exc) {throw new Error("Error reading vendor ID for " + geo, exc);}
        catch (NumberFormatException exc) {throw new Error("Error parsing vendor ID for " + geo, exc);}
    }

    public String toString() {
        return
	    Stream.concat
	    (Stream.of(String.format("IRQ %d", irq)),
	     resources
	     .stream()
	     .map(Resource::toString))
            .collect(Collectors.joining("\n"));
    }


    /** Prints the information about two PCI/PCIe/mPCIe devices.
        @param arg The first element is the geographic address string of the first device.
        The second element is a vendor ID number.
    */
    public static void main(String[] arg) {
        Optional<String> card1 =
            PciCard
            .fromGeographicAddress(arg[0])
            .flatMap(c->Optional.of(c.toString()));
        Optional<String> card2 =
            PciCard.
            fromVendorId(Integer.decode(arg[1]))
            .flatMap(c->Optional.of(c.toString()));
        System.out.println(card1.orElseGet(()->"No PCI card at geo address " + arg[0]));
        System.out.println();
        System.out.println(card2.orElseGet(()->"No PCI card from vendor " + arg[1]));
    }
}

