package org.lsst.ccs.drivers.iocard;

import java.util.Optional;

/**
 ***************************************************************************
 **
 **  \file  AccesDio.java
 **
 **  Class for accessing the Acces DIO card
 **
 **  @author Owen Saxton
 **  @author Stephen Tether
 **
 ***************************************************************************
 */
public final class AccesDio extends Iocard {

   /**
    ***************************************************************************
    **
    **  Various constants
    **
    ***************************************************************************
    */
    public final static int
        ISA_N_REGS = 25,
        PCI_N_REGS = 16,
        PORTA_REG  = 0,
        PORTB_REG  = 1,
        PORTC_REG  = 2,
        CONF_REG   = 3,
        SETUP_REG  = 11,
        RESET_REG  = 15,
        ISA_STATUS_AND_RESET_REG  = 24,
        PCI_VENDOR_ID = 0x494f,
        PCI_MIN_IO_ADDR = 0xc000;

   /**
    ***************************************************************************
    **
    **  Private data
    **
    ***************************************************************************
    */
    private final static int[] dioConf = {0x80, 0x90, 0x82, 0x92,
                                          0x81, 0x91, 0x83, 0x93,
                                          0x88, 0x98, 0x8a, 0x9a,
                                          0x89, 0x99, 0x8b, 0x9b};

    private final static class CardInfo {
        final int irq;   // IRQ no.
        final int base;  // I/O base address.
        final int nregs; // No. of I/O registers (all eight-bit).
        final boolean isPCI; // Is this a PCI/PCIe/mPCIe card?

        CardInfo(int base, int irq) {
            if (base >= 0) {
                // User specified the IRQ and base address.
                this.irq = irq;
                this.base = base;
                this.isPCI =  base >= PCI_MIN_IO_ADDR;
                this.nregs = isPCI ? PCI_N_REGS : ISA_N_REGS;
            }
            else {
                // Probe the PCI bus.
                Optional<PciCard> card = PciCard.fromVendorId(PCI_VENDOR_ID);
                if (card.isPresent()) {
                    // Found a PCI card. For Acces DIO cards the third resource has the port address.
                    this.irq = card.get().getIrq();
                    this.base = (int)card.get().getResources().get(2).start();
                    this.nregs = PCI_N_REGS;
                    this.isPCI = true;
                }
                else {
                    throw new PciCard.Error("Can't find an Acces PCI card. Provide I/O addr and IRQ", null);
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    ** Constructors
    **
    ***************************************************************************
    */
    public AccesDio(int base, int irq)
    {
        this.card = new CardInfo(base, irq);
        super.init(card.base, card.nregs, card.irq);
    }
    
    public AccesDio() {this.card = null;}

    private CardInfo card;

   /**
    ***************************************************************************
    **
    **  Initialize card
    **
    ***************************************************************************
    */
    public void init(int base, int irq) {
        card = new CardInfo(base, irq);
        super.init(card.base, card.nregs, card.irq);
    }
    
    /**
     ***************************************************************************
     **
     ** Card properties
     **
     ***************************************************************************
     */
    public int getIrq() {return card.irq;}
    
    public int getBase() {return card.base;}
    
    public int getNregs() {return card.nregs;}

   /**
    ***************************************************************************
    **
    **  Set DIO configuration
    **
    ***************************************************************************
    */
    public void dioConfig(int value)
    {
        writeB(CONF_REG, dioConf[value & 0x0f]);
    }


   /**
    ***************************************************************************
    **
    **  Read from a DIO port
    **
    ***************************************************************************
    */
    public int dioInp(int port)
    {
        if (port < 0 || port > 2) throwInvPort();
        return readB(port);
    }


   /**
    ***************************************************************************
    **
    **  Write to a DIO port
    **
    ***************************************************************************
    */
    public void dioOut(int port, int value)
    {
        if (port < 0 || port > 2) throwInvPort();
        writeB(port, value);
    }


   /**
    ***************************************************************************
    **
    **  Read a bit from a DIO port
    **
    ***************************************************************************
    */
    public int dioInpBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        return (readB(port) >> bit) & 1;
    }


   /**
    ***************************************************************************
    **
    **  Write a bit to a DIO port
    **
    ***************************************************************************
    */
    public void dioOutBit(int port, int bit, int value)
    {
        if (port < 0 || port > 2) throwInvPort();
        updateB(port, 1 << bit, value << bit);
    }


   /**
    ***************************************************************************
    **
    **  Set a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioSetBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        int value = 1 << bit;
        updateB(port, value, value);
    }


   /**
    ***************************************************************************
    **
    **  Clear a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioClrBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        updateB(port, 1 << bit, 0);
    }


   /**
    ***************************************************************************
    **
    **  Enable interrupt handling.
    **  @param mask Has a 1 bit for each desired source of interrupts.
    **  Value 1 for port A, 2 for port B, 4 for port C. 
    **  @param cbObj The call-back object.
    **  @param cbMeth The name of the method to call on cbObj when an interrupt occurs.
    **  @param cbParm The argument to be given to cbObj.cbMeth()
    **
    **
    ***************************************************************************
    */
    public void attachInt(int mask, Object cbObj, String cbMeth, Object cbParm)
    {
        // Clearing interrupts:
        // ISA: Read from ISA_STATUS_AND_RESET_REG to get and reset interrupt status.
        // PCI: Write anything to RESET_REG.
        if (card.isPCI) {
            super.attachInt
                (0,                        // A unique ID no. for this attachment.
                 OPT_WRITER,               // Write to a register to reset interrupt status.
                 0,                        // Unused.
                 0,                        // Unused.
                 RESET_REG,                // The register to write to reset interrupt status.
                 0,                        // The value to write.
                 0,                        // Unused.
                 cbObj, cbMeth, cbParm);
        }
        else {
            // The iocardk kernal module's interrupt handler won't read a register
            // twice if it's both the check and read-to-reset register,
            super.attachInt
                (0,                        // A unique ID no. for this attachment.
                 OPT_CHECK,                // Check card interrupt status in the iocard handler.
                 ISA_STATUS_AND_RESET_REG, // The register to read for the check.
                 0x0f,                     // The mask to apply to the register value before checking.
                 ISA_STATUS_AND_RESET_REG, // The register to read to reset interrupt status.
                 0,                        // Unused.
                 0,                        // Unused.
                 cbObj, cbMeth, cbParm);
        }
        writeB(SETUP_REG, 0x07 ^ mask);    // A 0 bit means enable.
    }


   /**
    ***************************************************************************
    **
    **  Disable interrupt handling
    **
    ***************************************************************************
    */
    public void detachInt()
    {
        writeB(SETUP_REG, 0x07);
        super.detachInt(0);
    }


   /**
    ***************************************************************************
    **
    **  Throw an invalid port exception
    **
    ***************************************************************************
    */
    private void throwInvPort()
    {
        throw new AccesDioException("Invalid port number");
    }

}
