package org.lsst.ccs.drivers.iocard;

import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test the Java DSC universal driver routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestIocard implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_INIT        = 0,
        CMD_TERM        = 1,
        CMD_SHOW        = 2,
        CMD_READB       = 3,
        CMD_READW       = 4,
        CMD_READL       = 5,
        CMD_WRITEB      = 6,
        CMD_WRITEW      = 7,
        CMD_WRITEL      = 8,
        CMD_UPDATEB     = 9,
        CMD_UPDATEW     = 10,
        CMD_UPDATEL     = 11,
        CMD_REGSHOW     = 12,
        CMD_INTATTACH   = 13,
        CMD_INTDETACH   = 14,
        CMD_INTSHOW     = 15,
        CMD_TIMEREAD    = 16,
        NUM_CMDS        = 17;

    /*
    **  Command help
    */
    private final static String[] helpInit = {
        "Initialize access to an I/O card",
        "init <address> <size> <irq>",
        "address    The address of the I/O card",
        "size       The number of bytes used by the card's registers",
        "irq        The IRQ level to use for interrupts",
    };

    private final static String[] helpTerm = {
        "Terminate card access",
        "term",
    };

    private final static String[] helpShow = {
        "Show the current card initialization parameters",
        "show",
    };

    private final static String[] helpReadb = {
        "Read and display a byte from the card",
        "read <offset>",
        "offset  The offset from which to read a byte",
    };

    private final static String[] helpReadw = {
        "Read and display a 16-bit word from the card",
        "readw <offset>",
        "offset  The offset from which to read a 16-bit word",
    };

    private final static String[] helpReadl = {
        "Read and display a 32-bit word from the card",
        "readl <offset>",
        "offset  The offset from which to read a 32-bit integer",
    };

    private final static String[] helpWriteb = {
        "Write a byte to the card",
        "write <offset> <value>",
        "offset  The offset at which to write a byte",
        "value   The value to write",
    };

    private final static String[] helpWritew = {
        "Write a 16-bit word to the card",
        "writew <offset> <value>",
        "offset  The offset at which to write a 16-bit word",
        "value   The value to write",
    };

    private final static String[] helpWritel = {
        "Write a 32-bit word to the card",
        "writel <offset> <value>",
        "offset  The offset at which to write a 32-bit integer",
        "value   The value to write",
    };

    private final static String[] helpUpdateb = {
        "Update a byte on the card",
        "update <offset> <mask> <value>",
        "offset  The offset at which to update a byte",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private final static String[] helpUpdatew = {
        "Update a 16-bit word to the card",
        "updatew <offset> <mask> <value>",
        "offset  The offset at which to update a 16-bit word",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private final static String[] helpUpdatel = {
        "Update a 32-bit word to the card",
        "updatel <offset> <mask> <value>",
        "offset  The offset at which to update a 32-bit integer",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private final static String[] helpRegshow = {
        "Read and display the card's registers",
        "regshow",
    };

    private final static String[] helpIntAttach = {
        "Attach to interrupt processing",
        "intattach <optns> <cMask> <cValue> <rOffs> <rValue> <vOffs>",
        "optns   Interrupt options: 1 = check, 2 = write to reset, 4 = read",
        "cOffs   The offset to the register to check",
        "cMask   The mask of bits to check",
        "rOffs   The offset to the reset register",
        "rValue  The value to use for resetting",
        "vOffs   The offset to the value to read",
    };

    private final static String[] helpIntDetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    private final static String[] helpIntshow = {
        "Show interrupt processing parameters and statistics",
        "intshow",
    };

    private final static String[] helpTimeread = {
        "Time reading from an address",
        "timeread <count> <address> <width>",
        "count    The number of times to perform the read",
        "address  The address at which to read (default board address)",
        "width    The size (1, 2 or 4) of the word to read (default 1)",
    };

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("init",       CMD_INIT,       helpInit,       "iii");
        cmnd.add("term",       CMD_TERM,       helpTerm,       "");
        cmnd.add("show",       CMD_SHOW,       helpShow,       "");
        cmnd.add("readb",      CMD_READB,      helpReadb,      "i");
        cmnd.add("readw",      CMD_READW,      helpReadw,      "i");
        cmnd.add("readl",      CMD_READL,      helpReadl,      "i");
        cmnd.add("writeb",     CMD_WRITEB,     helpWriteb,     "ii");
        cmnd.add("writew",     CMD_WRITEW,     helpWritew,     "ii");
        cmnd.add("writel",     CMD_WRITEL,     helpWritel,     "ii");
        cmnd.add("updateb",    CMD_UPDATEB,    helpUpdateb,    "iii");
        cmnd.add("updatew",    CMD_UPDATEW,    helpUpdatew,    "iii");
        cmnd.add("updatel",    CMD_UPDATEL,    helpUpdatel,    "iii");
        cmnd.add("regshow",    CMD_REGSHOW,    helpRegshow,    "");
        cmnd.add("intattach",  CMD_INTATTACH,  helpIntAttach,  "iiiiii");
        cmnd.add("intdetach",  CMD_INTDETACH,  helpIntDetach,  "");
        cmnd.add("intshow",    CMD_INTSHOW,    helpIntshow,    "");
        cmnd.add("timeread",   CMD_TIMEREAD,   helpTimeread,   "iii");
    }

    /*
    **  Private fields
    */
    private final Output out;
    private final Iocard card;
    private final CmndProcess proc;

    // Card parameters
    private int     baseAddr = 0,
                    cardSize = 0,
                    irqLevel = 0;
    private boolean initd = false;

    // Direct I/O parameters
    private int     dValue = 0,
                    dMask = 0,
                    dCount = 0,
                    dOffset = 0,
                    dWidth = 1;

    // Interrupt parameters
    private int     iCount = 0,
                    iOptns = 0,
                    cOffset = 0,
                    cMask = 0,
                    rOffset = 0,
                    rValue = 0,
                    vOffset = 0;

   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    TestIocard(Iocard card, CmndProcess proc, Output out)
    {
        this.out = (out != null) ? out : new ConsOut();
        this.card = (card != null) ? card : new Iocard();
        this.proc = (proc != null) ? proc : new CmndProcess();
        this.proc.add(this, cmnd);
    }

    TestIocard()
    {
        this(null, null, null);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestIocard()).run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null) break;
                try {
                    if (!proc.process(line)) break;
                }
                catch (IocardException e) {
                    out.println(e);
                }
            }
        }
        catch (Exception e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        switch (code) {
        case CMD_INIT:
            procInit(found, args); break;
        case CMD_TERM:
            procTerm(found, args); break;
        case CMD_SHOW:
            procShow(found, args); break;
        case CMD_READB:
            procReadb(found, args); break;
        case CMD_READW:
            procReadw(found, args); break;
        case CMD_READL:
            procReadl(found, args); break;
        case CMD_WRITEB:
            procWriteb(found, args); break;
        case CMD_WRITEW:
            procWritew(found, args); break;
        case CMD_WRITEL:
            procWritel(found, args); break;
        case CMD_UPDATEB:
            procUpdateb(found, args); break;
        case CMD_UPDATEW:
            procUpdatew(found, args); break;
        case CMD_UPDATEL:
            procUpdatel(found, args); break;
        case CMD_REGSHOW:
            procRegshow(found, args); break;
        case CMD_TIMEREAD:
            procTimeread(found, args); break;
        case CMD_INTATTACH:
            procIntattach(found, args); break;
        case CMD_INTDETACH:
            procIntdetach(found, args); break;
        case CMD_INTSHOW:
            procIntshow(found, args); break;
        default:
            out.println("Command not fully implemented");
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the INIT command
    **
    ***************************************************************************
    */
    private void procInit(int found, Object[] args)
    {
        if (initd) {
            initd = false;
            card.term();
        }
        if ((found & 0x01) != 0) baseAddr = (Integer)args[0];
        if ((found & 0x02) != 0) cardSize = (Integer)args[1];
        if ((found & 0x04) != 0) irqLevel = (Integer)args[2];
        card.init(baseAddr, cardSize, irqLevel);
        initd = true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the TERM command
    **
    ***************************************************************************
    */
    private void procTerm(int found, Object[] args)
    {
        initd = false;
        card.term();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOW command
    **
    ***************************************************************************
    */
    private void procShow(int found, Object[] args)
    {
        out.println("Initialization parameters:");
        out.format("  Base address     = %04x\n", baseAddr);
        out.format("  Card size        = %04x\n", cardSize);
        out.println("  IRQ level        = " + irqLevel);
        out.println("  Initialized      = " + initd);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READB command
    **
    ***************************************************************************
    */
    private void procReadb(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        out.format("Value = 0x%02x\n", card.readB(dOffset));
    }


   /**
    ***************************************************************************
    **
    **  Processes the READW command
    **
    ***************************************************************************
    */
    private void procReadw(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        out.format("Value = 0x%04x\n", card.readW(dOffset));
    }


   /**
    ***************************************************************************
    **
    **  Processes the READL command
    **
    ***************************************************************************
    */
    private void procReadl(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        out.format("Value = 0x%08x\n", card.readL(dOffset));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEB command
    **
    ***************************************************************************
    */
    private void procWriteb(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dValue = (Integer)args[1];
        card.writeB(dOffset, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEW command
    **
    ***************************************************************************
    */
    private void procWritew(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dValue = (Integer)args[1];
        card.writeW(dOffset, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEL command
    **
    ***************************************************************************
    */
    private void procWritel(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dValue = (Integer)args[1];
        card.writeL(dOffset, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the UPDATEB command
    **
    ***************************************************************************
    */
    private void procUpdateb(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dMask = (Integer)args[1];
        if ((found & 0x04) != 0) dValue = (Integer)args[2];
        card.updateB(dOffset, dMask, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the UPDATEW command
    **
    ***************************************************************************
    */
    private void procUpdatew(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dMask = (Integer)args[1];
        if ((found & 0x04) != 0) dValue = (Integer)args[2];
        card.updateW(dOffset, dMask, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the UPDATEL command
    **
    ***************************************************************************
    */
    private void procUpdatel(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dOffset = (Integer)args[0];
        if ((found & 0x02) != 0) dMask = (Integer)args[1];
        if ((found & 0x04) != 0) dValue = (Integer)args[2];
        card.updateL(dOffset, dMask, dValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGSHOW command
    **
    ***************************************************************************
    */
    private void procRegshow(int found, Object[] args)
    {
        out.println("Card Registers:");
        for (int j = 0; j < cardSize; j++) {
            if ((j & 3) == 0 && j > 0) out.println();
            out.format("   %02x: %02x", j, card.readB(j));
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the TIMEREAD command
    **
    ***************************************************************************
    */
    private void procTimeread(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dCount = (Integer)args[0];
        if ((found & 0x02) != 0) dOffset = (Integer)args[1];
        if ((found & 0x04) != 0) dWidth = (Integer)args[2];
        long msec = System.currentTimeMillis();
        if (dWidth == 4) {
            for (int j = 0; j < dCount; j++)
                card.readL(dOffset);
        }
        else if (dWidth == 2) {
            for (int j = 0; j < dCount; j++)
                card.readW(dOffset);
        }
        else {
            for (int j = 0; j < dCount; j++)
                card.readB(dOffset);
        }
        msec = System.currentTimeMillis() - msec;
        out.println("Performed " + dCount + " operations in " + msec + " msec");
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTATTACH command
    **
    ***************************************************************************
    */
    private void procIntattach(int found, Object[] args)
    {
        if ((found & 0x01) != 0) iOptns = (Integer)args[0];
        if ((found & 0x02) != 0) cOffset = (Integer)args[1];
        if ((found & 0x04) != 0) cMask = (Integer)args[2];
        if ((found & 0x08) != 0) rOffset = (Integer)args[3];
        if ((found & 0x10) != 0) rValue = (Integer)args[4];
        if ((found & 0x20) != 0) vOffset = (Integer)args[5];
        iCount = 0;
        card.attachInt(0, iOptns, cOffset, cMask, rOffset, rValue,
                       vOffset, this, "intProc", this);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTDETACH command
    **
    ***************************************************************************
    */
    private void procIntdetach(int found, Object[] args)
    {
        card.detachInt(0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTSHOW command
    **
    ***************************************************************************
    */
    private void procIntshow(int found, Object[] args)
    {
        out.println("Interrupt Parameters:");
        out.format("  Options               = %02x\n", iOptns);
        out.format("  Check Register Offset = %02x\n", cOffset);
        out.format("  Check Mask            = %02x\n", cMask);
        out.format("  Reset Register Offset = %02x\n", rOffset);
        out.format("  Reset Value           = %02x\n", rValue);
        out.format("  Read Register Offset  = %02x\n", vOffset);
        out.println("  Count                 = " + iCount);
    }


   /**
    ***************************************************************************
    **
    **  Processes user interrupts
    **
    ***************************************************************************
    */
    private void intProc(int value, Object parm)
    {
        TestIocard tioc = (TestIocard)parm;
        iCount++;
        out.format("Interrupt occurred: count = %s, value = %02x\n",
                  tioc.iCount, value);
    }

}
