package org.lsst.ccs.drivers.iocard;

import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test the Java Helios driver routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestHelios implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_INIT        = 0,
        CMD_SHOW        = 1,
        CMD_REGSHOW     = 2,
        CMD_DIOCONFIG   = 3,
        CMD_DIOIN       = 4,
        CMD_DIOOUT      = 5,
        CMD_DIOINBIT    = 6,
        CMD_DIOOUTBIT   = 7,
        CMD_DIOSETBIT   = 8,
        CMD_DIOCLRBIT   = 9,
        CMD_DIOSHOW     = 10,
        CMD_ADCONFIG    = 11,
        CMD_ADSETCHAN   = 12,
        CMD_ADSAMPLE    = 13,
        CMD_ADSCAN      = 14,
        CMD_ADCTOV      = 15,
        CMD_ADVTOC      = 16,
        CMD_ADSHOW      = 17,
        CMD_DACONFIG    = 18,
        CMD_DASETGAIN   = 19,
        CMD_DAOUTPUT    = 20,
        CMD_DAUPDATE    = 21,
        CMD_DAVTOC      = 22,
        CMD_DACTOV      = 23,
        CMD_DASHOW      = 24,
        CMD_INTATTACH   = 25,
        CMD_INTDETACH   = 26,
        CMD_CNTRSTART   = 27,
        CMD_CNTRSTOP    = 28,
        CMD_CNTRSHOW    = 29,
        NUM_CMDS        = 30;

    /*
    **  Command help
    */
    private final static String[] helpInit = {
        "Initialize Helios card",
        "init <addr> <irq>",
        "addr   The base address of the card (initially 0x340)",
        "irq    The IRQ level of the card (initially 6)",
    };

    private final static String[] helpShow = {
        "Show initialization parameters",
        "show",
    };

    private final static String[] helpRegshow = {
        "Show register contents",
        "regshow",
    };

    private final static String[] helpDioconfig = {
        "Set digital I/O parameters",
        "dioconfig <value>",
        "value    The value to set",
    };

    private final static String[] helpDioin = {
        "Read and display a byte from a digital I/O port",
        "dioin <port>",
        "port    The digital I/O port to read",
    };

    private final static String[] helpDioout = {
        "Write a byte to a digital I/O port",
        "dioout <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private final static String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read",
        "bitnum  The number of the bit to read",
    };

    private final static String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private final static String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to set",
    };

    private final static String[] helpDioclrbit = {
        "Clear a bit in a digital I/O port",
        "dioclrbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to clear",
    };

    private final static String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private final static String[] helpAdconfig = {
        "Set the A/D configuration",
        "adconfig <gain> <polarity> <input>",
        "gain      The gain setting to use (0 - 3)",
        "polarity  The polarity: 0 = bipolar; 1 = unipolar",
        "input     The input type: single-ended = 0; differential = 1",
    };

    private final static String[] helpAdsetchan = {
        "Set the A/D channel range to use in samples and scans",
        "adsetchan <low> <high>",
        "low   The low (first) channel to use (0 - 15)",
        "high  The high (last) channel to use (0 - 15)",
    };

    private final static String[] helpAdsample = {
        "Sample and display the current A/D channel value",
        "adsample",
    };

    private final static String[] helpAdscan = {
        "Scan and display the current range of A/D channel value",
        "adscan",
    };

    private final static String[] helpAdctov = {
        "Convert an A/D count to a voltage",
        "adctov <count>",
        "count  The A/D count to convert",
    };

    private final static String[] helpAdvtoc = {
        "Convert a voltage to an A/D count",
        "advtoc <voltage>",
        "voltage  The voltage to convert",
    };

    private final static String[] helpAdshow = {
        "Show the current A/D parameters",
        "adshow",
    };

    private final static String[] helpDaconfig = {
        "Set the D/A configuration",
        "daconfig <gain> <polarity> <update> <resoln>",
        "gain      The gain setting to use (0 - 3)",
        "polarity  The polarity: 0 = bipolar; 1 = unipolar",
        "update    The update type: individual = 0; simultaneous = 1",
        "resoln    The resolution: 12-bit = 0; 16-bit = 1",
    };

    private final static String[] helpDasetgain = {
        "Set the D/A gain for a channel",
        "dasetgain <chan> <gain>",
        "chan  The channel to use (0 - 3)",
        "gain  The gain to set (0 - 3)",
    };

    private final static String[] helpDaoutput = {
        "Output a code value to a D/A channel",
        "daoutput <chan> <code>",
        "chan  The channel to use (0 - 3)",
        "code  The code value to output",
    };

    private final static String[] helpDaupdate = {
        "Update all the D/A output values (simultaneous updating mode)",
        "daupdate",
    };

    private final static String[] helpDavtoc = {
        "Convert a voltage to a D/A code",
        "davtoc <voltage>",
        "voltage  The voltage to convert",
    };

    private final static String[] helpDactov = {
        "Convert a D/A code to a voltage",
        "dactov <code>",
        "code  The D/A code to convert",
    };

    private final static String[] helpDashow = {
        "Show the current D/A parameters",
        "dashow",
    };

    private final static String[] helpIntattach = {
        "Attach to interrupt processing",
        "intattach <tmask>",
        "tmask  A bit mask of interrupt types (1 = AD, 2 = DIO, 4 = CNTR)",
    };

    private final static String[] helpIntdetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    private final static String[] helpCntrstart = {
        "Start generating counter interrupts",
        "cntrstart <rate>",
        "rate  The rate of interrupt generation (number / second)",
    };

    private final static String[] helpCntrstop = {
        "Stop generating counter interrupts",
        "cntrstop",
    };

    private final static String[] helpCntrshow = {
        "Show generated counter statistics",
        "cntrshow <optns>",
        "optns  Display option bits: 1 = intervals, 2 = counts, 4 = combined",
    };

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("init",       CMD_INIT,       helpInit,       "ii");
        cmnd.add("show",       CMD_SHOW,       helpShow,       "");
        cmnd.add("regshow",    CMD_REGSHOW,    helpRegshow,    "");
        cmnd.add("dioconfig",  CMD_DIOCONFIG,  helpDioconfig,  "i");
        cmnd.add("dioin",      CMD_DIOIN,      helpDioin,      "i");
        cmnd.add("dioout",     CMD_DIOOUT,     helpDioout,     "ii");
        cmnd.add("dioinbit",   CMD_DIOINBIT,   helpDioinbit,   "ii");
        cmnd.add("diooutbit",  CMD_DIOOUTBIT,  helpDiooutbit,  "iii");
        cmnd.add("diosetbit",  CMD_DIOSETBIT,  helpDiosetbit,  "ii");
        cmnd.add("dioclrbit",  CMD_DIOCLRBIT,  helpDioclrbit,  "ii");
        cmnd.add("dioshow",    CMD_DIOSHOW,    helpDioshow,    "");
        cmnd.add("adconfig",   CMD_ADCONFIG,   helpAdconfig,   "iii");
        cmnd.add("adsetchan",  CMD_ADSETCHAN,  helpAdsetchan,  "ii");
        cmnd.add("adsample",   CMD_ADSAMPLE,   helpAdsample,   "");
        cmnd.add("adscan",     CMD_ADSCAN,     helpAdscan,     "");
        cmnd.add("adctov",     CMD_ADCTOV,     helpAdctov,     "W");
        cmnd.add("advtoc",     CMD_ADVTOC,     helpAdvtoc,     "F");
        cmnd.add("adshow",     CMD_ADSHOW,     helpAdshow,     "");
        cmnd.add("daconfig",   CMD_DACONFIG,   helpDaconfig,   "iiii");
        cmnd.add("dasetgain",  CMD_DASETGAIN,  helpDasetgain,  "ii");
        cmnd.add("daoutput",   CMD_DAOUTPUT,   helpDaoutput,   "ii");
        cmnd.add("daupdate",   CMD_DAUPDATE,   helpDaupdate,   "");
        cmnd.add("davtoc",     CMD_DAVTOC,     helpDavtoc,     "iF");
        cmnd.add("dactov",     CMD_DACTOV,     helpDactov,     "ii");
        cmnd.add("dashow",     CMD_DASHOW,     helpDashow,     "");
        cmnd.add("intattach",  CMD_INTATTACH,  helpIntattach,  "i");
        cmnd.add("intdetach",  CMD_INTDETACH,  helpIntdetach,  "");
        cmnd.add("cntrstart",  CMD_CNTRSTART,  helpCntrstart,  "f");
        cmnd.add("cntrstop",   CMD_CNTRSTOP,   helpCntrstop,   "");
        cmnd.add("cntrshow",   CMD_CNTRSHOW,   helpCntrshow,   "");
    }

    /*
    **  Private fields
    */
    private final static int BASE_ADDR = 0x280,
                             IRQ_LEVEL = 5;

    private final static int MAX_CNTR_SAVED = 64;

    private final Output out;
    private final Helios hel;
    private final CmndProcess proc;
    private final TestIocard tioc;

    // Initialization parameters
    private int     baseAddr = BASE_ADDR,
                    irqLevel = IRQ_LEVEL;

    // Digital I/O parameters
    private int     dioPort = 0,
                    dioConf = 0,
                    dioValue = 0,
                    dioBit = 0,
                    adGain = 0;

    // A/D parameters
    private int     adLowChan = 0,
                    adHighChan = 0;
    private boolean adUnip = false,
                    adDiff = false;

    // D/A parameters
    private int     daChan = 0,
                    daGain = 0,
                    daCode = 0;
    private boolean daUnip = false,
                    daSimUp = false,
                    daHiRes = false;

    // General interrupt parameters
    private int     intMask = 0x07;

    // Counter interrupt parameters
    private long    cntrStart = 0,
                    cntrStop = 0,
                    cntrLast = 0,
                    cntrSavst = 0,
                    cntrTimes[] = new long[MAX_CNTR_SAVED];
    private int     cntrCount = 0,
                    cntrSaved = 0,
                    cntrCounts[] = new int[MAX_CNTR_SAVED];
    private float   cntrRate = 100.0F;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    TestHelios(Output out)
    {
        this.out = (out != null) ? out : new ConsOut();
        hel = new Helios();
        proc = new CmndProcess();
        proc.add(this, cmnd);
        tioc = new TestIocard(hel, proc, this.out);
    }

    TestHelios()
    {
        this(null);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestHelios()).tioc.run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Routine to dispatch a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        switch (code) {
        case CMD_INIT:
            procInit(found, args); break;
        case CMD_SHOW:
            procShow(found, args); break;
        case CMD_REGSHOW:
            procRegshow(found, args); break;
        case CMD_DIOCONFIG:
            procDioconfig(found, args); break;
        case CMD_DIOIN:
            procDioin(found, args); break;
        case CMD_DIOOUT:
            procDioout(found, args); break;
        case CMD_DIOINBIT:
            procDioinbit(found, args); break;
        case CMD_DIOOUTBIT:
            procDiooutbit(found, args); break;
        case CMD_DIOSETBIT:
            procDiosetbit(found, args); break;
        case CMD_DIOCLRBIT:
            procDioclrbit(found, args); break;
        case CMD_DIOSHOW:
            procDioshow(found, args); break;
        case CMD_ADCONFIG:
            procAdconfig(found, args); break;
        case CMD_ADSETCHAN:
            procAdsetchan(found, args); break;
        case CMD_ADSAMPLE:
            procAdsample(found, args); break;
        case CMD_ADSCAN:
            procAdscan(found, args); break;
        case CMD_ADCTOV:
            procAdctov(found, args); break;
        case CMD_ADVTOC:
            procAdvtoc(found, args); break;
        case CMD_ADSHOW:
            procAdshow(found, args); break;
        case CMD_DACONFIG:
            procDaconfig(found, args); break;
        case CMD_DASETGAIN:
            procDasetgain(found, args); break;
        case CMD_DAOUTPUT:
            procDaoutput(found, args); break;
        case CMD_DAUPDATE:
            procDaupdate(found, args); break;
        case CMD_DAVTOC:
            procDavtoc(found, args); break;
        case CMD_DACTOV:
            procDactov(found, args); break;
        case CMD_DASHOW:
            procDashow(found, args); break;
        case CMD_INTATTACH:
            procIntattach(found, args); break;
        case CMD_INTDETACH:
            procIntdetach(found, args); break;
        case CMD_CNTRSTART:
            procCntrstart(found, args); break;
        case CMD_CNTRSTOP:
            procCntrstop(found, args); break;
        case CMD_CNTRSHOW:
            procCntrshow(found, args); break;
        default:
            out.println("Command not fully implemented");
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the INIT command
    **
    ***************************************************************************
    */
    private void procInit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) baseAddr = (Integer)args[0];
        if ((found & 0x02) != 0) irqLevel = (Integer)args[1];
        hel.init(baseAddr, irqLevel);
        dioConf = hel.dioGetConfig();
        int[] chans = hel.adGetChans();
        adLowChan = chans[0];
        adHighChan = chans[1];
        adGain = hel.adGetGain();
        adUnip = hel.adIsUnipolar();
        adDiff = hel.adIsDifferential();
        daUnip = hel.daIsUnipolar();
        daSimUp = hel.daIsSimUpdate();
        daHiRes = hel.daIsHighRes();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOW command
    **
    ***************************************************************************
    */
    private void procShow(int found, Object[] args)
    {
        out.println("Initialization parameters:");
        out.format("  Base address     = %04x\n", baseAddr);
        out.format("  Card size        = %04x\n", Helios.N_REGS);
        out.println("  IRQ level        = " + irqLevel);
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGSHOW command
    **
    ***************************************************************************
    */
    private final static String[] regDesc = {
        "| AD7  | AD6  | AD5  | AD4  | AD3  | AD2  | AD1  | AD0  |",
        "| AD15 | AD14 | AD13 | AD12 | AD11 | AD10 | AD9  | AD8  |",
        "|  H3  |  H2  |  H1  |  H0  |  L3  |  L2  |  L1  |  L0  |",
        "|ADBUSY|SE/DIF|ADWAIT|DACBSY|  OF  |SCANEN| ADG1 | ADG0 |",
        "|CKSEL1|FRQSL1|FRQSL0|ADCLK |  -   |TINTE |DINTE |AINTE |",
        "|  -   |  -   | FT5  | FT4  | FT3  | FT2  | FT1  | FT0  |",
        "|  -   |  -   | FD5  | FD4  | FD3  | FD2  | FD1  | FD0  |",
        "|  -   | TINT | DINT | AINT |ADCH3 |ADCH2 |ADCH1 |ADCH0 |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |",
        "|  C7  |  C6  |  C5  |  C4  |  C3  |  C2  |  C1  |  C0  |",
        "|DIOCTR|DAMODE|DASIM | DIRA |DIRCH |  -   | DIRB |DIRCL |",
        "|CTRD7 |CTRD6 |CTRD5 |CTRD4 |CTRD3 |CTRD2 |CTRD1 |CTRD0 |",
        "|CTRD15|CTRD14|CTRD13|CTRD12|CTRD11|CTRD10|CTRD9 |CTRD8 |",
        "|CTRD23|CTRD22|CTRD21|CTRD20|CTRD19|CTRD18|CTRD17|CTRD16|",
        "|                  FPGA Revision Code                   |",
        "|  D7  |  D6  |  D5  |  D4  |  D3  |  D2  |  D1  |  D0  |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  -   |TDBUSY|EEBUSY|CALMUX|  -   |  -   |  -   |  -   |",
        "|                       Not Used                        |",
        "|  -   |  -   |  -   |  -   |  -   |  -   |  -   |EXFIFO|",
        "| DAUR |DACH1 |DACH0 |  -   |ADPOL |  -   | ADSD |  -   |",
        "|  -   |  -   |  -   |DASIZE|DAPOL |  -   | DAG1 | DAG0 |",
        "|                D/A Simultaneous Update                |",
        "| FD07 | FD06 | FD05 | FD04 | FD03 | FD02 | FD01 | FD00 |",
        "| FD11 | FD10 | FD09 | FD08 |  OF  |  FF  |  HF  |  EF  |",
    };

    private final static String[] pageDesc = {
        "Counter/Timer Access",
        "AutoCal Control",
        "Expanded FIFO and AD/DA Control"
    };

    private void procRegshow(int found, Object[] args)
    {
        int[] regs = new int[Helios.N_REGS + 8];
        for (int r = 0; r < Helios.N_REGS - 4; r++)
            regs[r] = hel.readB(r);
        for (int p = 0; p < 3; p++) {
            hel.writeB(Helios.PAGE_REG, p);
            for (int r = Helios.N_REGS - 4; r < Helios.N_REGS; r++)
                regs[4 * p + r] = hel.readB(r);
        }
        out.println("Main registers:");
        for (int r = 0; r < Helios.N_REGS - 4; r++) {
            if ((r == Helios.FIFO_THR_REG || r == Helios.FIFO_STS_REG)
                  && ((regs[Helios.EXP_FIFO_REG + 8] & 1) != 0))
                out.format("  %2s: %02x  %s\n", r, regs[r], regDesc[r + 19]);
            else
                out.format("  %2s: %02x  %s\n", r, regs[r], regDesc[r]);
        }
        for (int p = 0; p < 3; p++) {
            out.format("\nPage %s: %s\n", p, pageDesc[p]);
            for (int r = Helios.N_REGS - 4; r < Helios.N_REGS; r++) {
                int ri = 4 * p + r;
                out.format("  %2s: %02x  %s\n", r, regs[ri], regDesc[ri]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOCONFIG command
    **
    ***************************************************************************
    */
    private void procDioconfig(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioConf = (Integer)args[0];
        hel.dioConfig(dioConf);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOIN command
    **
    ***************************************************************************
    */
    private void procDioin(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        int value = hel.dioInp(dioPort);
        out.format("Value = 0x%02x\n", value);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOOUT command
    **
    ***************************************************************************
    */
    private void procDioout(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioValue = (Integer)args[1];
        hel.dioOut(dioPort, dioValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOINBIT command
    **
    ***************************************************************************
    */
    private void procDioinbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        int value = hel.dioInpBit(dioPort, dioBit);
        out.println("Value = " + value);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOOUTBIT command
    **
    ***************************************************************************
    */
    private void procDiooutbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        if ((found & 0x04) != 0) dioValue = (Integer)args[2];
        hel.dioOutBit(dioPort, dioBit, dioValue);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOSETBIT command
    **
    ***************************************************************************
    */
    private void procDiosetbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        hel.dioSetBit(dioPort, dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOCLRBIT command
    **
    ***************************************************************************
    */
    private void procDioclrbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0) dioPort = (Integer)args[0];
        if ((found & 0x02) != 0) dioBit = (Integer)args[1];
        hel.dioClrBit(dioPort, dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DIOSHOW command
    **
    ***************************************************************************
    */
    private void procDioshow(int found, Object[] args)
    {
        out.println("Digital I/O parameters:");
        out.format("  Configuration = %02x\n", dioConf);
        out.println("  Port          = " + dioPort);
        out.format("  Output value  = %02x\n", dioValue);
        out.println("  Bit number    = " + dioBit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADCONFIG command
    **
    ***************************************************************************
    */
    private void procAdconfig(int found, Object[] args)
    {
        if ((found & 0x01) != 0) adGain = (Integer)args[0];
        if ((found & 0x02) != 0) adUnip = (Integer)args[1] != 0;
        if ((found & 0x04) != 0) adDiff = (Integer)args[2] != 0;
        hel.adConfig(adGain, adUnip, adDiff);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADSETCHAN command
    **
    ***************************************************************************
    */
    private void procAdsetchan(int found, Object[] args)
    {
        if ((found & 0x01) != 0) adLowChan = (Integer)args[0];
        if ((found & 0x02) != 0) adHighChan = (Integer)args[1];
        hel.adSetChans(adLowChan, adHighChan);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADSAMPLE command
    **
    ***************************************************************************
    */
    private void procAdsample(int found, Object[] args)
    {
        int chan = hel.adGetCurrChan();
        short value = hel.adSample();
        float volts = hel.adCountToVolts(value);
        out.format("Channel %s: %04x ", chan, value);
        if (volts > -1.0 && volts < 1.0)
            out.format("(%.5f)\n", volts);
        else
            out.format("(%.6g)\n", volts);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADSCAN command
    **
    ***************************************************************************
    */
    private void procAdscan(int found, Object[] args)
    {
        short[] value = new short[Helios.N_AD_CHANS];
        int count = hel.adScan(value);
        for (int j = 0; j < count; j++) {
            int chan = adLowChan + j;
            if (chan >= Helios.N_AD_CHANS)
                chan -= Helios.N_AD_CHANS;
            if (j > 0) out.print(((j & 3) == 0) ? "\n" : " ");
            float volts = hel.adCountToVolts(value[j]);
            out.format("%2s: %04x ", chan, value[j]);
            if (volts > -1.0 && volts < 1.0)
                out.format("(% -8.5f)", volts);
            else
                out.format("(% -8.6g)", volts);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADCTOV command
    **
    ***************************************************************************
    */
    private void procAdctov(int found, Object[] args)
    {
        float volts = hel.adCountToVolts((Short)args[0]);
        if (volts > -1.0 && volts < 1.0)
            out.format("Value = %.5f\n", volts);
        else
            out.format("Value = %.6g\n", volts);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADVTOC command
    **
    ***************************************************************************
    */
    private void procAdvtoc(int found, Object[] args)
    {
        short count = hel.adVoltsToCount((Float)args[0]);
        out.format("Value = %04x\n", count);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ADSHOW command
    **
    ***************************************************************************
    */
    private void procAdshow(int found, Object[] args)
    {
        out.println("A/D Parameters:");
        out.println("  Gain setting       = " + adGain);
        out.println("  Unipolar input     = " + adUnip);
        out.println("  Differential input = " + adDiff);
        out.println("  Low channel        = " + adLowChan);
        out.println("  High channel       = " + adHighChan);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DACONFIG command
    **
    ***************************************************************************
    */
    private void procDaconfig(int found, Object[] args)
    {
        if ((found & 0x01) != 0) daGain = (Integer)args[0];
        if ((found & 0x02) != 0) daUnip = (Integer)args[1] != 0;
        if ((found & 0x04) != 0) daSimUp = (Integer)args[2] != 0;
        if ((found & 0x08) != 0) daHiRes = (Integer)args[3] != 0;
        hel.daConfig(daGain, daUnip, daSimUp, daHiRes);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DASETGAIN command
    **
    ***************************************************************************
    */
    private void procDasetgain(int found, Object[] args)
    {
        if ((found & 0x01) != 0) daChan = (Integer)args[0];
        if ((found & 0x02) != 0) daGain = (Integer)args[1];
        hel.daSetGain(daChan, daGain);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAOUTPUT command
    **
    ***************************************************************************
    */
    private void procDaoutput(int found, Object[] args)
    {
        if ((found & 0x01) != 0) daChan = (Integer)args[0];
        if ((found & 0x02) != 0) daCode = (Integer)args[1];
        hel.daOutput(daChan, daCode);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAUPDATE command
    **
    ***************************************************************************
    */
    private void procDaupdate(int found, Object[] args)
    {
        hel.daUpdate();
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAVTOC command
    **
    ***************************************************************************
    */
    private void procDavtoc(int found, Object[] args)
    {
        if ((found & 0x01) != 0) daChan = (Integer)args[0];
        daCode = hel.daVoltsToCode(daChan, (Float)args[1]);
        out.format("Value = %04x\n", daCode);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DACTOV command
    **
    ***************************************************************************
    */
    private void procDactov(int found, Object[] args)
    {
        if ((found & 0x01) != 0) daChan = (Integer)args[0];
        if ((found & 0x02) != 0) daCode = (Integer)args[1];
        float volts = hel.daCodeToVolts(daChan, daCode);
        if (volts > -1.0 && volts < 1.0)
            out.format("Value = %.5f\n", volts);
        else
            out.format("Value = %.6g\n", volts);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DASHOW command
    **
    ***************************************************************************
    */
    private void procDashow(int found, Object[] args)
    {
        out.println("D/A Parameters:");
        out.println("  Gain setting        = " + daGain);
        out.println("  Unipolar output     = " + daUnip);
        out.println("  Simultaneous update = " + daSimUp);
        out.println("  High resolution     = " + daHiRes);
        out.println("  Channel             = " + daChan);
        out.println("  Code                = " + daCode);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTATTACH command
    **
    ***************************************************************************
    */
    private void procIntattach(int found, Object[] args)
    {
        if ((found & 0x01) != 0) intMask = (Integer)args[0];
        hel.attachInt(0, Helios.OPT_CHECK | Helios.OPT_WRITER,
                      Helios.INTAD_STS_REG, Helios.PEND_INTS,
                      Helios.CMND_REG, Helios.CMND_CLEAR_INTS, 0,
                      this, "reportInt", hel);
        hel.updateB(Helios.INT_CTL_REG, Helios.ENAB_INTS, intMask);
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTDETACH command
    **
    ***************************************************************************
    */
    private void procIntdetach(int found, Object[] args)
    {
        hel.updateB(Helios.INT_CTL_REG, Helios.ENAB_INTS, 0);
        hel.detachInt(0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the CNTRSTART command
    **
    ***************************************************************************
    */
    private void procCntrstart(int found, Object[] args)
    {
        if ((found & 0x01) != 0) cntrRate = (Float)args[0];
        cntrSavst = cntrStart = cntrLast = System.nanoTime();
        cntrStop = 0;
        cntrCount = 0;
        cntrSaved = 0;
        hel.cntrConfig(cntrRate, false);
        hel.cntrEnable(this, "cntrProc", null);
        hel.cntrStart();
    }


   /**
    ***************************************************************************
    **
    **  Processes the CNTRSTOP command
    **
    ***************************************************************************
    */
    private void procCntrstop(int found, Object[] args)
    {
        hel.cntrDisable();
        cntrStop = System.nanoTime();
    }


   /**
    ***************************************************************************
    **
    **  Processes the CNTRSHOW command
    **
    ***************************************************************************
    */
    private void procCntrshow(int found, Object[] args)
    {
        int option = ((found & 0x01) != 0 ? (Integer)args[0] : 0);
        boolean times = ((option & 0x01) != 0);
        boolean counts = ((option & 0x02) != 0);
        boolean comb = ((option & 0x04) != 0);
        long intvl = cntrStop;
        float rate = 0.0F;
        int count = cntrCount;
        if (intvl == 0)
            intvl = System.nanoTime();
        intvl -= cntrStart;
        if (intvl != 0)
            rate = 1.0e9F * count / intvl;
        out.println("Interrupt Parameters:");
        out.println("  Req. Rate    = " + cntrRate);
        out.println("  Count        = " + cntrCount);
        out.println("  Actual Rate  = " + rate);
        if (times) {
            out.print("  Interval samples (nsec):");
            long prev = cntrSavst;
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 3) == 0) out.println();
                out.format("%14s", cntrTimes[j] - prev);
                prev = cntrTimes[j];
            }
            out.println();
        }
        if (counts) {
            out.print("  Counter samples:");
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 7) == 0) out.println();
                out.format("%7s", cntrCounts[j]);
            }
            out.println();
        }
        if (comb) {
            out.println("  Combined samples:");
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "Intvl(ns)", "Cntr",
                           "Dvn(ns)", "Dly(ns)");
            out.println();
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "---------", "----",
                           "-------", "-------");
            long prev = cntrSavst;
            int expTime = (int)(1000000000.0 / cntrRate);
            boolean low
              = (hel.readB(Helios.INT_CTL_REG) & Helios.CNTR1_USE_LOW) != 0;
            int mult = 1000000000
                        / (low ? Helios.CNTR1_LOW_FREQ : Helios.CNTR_HIGH_FREQ);
            int cntMax = expTime / mult;
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 1) == 0) out.println();
                int period = (int)(cntrTimes[j] - prev);
                int delay = cntMax - cntrCounts[j];
                if (delay < 0) delay += 0x10000;
                out.format("%13s %5s %8s %8s", period, cntrCounts[j],
                           period - expTime, mult * delay);
                prev = cntrTimes[j];
            }
            out.println();
        }
        if (times || counts || comb) {
            cntrSaved = 0;
            cntrSavst = cntrLast;
        }
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to report interrupts
    **
    ***************************************************************************
    */
    private void reportInt(int value, Object parm)
    {
        Helios heli = (Helios)parm;

        out.format("Flags = %02x, DIO lines = %06x\n", value,
                   heli.readL(Helios.PORTA_REG) & 0xffffff);
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to process counter interrupts
    **
    ***************************************************************************
    */
    private void cntrProc(int value, Object parm)
    {
        cntrLast = System.nanoTime();
        if (cntrSaved < MAX_CNTR_SAVED) {
            cntrTimes[cntrSaved] = cntrLast;
            cntrCounts[cntrSaved++] = hel.cntrRead();
        }
        cntrCount++;
    }

}
