package org.lsst.ccs.drivers.iocard;

/**
 ***************************************************************************
 **
 **  \file  Helios.java
 **
 **  Class for accessing the Helios data IO functions
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Helios extends Iocard {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        N_REGS           = 16,
        N_AD_CHANS       = 16,
        N_DA_CHANS       = 4,

        CMND_REG         = 0,   // Write
        CMND_CLEAR_AINT  = 0x01,
        CMND_CLEAR_DINT  = 0x02,
        CMND_CLEAR_TINT  = 0x04,
        CMND_CLEAR_INTS  = 0x07,
        CMND_RESET_FIFO  = 0x10,
        CMND_RESET_DA    = 0x20,
        CMND_RESET_BOARD = 0x40,
        CMND_START_AD    = 0x80,
        AD_FIFO_REG      = 0,   // Read 16 bits

        PAGE_REG         = 1,   // Write

        AD_CHAN_REG      = 2,   // Write/read

        AD_STAT_REG      = 3,   // Write/read
        AD_GAIN_MASK     = 0x03,
        AD_SCAN_ENAB     = 0x04,
        AD_FIFO_OF       = 0x08,
        DAC_BUSY         = 0x10,
        AD_WAIT          = 0x20,
        AD_SING_END      = 0x40,
        AD_BUSY          = 0x80,

        INT_CTL_REG      = 4,   // Write/read
        ENAB_AINT        = 0x01,
        ENAB_DINT        = 0x02,
        ENAB_TINT        = 0x04,
        ENAB_INTS        = 0x07,
        AD_EXT_TRIG      = 0x10,
        CNTR0_USE_LOW    = 0x20,
        CNTR1_USE_LOW    = 0x40,
        CNTR1_EXT_CLOCK  = 0x80,

        FIFO_THR_REG     = 5,   // Write/read

        DA_OUTP_REG      = 6,   // Write 16 bits
        FIFO_STS_REG     = 6,   // Read
        FIFO_EMPTY       = 0x01,
        FIFO_HALF_FULL   = 0x02,
        FIFO_FULL        = 0x04,
        FIFO_OVERFLOW    = 0x08,
        FIFO_MSB_MASK    = 0xf0,

        INTAD_STS_REG    = 7,   // Read
        AD_CHAN_MASK     = 0x0f,
        PEND_AINT        = 0x10,
        PEND_DINT        = 0x20,
        PEND_TINT        = 0x40,
        PEND_INTS        = 0x70,

        PORTA_REG        = 8,   // Write/read
        PORTB_REG        = 9,   // Write/read
        PORTC_REG        = 10,  // Write/read

        DIODA_CTL_REG    = 11,  // Write/read
        DIO_INPUT_CL     = 0x01,
        DIO_INPUT_B      = 0x02,
        DIO_INPUT_CH     = 0x08,
        DIO_INPUT_A      = 0x10,
        DIO_CONF_MASK    = 0x1b,
        DA_SIM_UPDATE    = 0x20,
        DA_HIGH_RES      = 0x40,
        DIO_COUNTER      = 0x80,

        CNTR_VAL_REG     = 12,  // Page0: Write/read 16 - 32 bits

        CNTR_CTL_REG     = 15,  // Page0: Write
        CNTR_CLEAR       = 0x01,
        CNTR_LOAD        = 0x02,
        CNTR_ENABLE      = 0x04,
        CNTR_DISABLE     = 0x08,
        CNTR_GATE_ENAB   = 0x10,
        CNTR_GATE_DISA   = 0x20,
        CNTR_LATCH       = 0x40,
        CNTR_ONE         = 0x80,
        FPGA_REV_REG     = 15,  // Page0: Read

        EXP_FIFO_REG     = 12,  // Page2: Write/read
        FIFO_ENHANCED    = 0x01,

        AD_MODE_REG      = 13,  // Page2: Write/read
        AD_DIFF          = 0x02,
        AD_UNIPOLAR      = 0x08,

        DA_MODE_REG      = 14,  // Page2: Write/read
        DA_RANGE_MASK    = 0x03,
        DA_BIPOLAR       = 0x08,
        DA_CHAN_MASK     = 0x30,
        DA_UNIQUE_RNG    = 0x80,
        DA_HIGH_RES_HW   = 0x10,

        DA_MSB_REG       = 15,  // Page2: Write
        DA_SIMUPD_REG    = 15,  // Page2: Read

        CNTR0_LOW_FREQ   = 1000000,
        CNTR1_LOW_FREQ   = 100000,
        CNTR_HIGH_FREQ   = 10000000,

        CLK_HIGH_FREQ    = 0,
        CLK_LOW_FREQ     = 1,
        CLK_EXTERNAL     = 2;

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        AD_MAX_LOOP = 100,
        CNTR_INT_ID = 1,
        DIO_INT_ID  = 2;

    private static String
        INV_DIO_PORT  = "Invalid DIO port number",
        INV_AD_CHAN   = "Invalid A/D channel number",
        INV_AD_GAIN   = "Invalid A/D channel gain",
        AD_READY_TMO  = "A/D ready timeout",
        AD_CONV_TMO   = "A/D conversion timeout",
        INV_DA_CHAN   = "Invalid D/A channel number",
        INV_DA_GAIN   = "Invalid D/A channel gain",
        INV_DA_GPOL   = "Invalid D/A channel gain/polarity",
        HI_RES_UNAVL  = "16-bit resolution not available",
        DA_GAIN_UNKN  = "Gain not set for D/A channel",
        INV_FREQUENCY = "Invalid frequency",
        INV_COUNT     = "Invalid counter value",
        INV_CLOCK     = "Invalid clock specifier";

   /**
    ***************************************************************************
    **
    **  Private data
    **
    ***************************************************************************
    */
    private int[] daGain = {-1, -1, -1, -1};


   /**
    ***************************************************************************
    **
    **  Main constructors
    **
    ***************************************************************************
    */
    public Helios(int base, int irq)
    {
        super.init(base, N_REGS, irq);
    }

    public Helios()
    {
    }


   /**
    ***************************************************************************
    **
    **  Initiate access to a card
    **
    ***************************************************************************
    */
    public void init(int base, int irq)
    {
        super.init(base, N_REGS, irq);
    }


   /**
    ***************************************************************************
    **
    **  Set DIO configuration
    **
    ***************************************************************************
    */
    public void dioConfig(int value)
    {
        updateB(DIODA_CTL_REG, DIO_CONF_MASK,
                (value & 0x0a) | ((value << 4) & 0x10) | ((value >> 2) & 0x01));
    }


   /**
    ***************************************************************************
    **
    **  Get DIO configuration
    **
    ***************************************************************************
    */
    public int dioGetConfig()
    {
        int value = readB(DIODA_CTL_REG);
        return (value & 0x0a) | ((value >> 4) & 0x01) | ((value << 2) & 0x04);
    }


   /**
    ***************************************************************************
    **
    **  Read from a DIO port
    **
    ***************************************************************************
    */
    public int dioInp(int port)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        return readB(PORTA_REG + port);
    }


   /**
    ***************************************************************************
    **
    **  Write to a DIO port
    **
    ***************************************************************************
    */
    public void dioOut(int port, int value)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        writeB(PORTA_REG + port, value);
    }


   /**
    ***************************************************************************
    **
    **  Read a bit from a DIO port
    **
    ***************************************************************************
    */
    public int dioInpBit(int port, int bit)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        return (readB(PORTA_REG + port) >> bit) & 1;
    }


   /**
    ***************************************************************************
    **
    **  Write a bit to a DIO port
    **
    ***************************************************************************
    */
    public void dioOutBit(int port, int bit, int value)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        updateB(PORTA_REG + port, 1 << bit, value << bit);
    }


   /**
    ***************************************************************************
    **
    **  Set a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioSetBit(int port, int bit)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        int value = 1 << bit;
        updateB(PORTA_REG + port, value, value);
    }


   /**
    ***************************************************************************
    **
    **  Clear a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioClrBit(int port, int bit)
    {
        if (port < 0 || port > 2)
            throw new HeliosException(INV_DIO_PORT);
        updateB(PORTA_REG + port, 1 << bit, 0);
    }


   /**
    ***************************************************************************
    **
    **  Configure A/D conversion parameters
    **
    ***************************************************************************
    */
    public void adConfig(int gain, boolean unip, boolean diff)
    {
        if (gain < 0 || gain > 3)
            throw new HeliosException(INV_AD_GAIN);
        else {
            updateB(AD_STAT_REG, AD_GAIN_MASK, gain);
            writeB(PAGE_REG, 2);
            writeB(AD_MODE_REG, (unip ? AD_UNIPOLAR: 0) | (diff ? AD_DIFF : 0));
        }
    }


   /**
    ***************************************************************************
    **
    **  Get A/D conversion gain
    **
    ***************************************************************************
    */
    public int adGetGain()
    {
        return readB(AD_STAT_REG) & AD_GAIN_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Get whether A/D conversion is unipolar
    **
    ***************************************************************************
    */
    public boolean adIsUnipolar()
    {
        writeB(PAGE_REG, 2);
        return (readB(AD_MODE_REG) & AD_UNIPOLAR) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Get whether A/D input is differential
    **
    ***************************************************************************
    */
    public boolean adIsDifferential()
    {
        writeB(PAGE_REG, 2);
        return (readB(AD_MODE_REG) & AD_DIFF) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Set A/D channel range
    **
    ***************************************************************************
    */
    public void adSetChans(int lowChan, int highChan)
    {
        if (lowChan < 0 || lowChan >= N_AD_CHANS
              || highChan < 0 || highChan >= N_AD_CHANS)
            throw new HeliosException(INV_AD_CHAN);
        writeB(AD_CHAN_REG, lowChan | (highChan << 4));
    }


   /**
    ***************************************************************************
    **
    **  Set single A/D channel
    **
    ***************************************************************************
    */
    public void adSetChan(int chan)
    {
        adSetChans(chan, chan);
    }


   /**
    ***************************************************************************
    **
    **  Get A/D channel range
    **
    ***************************************************************************
    */
    public int[] adGetChans()
    {
        int value = readB(AD_CHAN_REG);
        int[] chans = {value & 0x0f, value >> 4};
        return chans;
    }


   /**
    ***************************************************************************
    **
    **  Get current A/D channel
    **
    ***************************************************************************
    */
    public int adGetCurrChan()
    {
        return readB(INTAD_STS_REG) & AD_CHAN_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Sample the next A/D channel
    **
    ***************************************************************************
    */
    public short adSample()
    {
        int status = 1;

        updateB(AD_STAT_REG, AD_SCAN_ENAB, 0);
        for (int j = 0; j < AD_MAX_LOOP; j++) {
            status = readB(AD_STAT_REG) & AD_WAIT;
            if (status == 0) break;
        }
        if (status != 0)
            throw new HeliosException(AD_READY_TMO);

        writeB(CMND_REG, CMND_START_AD | CMND_RESET_FIFO);
        for (int j = 0; j < AD_MAX_LOOP; j++) {
            status = readB(AD_STAT_REG) & AD_BUSY;
            if (status == 0) break;
        }
        if (status != 0)
            throw new HeliosException(AD_CONV_TMO);

        return (short)readW(AD_FIFO_REG);
    }


   /**
    ***************************************************************************
    **
    **  Scan a set of A/D channels
    **
    ***************************************************************************
    */
    public int adScan(short[] data)
    {
        int status = 1;

        updateB(AD_STAT_REG, AD_SCAN_ENAB, AD_SCAN_ENAB);
        for (int j = 0; j < AD_MAX_LOOP; j++) {
            status = readB(AD_STAT_REG) & AD_WAIT;
            if (status == 0) break;
        }
        if (status != 0)
            throw new HeliosException(AD_READY_TMO);

        writeB(CMND_REG, CMND_START_AD | CMND_RESET_FIFO);
        for (int j = 0; j < AD_MAX_LOOP; j++) {
            status = readB(AD_STAT_REG) & AD_BUSY;
            if (status == 0) break;
        }
        if (status != 0)
            throw new HeliosException(AD_CONV_TMO);

        int count, depth = readB(FIFO_STS_REG);
        for (count = 0; count < depth; count++)
            data[count] = (short)readW(AD_FIFO_REG);

        return count;
    }


   /**
    ***************************************************************************
    **
    **  Convert A/D count to volts
    **
    ***************************************************************************
    */
    public float adCountToVolts(short count)
    {
        writeB(PAGE_REG, 2);
        boolean unip = (readB(AD_MODE_REG) & AD_UNIPOLAR) != 0;
        int gain = readB(AD_STAT_REG) & AD_GAIN_MASK;
        if (unip)
            return (10f / 65536f / (1 << gain)) * (count + 32768);
        else
            return (10f / 32768f / (1 << gain)) * count;
    }

        
   /**
    ***************************************************************************
    **
    **  Convert volts to A/D count
    **
    ***************************************************************************
    */
    public short adVoltsToCount(float volts)
    {
        writeB(PAGE_REG, 2);
        boolean unip = (readB(AD_MODE_REG) & AD_UNIPOLAR) != 0;
        int gain = readB(AD_STAT_REG) & AD_GAIN_MASK;
        int count;
        if (unip)
            count = (int)((1 << gain) * (65536f / 10f) * volts) - 32768;
        else
            count = (int)((1 << gain) * (32768f / 10f) * volts);
        if (count > 32767) count = 32767;
        else if (count < -32768) count = -32768;

        return (short)count;
    }

        
   /**
    ***************************************************************************
    **
    **  Configure D/A conversion parameters
    **
    ***************************************************************************
    */
    public void daConfig(int gain, boolean uniPol, boolean simUpd,
                         boolean hiRes)
    {
        if (gain < 0 || gain > 2)
            throw new HeliosException(INV_DA_GAIN);
        if (uniPol && gain == 2)
            throw new HeliosException(INV_DA_GPOL);
        writeB(PAGE_REG, 2);
        writeB(DA_MODE_REG, gain | (uniPol ? 0 : DA_BIPOLAR));
        updateB(DIODA_CTL_REG, DA_SIM_UPDATE | DA_HIGH_RES,
                (simUpd ? DA_SIM_UPDATE : 0) | (hiRes ? DA_HIGH_RES : 0));
        for (int j = 0; j < N_DA_CHANS; j++)
            daGain[j] = gain;
    }


   /**
    ***************************************************************************
    **
    **  Set D/A channel gain
    **
    ***************************************************************************
    */
    public void daSetGain(int chan, int gain)
    {
        if (chan < -1 || chan >= N_DA_CHANS)
            throw new HeliosException(INV_DA_CHAN);
        if (gain < 0 || gain > 2)
            throw new HeliosException(INV_DA_GAIN);
        writeB(PAGE_REG, 2);
        int bipolar = readB(DA_MODE_REG) & DA_BIPOLAR;
        if (gain == 2 && bipolar == 0)
            throw new HeliosException(INV_DA_GPOL);
        writeB(DA_MODE_REG,
               bipolar | gain | (chan >= 0 ? DA_UNIQUE_RNG | (chan << 4) : 0));
        if (chan >= 0)
            daGain[chan] = gain;
        else
            for (int j = 0; j < N_DA_CHANS; j++)
                daGain[j] = gain;
    }


   /**
    ***************************************************************************
    **
    **  Get D/A channel gain
    **
    ***************************************************************************
    */
    public int daGetGain(int chan)
    {
        if (chan < -1 || chan >= N_DA_CHANS)
            throw new HeliosException(INV_DA_CHAN);
        if (chan >= 0)
            return daGain[chan];
        else {
            int gain = daGain[0];
            for (int j = 1; j < N_DA_CHANS; j++)
                if (gain != daGain[j]) return -1;
            return gain;
        }
    }


   /**
    ***************************************************************************
    **
    **  Get whether D/A converter is unipolar
    **
    ***************************************************************************
    */
    public boolean daIsUnipolar()
    {
        writeB(PAGE_REG, 2);
        return (readB(DA_MODE_REG) & DA_BIPOLAR) == 0;
    }


   /**
    ***************************************************************************
    **
    **  Get whether D/A converter is simultaneously updating
    **
    ***************************************************************************
    */
    public boolean daIsSimUpdate()
    {
        return (readB(DIODA_CTL_REG) & DA_SIM_UPDATE) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Get whether D/A converter is in high resolution (16-bit) mode
    **
    ***************************************************************************
    */
    public boolean daIsHighRes()
    {
        return (readB(DIODA_CTL_REG) & DA_HIGH_RES) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Get whether D/A converter hardware is high resolution (16-bit)
    **
    ***************************************************************************
    */
    public boolean daIsHighResHW()
    {
        writeB(PAGE_REG, 2);
        return (readB(DA_MODE_REG) & DA_HIGH_RES_HW) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Output a value to a D/A channel
    **
    ***************************************************************************
    */
    public void daOutput(int chan, int code)
    {
        if (chan < 0 || chan >= N_DA_CHANS)
            throw new HeliosException(INV_DA_CHAN);
        if ((readB(DIODA_CTL_REG) & DA_HIGH_RES) != 0) {
            writeB(PAGE_REG, 2);
            writeB(DA_MSB_REG, code >> 8);
            writeW(DA_OUTP_REG, (code & 0xff) | (chan << 14));
        }
        else
            writeW(DA_OUTP_REG, (code & 0x0fff) | (chan << 14));
    }


   /**
    ***************************************************************************
    **
    **  Simultaneously update D/A channels
    **
    ***************************************************************************
    */
    public void daUpdate()
    {
        writeB(PAGE_REG, 2);
        readB(DA_SIMUPD_REG);
    }

        
   /**
    ***************************************************************************
    **
    **  Convert volts to D/A code for a channel
    **
    ***************************************************************************
    */
    public int daVoltsToCode(int chan, float volts)
    {
        if (chan < 0 || chan >= N_DA_CHANS)
            throw new HeliosException(INV_DA_CHAN);
        int gain = daGain[chan];
        if (gain < 0)
            throw new HeliosException(DA_GAIN_UNKN);
        float range = 5f * (1 << gain);
        int resol = (readB(DIODA_CTL_REG) & DA_HIGH_RES) != 0 ? 32768 : 2048;
        int code = (int)(2 * resol * volts / range);
        writeB(PAGE_REG, 2);
        if ((readB(DA_MODE_REG) & DA_BIPOLAR) != 0)
            code += resol;
        if (code >= 2 * resol) code = 2 * resol - 1;
        else if (code < 0) code = 0;

        return code;
    }

        
   /**
    ***************************************************************************
    **
    **  Convert D/A code to volts for a channel
    **
    ***************************************************************************
    */
    public float daCodeToVolts(int chan, int code)
    {
        if (chan < 0 || chan >= N_DA_CHANS)
            throw new HeliosException(INV_DA_CHAN);
        int gain = daGain[chan];
        if (gain < 0)
            throw new HeliosException(DA_GAIN_UNKN);
        float range = 5f * (1 << gain);
        int resol = (readB(DIODA_CTL_REG) & DA_HIGH_RES) != 0 ? 32768 : 2048;
        writeB(PAGE_REG, 2);
        if ((readB(DA_MODE_REG) & DA_BIPOLAR) != 0)
            code -= resol;
        return (range * code) / (2 * resol);
    }

        
   /**
    ***************************************************************************
    **
    **  Configure counter
    **
    ***************************************************************************
    */
    public void cntrConfig(float freq, boolean gated)
    {
        int clock = CLK_HIGH_FREQ;
        int count = (int)(CNTR_HIGH_FREQ / freq);
        if (count >= 65536) {
            count = (int)(CNTR1_LOW_FREQ / freq);
            clock = CLK_LOW_FREQ;
        }
        if (count < 2 || count >= 65536)
            throw new HeliosException(INV_FREQUENCY);
        cntrConfig(count, clock, gated);
    }

    public void cntrConfig(int count, int clock, boolean gated)
    {
        if (count < 2 || count >= 65536)
            throw new HeliosException(INV_COUNT);
        if (clock < CLK_HIGH_FREQ || clock > CLK_EXTERNAL)
            throw new HeliosException(INV_CLOCK);

        int value = 0;
        if (clock == CLK_EXTERNAL) value = CNTR1_EXT_CLOCK;
        else if (clock == CLK_LOW_FREQ) value = CNTR1_USE_LOW;
        updateB(INT_CTL_REG, CNTR1_EXT_CLOCK | CNTR1_USE_LOW, value);

        if (clock == CLK_EXTERNAL || gated)
            updateB(DIODA_CTL_REG, DIO_COUNTER, DIO_COUNTER);

        writeB(PAGE_REG, 0);
        writeW(CNTR_VAL_REG, count);
        writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_LOAD);
        if (gated)
            writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_GATE_ENAB);
        else
            writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_GATE_DISA);
    }


   /**
    ***************************************************************************
    **
    **  Start counter
    **
    ***************************************************************************
    */
    public void cntrStart()
    {
        writeB(PAGE_REG, 0);
        writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_ENABLE);
    }


   /**
    ***************************************************************************
    **
    **  Stop counter
    **
    ***************************************************************************
    */
    public void cntrStop()
    {
        writeB(PAGE_REG, 0);
        writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_DISABLE);
    }


   /**
    ***************************************************************************
    **
    **  Clear counter
    **
    ***************************************************************************
    */
    public void cntrClear()
    {
        writeB(PAGE_REG, 0);
        writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_CLEAR);
    }


   /**
    ***************************************************************************
    **
    **  Read counter
    **
    ***************************************************************************
    */
    public int cntrRead()
    {
        writeB(PAGE_REG, 0);
        writeB(CNTR_CTL_REG, CNTR_ONE | CNTR_LATCH);
        return readW(CNTR_VAL_REG);
    }


   /**
    ***************************************************************************
    **
    **  Enable counter interrupt handling
    **
    ***************************************************************************
    */
    public void cntrEnable(Object cbObj, String cbMeth, Object cbParm)
    {
        super.attachInt(CNTR_INT_ID, OPT_CHECK | OPT_WRITER,
                        INTAD_STS_REG, PEND_TINT, CMND_REG, CMND_CLEAR_TINT, 0,
                        cbObj, cbMeth, cbParm);
        updateB(INT_CTL_REG, ENAB_TINT, ENAB_TINT);
    }


   /**
    ***************************************************************************
    **
    **  Disable counter interrupt handling
    **
    ***************************************************************************
    */
    public void cntrDisable()
    {
        updateB(INT_CTL_REG, ENAB_TINT, 0);
        super.detachInt(CNTR_INT_ID);
    }

}
