package org.lsst.ccs.drivers.iocard;

/**
 ***************************************************************************
 **
 **  \file  AccesDio.java
 **
 **  Class for accessing the Acces DIO card
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class AccesDio extends Iocard {

   /**
    ***************************************************************************
    **
    **  Various constants
    **
    ***************************************************************************
    */
    public final static int
        N_REGS    = 25,
        PORTA_REG = 0,
        PORTB_REG = 1,
        PORTC_REG = 2,
        CONF_REG  = 3,
        SETUP_REG = 11,
        STATE_REG = 15,
        RESET_REG = 24;

   /**
    ***************************************************************************
    **
    **  Private data
    **
    ***************************************************************************
    */
    private final static int[] dioConf = {0x80, 0x90, 0x82, 0x92,
                                          0x81, 0x91, 0x83, 0x93,
                                          0x88, 0x98, 0x8a, 0x9a,
                                          0x89, 0x99, 0x8b, 0x9b};


   /**
    ***************************************************************************
    **
    **  Main constructors
    **
    ***************************************************************************
    */
    public AccesDio(int base, int irq)
    {
        super.init(base, N_REGS, irq);
    }

    public AccesDio()
    {
    }


   /**
    ***************************************************************************
    **
    **  Initiate access to a card
    **
    ***************************************************************************
    */
    public void init(int base, int irq)
    {
        super.init(base, N_REGS, irq);
    }


   /**
    ***************************************************************************
    **
    **  Set DIO configuration
    **
    ***************************************************************************
    */
    public void dioConfig(int value)
    {
        writeB(CONF_REG, dioConf[value & 0x0f]);
    }


   /**
    ***************************************************************************
    **
    **  Read from a DIO port
    **
    ***************************************************************************
    */
    public int dioInp(int port)
    {
        if (port < 0 || port > 2) throwInvPort();
        return readB(port);
    }


   /**
    ***************************************************************************
    **
    **  Write to a DIO port
    **
    ***************************************************************************
    */
    public void dioOut(int port, int value)
    {
        if (port < 0 || port > 2) throwInvPort();
        writeB(port, value);
    }


   /**
    ***************************************************************************
    **
    **  Read a bit from a DIO port
    **
    ***************************************************************************
    */
    public int dioInpBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        return (readB(port) >> bit) & 1;
    }


   /**
    ***************************************************************************
    **
    **  Write a bit to a DIO port
    **
    ***************************************************************************
    */
    public void dioOutBit(int port, int bit, int value)
    {
        if (port < 0 || port > 2) throwInvPort();
        updateB(port, 1 << bit, value << bit);
    }


   /**
    ***************************************************************************
    **
    **  Set a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioSetBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        int value = 1 << bit;
        updateB(port, value, value);
    }


   /**
    ***************************************************************************
    **
    **  Clear a bit in a DIO port
    **
    ***************************************************************************
    */
    public void dioClrBit(int port, int bit)
    {
        if (port < 0 || port > 2) throwInvPort();
        updateB(port, 1 << bit, 0);
    }


   /**
    ***************************************************************************
    **
    **  Enable interrupt handling
    **
    ***************************************************************************
    */
    public void attachInt(int mask, Object cbObj, String cbMeth, Object cbParm)
    {
        super.attachInt(0, OPT_CHECK | OPT_WRITER, STATE_REG, 0x0f,
                        STATE_REG, 0, 0, cbObj, cbMeth, cbParm);
        writeB(SETUP_REG, 0x07 ^ mask);
    }


   /**
    ***************************************************************************
    **
    **  Disable interrupt handling
    **
    ***************************************************************************
    */
    public void detachInt()
    {
        writeB(SETUP_REG, 0x07);
        super.detachInt(0);
    }


   /**
    ***************************************************************************
    **
    **  Throw an invalid port exception
    **
    ***************************************************************************
    */
    private void throwInvPort()
    {
        throw new AccesDioException("Invalid port number");
    }

}
