package org.lsst.ccs.drivers.iocard;

import java.io.IOException;
import java.util.Scanner;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.Output;
import org.lsst.ccs.utilities.sa.ConsOut;

/**
 ***************************************************************************
 **
 **  \file  TestIocard.java
 **
 **  Program to test the Java DSC universal driver routines
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
public class TestIocard implements CmndProc.Dispatch {

    private final static int
        CMD_INIT        = 0,
        CMD_TERM        = 1,
        CMD_SHOW        = 2,
        CMD_READB       = 3,
        CMD_READW       = 4,
        CMD_READL       = 5,
        CMD_WRITEB      = 6,
        CMD_WRITEW      = 7,
        CMD_WRITEL      = 8,
        CMD_UPDATEB     = 9,
        CMD_UPDATEW     = 10,
        CMD_UPDATEL     = 11,
        CMD_REGSHOW     = 12,
        CMD_INTATTACH   = 13,
        CMD_INTDETACH   = 14,
        CMD_INTSHOW     = 15,
        CMD_TIMEREAD    = 16,
        NUM_CMDS        = 17;

    private static final String[] helpInit = {
        "Initialize access to an I/O card",
        "init <address> <size> <irq>",
        "address    The address of the I/O card",
        "size       The number of bytes used by the card's registers",
        "irq        The IRQ level to use for interrupts",
    };

    private static final String[] helpTerm = {
        "Terminate card access",
        "term",
    };

    private static final String[] helpShow = {
        "Show the current card initialization parameters",
        "show",
    };

    private static final String[] helpReadb = {
        "Read and display a byte from the card",
        "read <offset>",
        "offset  The offset from which to read a byte",
    };

    private static final String[] helpReadw = {
        "Read and display a 16-bit word from the card",
        "readw <offset>",
        "offset  The offset from which to read a 16-bit word",
    };

    private static final String[] helpReadl = {
        "Read and display a 32-bit word from the card",
        "readl <offset>",
        "offset  The offset from which to read a 32-bit integer",
    };

    private static final String[] helpWriteb = {
        "Write a byte to the card",
        "write <offset> <value>",
        "offset  The offset at which to write a byte",
        "value   The value to write",
    };

    private static final String[] helpWritew = {
        "Write a 16-bit word to the card",
        "writew <offset> <value>",
        "offset  The offset at which to write a 16-bit word",
        "value   The value to write",
    };

    private static final String[] helpWritel = {
        "Write a 32-bit word to the card",
        "writel <offset> <value>",
        "offset  The offset at which to write a 32-bit integer",
        "value   The value to write",
    };

    private static final String[] helpUpdateb = {
        "Update a byte on the card",
        "update <offset> <mask> <value>",
        "offset  The offset at which to update a byte",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private static final String[] helpUpdatew = {
        "Update a 16-bit word to the card",
        "updatew <offset> <mask> <value>",
        "offset  The offset at which to update a 16-bit word",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private static final String[] helpUpdatel = {
        "Update a 32-bit word to the card",
        "updatel <offset> <mask> <value>",
        "offset  The offset at which to update a 32-bit integer",
        "mask    The mask of bits to update",
        "value   The value to write",
    };

    private static final String[] helpRegshow = {
        "Read and display the card's registers",
        "regshow",
    };

    private static final String[] helpIntAttach = {
        "Attach to interrupt processing",
        "intattach <optns> <cMask> <cValue> <rOffs> <rValue> <vOffs>",
        "optns   Interrupt options: 1 = check, 2 = write to reset, 4 = read",
        "cOffs   The offset to the register to check",
        "cMask   The mask of bits to check",
        "rOffs   The offset to the reset register",
        "rValue  The value to use for resetting",
        "vOffs   The offset to the value to read",
    };

    private static final String[] helpIntDetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    private static final String[] helpIntshow = {
        "Show interrupt processing parameters and statistics",
        "intshow",
    };

    private static final String[] helpTimeread = {
        "Time reading from an address",
        "timeread <count> <address> <width>",
        "count    The number of times to perform the read",
        "address  The address at which to read (default board address)",
        "width    The size (1, 2 or 4) of the word to read (default 1)",
    };

    private final Output out;
    private final ConsoleReader reader;
    private final Iocard card;
    private final CmndProc proc;

    // Card parameters
    private int     baseAddr = 0,
                    cardSize = 0,
                    irqLevel = 0;
    private boolean initd = false;

    // Direct I/O parameters
    private int     dValue = 0,
                    dMask = 0,
                    dCount = 0,
                    dOffset = 0,
                    dWidth = 1;

    // Interrupt parameters
    private int     iCount = 0,
                    iOptns = 0,
                    cOffset = 0,
                    cMask = 0,
                    rOffset = 0,
                    rValue = 0,
                    vOffset = 0;

/**
 ***************************************************************************
 **
 **  Main constructor
 **
 ***************************************************************************
 */
    TestIocard(Iocard iCard, CmndProc iProc, Output iOut) throws IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("init",        CMD_INIT,        helpInit);
        cmnd.add("term",        CMD_TERM,        helpTerm);
        cmnd.add("show",        CMD_SHOW,        helpShow);
        cmnd.add("readb",       CMD_READB,       helpReadb);
        cmnd.add("readw",       CMD_READW,       helpReadw);
        cmnd.add("readl",       CMD_READL,       helpReadl);
        cmnd.add("writeb",      CMD_WRITEB,      helpWriteb);
        cmnd.add("writew",      CMD_WRITEW,      helpWritew);
        cmnd.add("writel",      CMD_WRITEL,      helpWritel);
        cmnd.add("updateb",     CMD_UPDATEB,     helpUpdateb);
        cmnd.add("updatew",     CMD_UPDATEW,     helpUpdatew);
        cmnd.add("updatel",     CMD_UPDATEL,     helpUpdatel);
        cmnd.add("regshow",     CMD_REGSHOW,     helpRegshow);
        cmnd.add("intattach",   CMD_INTATTACH,   helpIntAttach);
        cmnd.add("intdetach",   CMD_INTDETACH,   helpIntDetach);
        cmnd.add("intshow",     CMD_INTSHOW,     helpIntshow);
        cmnd.add("timeread",    CMD_TIMEREAD,    helpTimeread);

        card = (iCard == null) ? new Iocard() : iCard;
        proc = (iProc == null) ? new CmndProc() : iProc;
        proc.add(this, cmnd);

        reader  = new ConsoleReader();

    }

    TestIocard() throws IOException
    {
        this(null, null, null);
    }


/**
 ***************************************************************************
 **
 **  Main program
 **
 **  Loops reading and processing each new typed command line.
 **
 ***************************************************************************
 */
    public static void main(String[] args) throws IOException
    {
        TestIocard test = new TestIocard();
        test.run();

        System.exit(0);
    }


/**
 ***************************************************************************
 **
 **  Routine to run the test
 **
 ***************************************************************************
 */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (IocardException e) {
                out.println(e);
            }
        }
    }


/**
 ***************************************************************************
 **
 **  Routine to dispatch a command
 **
 ***************************************************************************
 */
    @Override
    public boolean dispatch(int cCode, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (cCode) {

        case CMD_INIT:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if (initd) {
                    initd = false;
                    card.term();
                }
                if ((found & 0x01) != 0) baseAddr = (Integer)args[0];
                if ((found & 0x02) != 0) cardSize = (Integer)args[1];
                if ((found & 0x04) != 0) irqLevel = (Integer)args[2];
                card.init(baseAddr, cardSize, irqLevel);
                initd = true;
            }
            break;

        case CMD_TERM:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                initd = false;
                card.term();
            }
            break;

        case CMD_SHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Initialization parameters:");
                out.format("  Base address     = %04x\n", baseAddr);
                out.format("  Card size        = %04x\n", cardSize);
                out.println("  IRQ level        = " + irqLevel);
                out.println("  Initialized      = " + initd);
            }
            break;

        case CMD_READB:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                out.format("Value = 0x%02x\n", card.readB(dOffset));
            }
            break;

        case CMD_READW:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                out.format("Value = 0x%04x\n", card.readW(dOffset));
            }
            break;

        case CMD_READL:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                out.format("Value = 0x%08x\n", card.readL(dOffset));
            }
            break;

        case CMD_WRITEB:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dValue = (Integer)args[1];
                card.writeB(dOffset, dValue);
            }
            break;

        case CMD_WRITEW:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dValue = (Integer)args[1];
                card.writeW(dOffset, dValue);
            }
            break;

        case CMD_WRITEL:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dValue = (Integer)args[1];
                card.writeL(dOffset, dValue);
            }
            break;

        case CMD_UPDATEB:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dMask = (Integer)args[1];
                if ((found & 0x04) != 0) dValue = (Integer)args[2];
                card.updateB(dOffset, dMask, dValue);
            }
            break;

        case CMD_UPDATEW:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dMask = (Integer)args[1];
                if ((found & 0x04) != 0) dValue = (Integer)args[2];
                card.updateW(dOffset, dMask, dValue);
            }
            break;

        case CMD_UPDATEL:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dOffset = (Integer)args[0];
                if ((found & 0x02) != 0) dMask = (Integer)args[1];
                if ((found & 0x04) != 0) dValue = (Integer)args[2];
                card.updateL(dOffset, dMask, dValue);
            }
            break;

        case CMD_REGSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Card Registers:");
                for (int j = 0; j < cardSize; j++) {
                    if ((j & 3) == 0 && j > 0) out.println();
                    out.format("   %02x: %02x", j, card.readB(j));
                }
                out.println();
            }
            break;

        case CMD_TIMEREAD:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dCount = (Integer)args[0];
                if ((found & 0x02) != 0) dOffset = (Integer)args[1];
                if ((found & 0x04) != 0) dWidth = (Integer)args[2];
                long msec = System.currentTimeMillis();
                if (dWidth == 4) {
                    for (int j = 0; j < dCount; j++)
                        card.readL(dOffset);
                }
                else if (dWidth == 2) {
                    for (int j = 0; j < dCount; j++)
                        card.readW(dOffset);
                }
                else {
                    for (int j = 0; j < dCount; j++)
                        card.readB(dOffset);
                }
                msec = System.currentTimeMillis() - msec;
                out.println("Performed " + dCount + " operations in " + msec
                             + " msec");
            }
            break;

        case CMD_INTATTACH:
            if ((found = CmndProc.scanArgs(scan, "iiiiii", args)) >= 0) {
                if ((found & 0x01) != 0) iOptns = (Integer)args[0];
                if ((found & 0x02) != 0) cOffset = (Integer)args[1];
                if ((found & 0x04) != 0) cMask = (Integer)args[2];
                if ((found & 0x08) != 0) rOffset = (Integer)args[3];
                if ((found & 0x10) != 0) rValue = (Integer)args[4];
                if ((found & 0x20) != 0) vOffset = (Integer)args[5];
                iCount = 0;
                card.attachInt(0, iOptns, cOffset, cMask, rOffset, rValue,
                               vOffset, this, "intProc", this);
            }
            break;

        case CMD_INTDETACH:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                card.detachInt(0);
            }
            break;

        case CMD_INTSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Interrupt Parameters:");
                out.format("  Options               = %02x\n", iOptns);
                out.format("  Check Register Offset = %02x\n", cOffset);
                out.format("  Check Mask            = %02x\n", cMask);
                out.format("  Reset Register Offset = %02x\n", rOffset);
                out.format("  Reset Value           = %02x\n", rValue);
                out.format("  Read Register Offset  = %02x\n", vOffset);
                out.println("  Count                 = " + iCount);
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


/**
 ***************************************************************************
 **
 **  Routine to process user interrupts
 **
 ***************************************************************************
 */
    private void intProc(int value, Object parm)
    {
        TestIocard tioc = (TestIocard)parm;
        iCount++;
        out.format("Interrupt occurred: count = %s, value = %02x\n",
                  tioc.iCount, value);
    }

}
