package org.lsst.ccs.drivers.iocard;

import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;


/**
 **************************************************************************
 **
 **  \file testHelios2.java
 **
 **  Program to test a Helios board's counter interrupts
 **
 **  \author Owen H Saxton
 **
 **************************************************************************
 */
public class TestHelios2 {

    private final static int
        BASE_ADDR = 0x280,
        IRQ_LEVEL = 5,
        N_REGS    = 16,
        CTRL_REG  = 0,
        PAGE_REG  = 1,
        ENAB_REG  = 4,
        TEST_REG  = 7,
        CNTR_REG  = 12,
        CCTRL_REG = 15;

    private final static Output out = new ConsOut();

    private Iocard ioc = new Iocard();


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestHelios2 test = new TestHelios2();
        test.run();
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    ***************************************************************************
    */
    public void run()
    {
        /*
        **  Initialize
        */
        ioc.init(BASE_ADDR, N_REGS, IRQ_LEVEL);

        /*
        **  Attach to interrupts
        */
        ioc.attachInt(105, Iocard.OPT_CHECK | Iocard.OPT_WRITER,
                      TEST_REG, 0x40, CTRL_REG, 0x04, 0,
                      this, "procIntr", null);

        /*
        **  Set up counter and enable interrupts
        */
        int count = 10000;             // Int rate = 100000 / count = 10 Hz
        ioc.writeB(PAGE_REG,  0);      // Select page 0 registers
        ioc.writeB(CCTRL_REG, 0x88);   // Disable counter
        ioc.writeB(ENAB_REG,  0x44);   // Select 100 KHz clock and enab ints
        ioc.writeW(CNTR_REG,  count);  // Store count value
        ioc.writeB(CCTRL_REG, 0x82);   // Load the counter
        ioc.writeB(CCTRL_REG, 0x84);   // Enable (start) the counter

        /*
        **  Sleep for a second
        */
        try {
            Thread.sleep(1000);
        }
        catch (InterruptedException e) {
        }

        /*
        **  Stop counter and disable interrupts
        */
        ioc.writeB(CCTRL_REG, 0x88); // Disable (stop) the counter
        ioc.writeB(ENAB_REG,  0x40); // Disable interrupts

        /*
        **  Detach from interrupts
        */
        ioc.detachInt(105);

        /*
        **  Terminate
        */
        ioc.term();
    }

   /**
    ***************************************************************************
    **
    **  Handle interrupts
    **
    ***************************************************************************
    */
    void procIntr(int value, Object parm)
    {
        ioc.writeB(CCTRL_REG, 0xc0);
        out.format("Flag = %02x, Data = %04x\n", value, ioc.readW(CNTR_REG));
    }
}
