package org.lsst.ccs.drivers.iocard;

import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;


/**
 **************************************************************************
 **
 **  \file testHelios1.java
 **
 **  Program to test basic I/O to a Helios board's I/O registers
 **
 **  \author Owen H Saxton
 **
 **************************************************************************
 */
public class TestHelios1 {

    private final static int
        BASE_ADDR = 0x280,
        IRQ_LEVEL = 5,
        N_REGS    = 16,
        PAGE_REG  = 1;

    private final static Output out = new ConsOut();

    private Iocard ioc = new Iocard();


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestHelios1 test = new TestHelios1();
        test.run();
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    ***************************************************************************
    */
    public void run()
    {
        /*
        **  Initialize
        */
        ioc.init(BASE_ADDR, N_REGS, IRQ_LEVEL);

        /*
        **  Read and display the registers as bytes
        */
        out.println("Register values (bytes):");
        for (int reg = 0; reg < N_REGS - 4; reg++)
            out.format(" %02x", ioc.readB(reg));
        for (int page = 0; page < 3; page++) {
            ioc.writeB(PAGE_REG, page);
            if (page != 0) out.format("%36s", "");
            for (int reg = N_REGS - 4; reg < N_REGS; reg++)
                out.format(" %02x", ioc.readB(reg));
            out.println();
        }

        /*
        **  Read and display the registers as shorts
        */
        out.println("Register values (shorts):");
        for (int reg = 0; reg < N_REGS - 4; reg += 2)
            out.format(" %04x", ioc.readW(reg));
        for (int page = 0; page < 3; page++) {
            ioc.writeB(PAGE_REG, page);
            if (page != 0) out.format("%30s", "");
            for (int reg = N_REGS - 4; reg < N_REGS; reg += 2)
                out.format(" %04x", ioc.readW(reg));
            out.println();
        }

        /*
        **  Read and display the registers as ints
        */
        out.println("Register values (ints):");
        for (int reg = 0; reg < N_REGS - 4; reg += 4)
            out.format(" %08x", ioc.readL(reg));
        for (int page = 0; page < 3; page++) {
            ioc.writeB(PAGE_REG, page);
            if (page != 0) out.format("%27s", "");
            for (int reg = N_REGS - 4; reg < N_REGS; reg += 4)
                out.format(" %08x", ioc.readL(reg));
            out.println();
        }

        /*
        **  Do out-of range operations
        */
        int reg = -1;
        try {
            ioc.readB(reg);
        }
        catch (IocardException e) {
            out.format("%s reading byte at offset %s\n", e, reg);
        }

        reg = N_REGS;
        try {
            ioc.readB(reg);
        }
        catch (IocardException e) {
            out.format("%s reading byte at offset %s\n", e, reg);
        }

        reg = N_REGS - 1;
        try {
            ioc.readW(reg);
        }
        catch (IocardException e) {
            out.format("%s reading short integer at offset %s\n", e, reg);
        }

        reg = N_REGS - 3;
        try {
            ioc.readL(reg);
        }
        catch (IocardException e) {
            out.format("%s reading integer at offset %s\n", e, reg);
        }

        /*
        **  Terminate
        */
        ioc.term();
    }
}
