package org.lsst.ccs.drivers.iocard;

import java.io.IOException;
import java.util.Scanner;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.Output;
import org.lsst.ccs.utilities.sa.ConsOut;

/**
 ***************************************************************************
 **
 **  \file  TestHelios.java
 **
 **  Program to test the Java Helios driver routines
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
public class TestHelios implements CmndProc.Dispatch {

    private final static int
        CMD_INIT        = 0,
        CMD_SHOW        = 1,
        CMD_REGSHOW     = 2,
        CMD_DIOCONFIG   = 3,
        CMD_DIOIN       = 4,
        CMD_DIOOUT      = 5,
        CMD_DIOINBIT    = 6,
        CMD_DIOOUTBIT   = 7,
        CMD_DIOSETBIT   = 8,
        CMD_DIOCLRBIT   = 9,
        CMD_DIOSHOW     = 10,
        CMD_ADCONFIG    = 11,
        CMD_ADSETCHAN   = 12,
        CMD_ADSAMPLE    = 13,
        CMD_ADSCAN      = 14,
        CMD_ADCTOV      = 15,
        CMD_ADVTOC      = 16,
        CMD_ADSHOW      = 17,
        CMD_DACONFIG    = 18,
        CMD_DASETGAIN   = 19,
        CMD_DAOUTPUT    = 20,
        CMD_DAUPDATE    = 21,
        CMD_DAVTOC      = 22,
        CMD_DACTOV      = 23,
        CMD_DASHOW      = 24,
        CMD_INTATTACH   = 25,
        CMD_INTDETACH   = 26,
        CMD_CNTRSTART   = 27,
        CMD_CNTRSTOP    = 28,
        CMD_CNTRSHOW    = 29,
        NUM_CMDS        = 30;

    private final static int
        BASE_ADDR = 0x280,
        IRQ_LEVEL = 5;

    private static final String[] helpInit = {
        "Initialize Helios card",
        "init <addr> <irq>",
        "addr   The base address of the card (initially 0x340)",
        "irq    The IRQ level of the card (initially 6)",
    };

    private static final String[] helpShow = {
        "Show initialization parameters",
        "show",
    };

    private static final String[] helpRegshow = {
        "Show register contents",
        "regshow",
    };

    private static final String[] helpDioconfig = {
        "Set digital I/O parameters",
        "dioconfig <value>",
        "value    The value to set",
    };

    private static final String[] helpDioin = {
        "Read and display a byte from a digital I/O port",
        "dioin <port>",
        "port    The digital I/O port to read",
    };

    private static final String[] helpDioout = {
        "Write a byte to a digital I/O port",
        "dioout <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private static final String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read",
        "bitnum  The number of the bit to read",
    };

    private static final String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private static final String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to set",
    };

    private static final String[] helpDioclrbit = {
        "Clear a bit in a digital I/O port",
        "dioclrbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to clear",
    };

    private static final String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private static final String[] helpAdconfig = {
        "Set the A/D configuration",
        "adconfig <gain> <polarity> <input>",
        "gain      The gain setting to use (0 - 3)",
        "polarity  The polarity: 0 = bipolar; 1 = unipolar",
        "input     The input type: single-ended = 0; differential = 1",
    };

    private static final String[] helpAdsetchan = {
        "Set the A/D channel range to use in samples and scans",
        "adsetchan <low> <high>",
        "low   The low (first) channel to use (0 - 15)",
        "high  The high (last) channel to use (0 - 15)",
    };

    private static final String[] helpAdsample = {
        "Sample and display the current A/D channel value",
        "adsample",
    };

    private static final String[] helpAdscan = {
        "Scan and display the current range of A/D channel value",
        "adscan",
    };

    private static final String[] helpAdctov = {
        "Convert an A/D count to a voltage",
        "adctov <count>",
        "count  The A/D count to convert",
    };

    private static final String[] helpAdvtoc = {
        "Convert a voltage to an A/D count",
        "advtoc <voltage>",
        "voltage  The voltage to convert",
    };

    private static final String[] helpAdshow = {
        "Show the current A/D parameters",
        "adshow",
    };

    private static final String[] helpDaconfig = {
        "Set the D/A configuration",
        "daconfig <gain> <polarity> <update> <resoln>",
        "gain      The gain setting to use (0 - 3)",
        "polarity  The polarity: 0 = bipolar; 1 = unipolar",
        "update    The update type: individual = 0; simultaneous = 1",
        "resoln    The resolution: 12-bit = 0; 16-bit = 1",
    };

    private static final String[] helpDasetgain = {
        "Set the D/A gain for a channel",
        "dasetgain <chan> <gain>",
        "chan  The channel to use (0 - 3)",
        "gain  The gain to set (0 - 3)",
    };

    private static final String[] helpDaoutput = {
        "Output a code value to a D/A channel",
        "daoutput <chan> <code>",
        "chan  The channel to use (0 - 3)",
        "code  The code value to output",
    };

    private static final String[] helpDaupdate = {
        "Update all the D/A output values (simultaneous updating mode)",
        "daupdate",
    };

    private static final String[] helpDavtoc = {
        "Convert a voltage to a D/A code",
        "davtoc <voltage>",
        "voltage  The voltage to convert",
    };

    private static final String[] helpDactov = {
        "Convert a D/A code to a voltage",
        "dactov <code>",
        "code  The D/A code to convert",
    };

    private static final String[] helpDashow = {
        "Show the current D/A parameters",
        "dashow",
    };

    private static final String[] helpIntattach = {
        "Attach to interrupt processing",
        "intattach <tmask>",
        "tmask  A bit mask of interrupt types (1 = AD, 2 = DIO, 4 = CNTR)",
    };

    private static final String[] helpIntdetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    private static final String[] helpCntrstart = {
        "Start generating counter interrupts",
        "cntrstart <rate>",
        "rate  The rate of interrupt generation (number / second)",
    };

    private static final String[] helpCntrstop = {
        "Stop generating counter interrupts",
        "cntrstop",
    };

    private static final String[] helpCntrshow = {
        "Show generated counter statistics",
        "cntrshow <optns>",
        "optns  Display option bits: 1 = intervals, 2 = counts, 4 = combined",
    };

    private static final int MAX_CNTR_SAVED = 64;

    private final Output out;
    private final Helios hel;
    private final CmndProc proc;
    private final TestIocard tioc;

    // Initialization parameters
    private int     baseAddr = BASE_ADDR,
                    irqLevel = IRQ_LEVEL;

    // Digital I/O parameters
    private int     dioPort = 0,
                    dioConf = 0,
                    dioValue = 0,
                    dioBit = 0,
                    adGain = 0;

    // A/D parameters
    private int     adLowChan = 0,
                    adHighChan = 0;
    private boolean adUnip = false,
                    adDiff = false;

    // D/A parameters
    private int     daChan = 0,
                    daGain = 0,
                    daCode = 0;
    private boolean daUnip = false,
                    daSimUp = false,
                    daHiRes = false;

    // General interrupt parameters
    private int     intMask = 0x07;

    // Counter interrupt parameters
    private long    cntrStart = 0,
                    cntrStop = 0,
                    cntrLast = 0,
                    cntrSavst = 0,
                    cntrTimes[] = new long[MAX_CNTR_SAVED];
    private int     cntrCount = 0,
                    cntrSaved = 0,
                    cntrCounts[] = new int[MAX_CNTR_SAVED];
    private float   cntrRate = 100.0F;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    TestHelios(Output iOut) throws IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("init",        CMD_INIT,        helpInit);
        cmnd.add("show",        CMD_SHOW,        helpShow);
        cmnd.add("regshow",     CMD_REGSHOW,     helpRegshow);
        cmnd.add("dioconfig",   CMD_DIOCONFIG,   helpDioconfig);
        cmnd.add("dioin",       CMD_DIOIN,       helpDioin);
        cmnd.add("dioout",      CMD_DIOOUT,      helpDioout);
        cmnd.add("dioinbit",    CMD_DIOINBIT,    helpDioinbit);
        cmnd.add("diooutbit",   CMD_DIOOUTBIT,   helpDiooutbit);
        cmnd.add("diosetbit",   CMD_DIOSETBIT,   helpDiosetbit);
        cmnd.add("dioclrbit",   CMD_DIOCLRBIT,   helpDioclrbit);
        cmnd.add("dioshow",     CMD_DIOSHOW,     helpDioshow);
        cmnd.add("adconfig",    CMD_ADCONFIG,    helpAdconfig);
        cmnd.add("adsetchan",   CMD_ADSETCHAN,   helpAdsetchan);
        cmnd.add("adsample",    CMD_ADSAMPLE,    helpAdsample);
        cmnd.add("adscan",      CMD_ADSCAN,      helpAdscan);
        cmnd.add("adctov",      CMD_ADCTOV,      helpAdctov);
        cmnd.add("advtoc",      CMD_ADVTOC,      helpAdvtoc);
        cmnd.add("adshow",      CMD_ADSHOW,      helpAdshow);
        cmnd.add("daconfig",    CMD_DACONFIG,    helpDaconfig);
        cmnd.add("dasetgain",   CMD_DASETGAIN,   helpDasetgain);
        cmnd.add("daoutput",    CMD_DAOUTPUT,    helpDaoutput);
        cmnd.add("daupdate",    CMD_DAUPDATE,    helpDaupdate);
        cmnd.add("davtoc",      CMD_DAVTOC,      helpDavtoc);
        cmnd.add("dactov",      CMD_DACTOV,      helpDactov);
        cmnd.add("dashow",      CMD_DASHOW,      helpDashow);
        cmnd.add("intattach",   CMD_INTATTACH,   helpIntattach);
        cmnd.add("intdetach",   CMD_INTDETACH,   helpIntdetach);
        cmnd.add("cntrstart",   CMD_CNTRSTART,   helpCntrstart);
        cmnd.add("cntrstop",    CMD_CNTRSTOP,    helpCntrstop);
        cmnd.add("cntrshow",    CMD_CNTRSHOW,    helpCntrshow);

        hel = new Helios();
        proc = new CmndProc();
        proc.add(this, cmnd);
        tioc = new TestIocard(hel, proc, out);
    }

    TestHelios() throws IOException
    {
        this(null);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws IOException
    {
        TestHelios test = new TestHelios();
        test.tioc.run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Routine to dispatch a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int cCode, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (cCode) {

        case CMD_INIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) baseAddr = (Integer)args[0];
                if ((found & 0x02) != 0) irqLevel = (Integer)args[1];
                hel.init(baseAddr, irqLevel);
                dioConf = hel.dioGetConfig();
                int[] chans = hel.adGetChans();
                adLowChan = chans[0];
                adHighChan = chans[1];
                adGain = hel.adGetGain();
                adUnip = hel.adIsUnipolar();
                adDiff = hel.adIsDifferential();
                daUnip = hel.daIsUnipolar();
                daSimUp = hel.daIsSimUpdate();
                daHiRes = hel.daIsHighRes();
            }
            break;

        case CMD_SHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Initialization parameters:");
                out.format("  Base address     = %04x\n", baseAddr);
                out.format("  Card size        = %04x\n", Helios.N_REGS);
                out.println("  IRQ level        = " + irqLevel);
            }
            break;

        case CMD_REGSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                regShow();
            }
            break;

        case CMD_DIOCONFIG:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dioConf = (Integer)args[0];
                hel.dioConfig(dioConf);
            }
            break;

        case CMD_DIOIN:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                int value = hel.dioInp(dioPort);
                out.format("Value = 0x%02x\n", value);
            }
            break;

        case CMD_DIOOUT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioValue = (Integer)args[1];
                hel.dioOut(dioPort, dioValue);
            }
            break;

        case CMD_DIOINBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                int value = hel.dioInpBit(dioPort, dioBit);
                out.println("Value = " + value);
            }
            break;

        case CMD_DIOOUTBIT:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                if ((found & 0x04) != 0) dioValue = (Integer)args[2];
                hel.dioOutBit(dioPort, dioBit, dioValue);
            }
            break;

        case CMD_DIOSETBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                hel.dioSetBit(dioPort, dioBit);
            }
            break;

        case CMD_DIOCLRBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                hel.dioClrBit(dioPort, dioBit);
            }
            break;

        case CMD_DIOSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Digital I/O parameters:");
                out.format("  Configuration = %02x\n", dioConf);
                out.println("  Port          = " + dioPort);
                out.format("  Output value  = %02x\n", dioValue);
                out.println("  Bit number    = " + dioBit);
            }
            break;

        case CMD_ADCONFIG:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) adGain = (Integer)args[0];
                if ((found & 0x02) != 0) adUnip = (Integer)args[1] != 0;
                if ((found & 0x04) != 0) adDiff = (Integer)args[2] != 0;
                hel.adConfig(adGain, adUnip, adDiff);
            }
            break;

        case CMD_ADSETCHAN:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) adLowChan = (Integer)args[0];
                if ((found & 0x02) != 0) adHighChan = (Integer)args[1];
                hel.adSetChans(adLowChan, adHighChan);
            }
            break;

        case CMD_ADSAMPLE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                int chan = hel.adGetCurrChan();
                short value = hel.adSample();
                float volts = hel.adCountToVolts(value);
                out.format("Channel %s: %04x ", chan, value);
                if (volts > -1.0 && volts < 1.0)
                    out.format("(%.5f)\n", volts);
                else
                    out.format("(%.6g)\n", volts);
            }
            break;

        case CMD_ADSCAN:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                short[] value = new short[Helios.N_AD_CHANS];
                int count = hel.adScan(value);
                for (int j = 0; j < count; j++) {
                    int chan = adLowChan + j;
                    if (chan >= Helios.N_AD_CHANS)
                        chan -= Helios.N_AD_CHANS;
                    if (j > 0) out.print(((j & 3) == 0) ? "\n" : " ");
                    float volts = hel.adCountToVolts(value[j]);
                    out.format("%2s: %04x ", chan, value[j]);
                    if (volts > -1.0 && volts < 1.0)
                        out.format("(% -8.5f)", volts);
                    else
                        out.format("(% -8.6g)", volts);
                }
                out.println();
            }
            break;

        case CMD_ADCTOV:
            if ((found = CmndProc.scanArgs(scan, "W", args)) >= 0) {
                float volts = hel.adCountToVolts((Short)args[0]);
                if (volts > -1.0 && volts < 1.0)
                    out.format("Value = %.5f\n", volts);
                else
                    out.format("Value = %.6g\n", volts);
            }
            break;

        case CMD_ADVTOC:
            if ((found = CmndProc.scanArgs(scan, "F", args)) >= 0) {
                short count = hel.adVoltsToCount((Float)args[0]);
                out.format("Value = %04x\n", count);
            }
            break;

        case CMD_ADSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("A/D Parameters:");
                out.println("  Gain setting       = " + adGain);
                out.println("  Unipolar input     = " + adUnip);
                out.println("  Differential input = " + adDiff);
                out.println("  Low channel        = " + adLowChan);
                out.println("  High channel       = " + adHighChan);
            }
            break;

        case CMD_DACONFIG:
            if ((found = CmndProc.scanArgs(scan, "iiii", args)) >= 0) {
                if ((found & 0x01) != 0) daGain = (Integer)args[0];
                if ((found & 0x02) != 0) daUnip = (Integer)args[1] != 0;
                if ((found & 0x04) != 0) daSimUp = (Integer)args[2] != 0;
                if ((found & 0x08) != 0) daHiRes = (Integer)args[3] != 0;
                hel.daConfig(daGain, daUnip, daSimUp, daHiRes);
            }
            break;

        case CMD_DASETGAIN:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) daChan = (Integer)args[0];
                if ((found & 0x02) != 0) daGain = (Integer)args[1];
                hel.daSetGain(daChan, daGain);
            }
            break;

        case CMD_DAOUTPUT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) daChan = (Integer)args[0];
                if ((found & 0x02) != 0) daCode = (Integer)args[1];
                hel.daOutput(daChan, daCode);
            }
            break;

        case CMD_DAUPDATE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                hel.daUpdate();
            }
            break;

        case CMD_DAVTOC:
            if ((found = CmndProc.scanArgs(scan, "iF", args)) >= 0) {
                if ((found & 0x01) != 0) daChan = (Integer)args[0];
                daCode = hel.daVoltsToCode(daChan, (Float)args[1]);
                out.format("Value = %04x\n", daCode);
            }
            break;

        case CMD_DACTOV:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) daChan = (Integer)args[0];
                if ((found & 0x02) != 0) daCode = (Integer)args[1];
                float volts = hel.daCodeToVolts(daChan, daCode);
                if (volts > -1.0 && volts < 1.0)
                    out.format("Value = %.5f\n", volts);
                else
                    out.format("Value = %.6g\n", volts);
            }
            break;

        case CMD_DASHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("D/A Parameters:");
                out.println("  Gain setting        = " + daGain);
                out.println("  Unipolar output     = " + daUnip);
                out.println("  Simultaneous update = " + daSimUp);
                out.println("  High resolution     = " + daHiRes);
                out.println("  Channel             = " + daChan);
                out.println("  Code                = " + daCode);
            }
            break;

        case CMD_INTATTACH:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) intMask = (Integer)args[0];
                hel.attachInt(0, Helios.OPT_CHECK | Helios.OPT_WRITER,
                              Helios.INTAD_STS_REG, Helios.PEND_INTS,
                              Helios.CMND_REG, Helios.CMND_CLEAR_INTS, 0,
                              this, "reportInt", hel);
                hel.updateB(Helios.INT_CTL_REG, Helios.ENAB_INTS, intMask);
            }
            break;

        case CMD_INTDETACH:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                hel.updateB(Helios.INT_CTL_REG, Helios.ENAB_INTS, 0);
                hel.detachInt(0);
            }
            break;

        case CMD_CNTRSTART:
            if ((found = CmndProc.scanArgs(scan, "f", args)) >= 0) {
                if ((found & 0x01) != 0) cntrRate = (Float)args[0];
                cntrSavst = cntrStart = cntrLast = System.nanoTime();
                cntrStop = 0;
                cntrCount = 0;
                cntrSaved = 0;
                hel.cntrConfig(cntrRate, false);
                hel.cntrEnable(this, "cntrProc", null);
                hel.cntrStart();
            }
            break;

        case CMD_CNTRSTOP:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                hel.cntrDisable();
                cntrStop = System.nanoTime();
            }
            break;

        case CMD_CNTRSHOW:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                cntrShow((found & 0x01) != 0 ? (Integer)args[0] : 0);
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Display the contents of a board's registers
    **
    ***************************************************************************
    */
    private final static String[] regDesc = {
        "| AD7  | AD6  | AD5  | AD4  | AD3  | AD2  | AD1  | AD0  |",
        "| AD15 | AD14 | AD13 | AD12 | AD11 | AD10 | AD9  | AD8  |",
        "|  H3  |  H2  |  H1  |  H0  |  L3  |  L2  |  L1  |  L0  |",
        "|ADBUSY|SE/DIF|ADWAIT|DACBSY|  OF  |SCANEN| ADG1 | ADG0 |",
        "|CKSEL1|FRQSL1|FRQSL0|ADCLK |  -   |TINTE |DINTE |AINTE |",
        "|  -   |  -   | FT5  | FT4  | FT3  | FT2  | FT1  | FT0  |",
        "|  -   |  -   | FD5  | FD4  | FD3  | FD2  | FD1  | FD0  |",
        "|  -   | TINT | DINT | AINT |ADCH3 |ADCH2 |ADCH1 |ADCH0 |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |",
        "|  C7  |  C6  |  C5  |  C4  |  C3  |  C2  |  C1  |  C0  |",
        "|DIOCTR|DAMODE|DASIM | DIRA |DIRCH |  -   | DIRB |DIRCL |",
        "|CTRD7 |CTRD6 |CTRD5 |CTRD4 |CTRD3 |CTRD2 |CTRD1 |CTRD0 |",
        "|CTRD15|CTRD14|CTRD13|CTRD12|CTRD11|CTRD10|CTRD9 |CTRD8 |",
        "|CTRD23|CTRD22|CTRD21|CTRD20|CTRD19|CTRD18|CTRD17|CTRD16|",
        "|                  FPGA Revision Code                   |",
        "|  D7  |  D6  |  D5  |  D4  |  D3  |  D2  |  D1  |  D0  |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  -   |TDBUSY|EEBUSY|CALMUX|  -   |  -   |  -   |  -   |",
        "|                       Not Used                        |",
        "|  -   |  -   |  -   |  -   |  -   |  -   |  -   |EXFIFO|",
        "| DAUR |DACH1 |DACH0 |  -   |ADPOL |  -   | ADSD |  -   |",
        "|  -   |  -   |  -   |DASIZE|DAPOL |  -   | DAG1 | DAG0 |",
        "|                D/A Simultaneous Update                |",
        "| FD07 | FD06 | FD05 | FD04 | FD03 | FD02 | FD01 | FD00 |",
        "| FD11 | FD10 | FD09 | FD08 |  OF  |  FF  |  HF  |  EF  |",
    };

    private final static String[] pageDesc = {
        "Counter/Timer Access",
        "AutoCal Control",
        "Expanded FIFO and AD/DA Control"
    };

    private void regShow()
    {
        int[] regs = new int[Helios.N_REGS + 8];
        for (int r = 0; r < Helios.N_REGS - 4; r++)
            regs[r] = hel.readB(r);
        for (int p = 0; p < 3; p++) {
            hel.writeB(Helios.PAGE_REG, p);
            for (int r = Helios.N_REGS - 4; r < Helios.N_REGS; r++)
                regs[4 * p + r] = hel.readB(r);
        }
        out.println("Main registers:");
        for (int r = 0; r < Helios.N_REGS - 4; r++) {
            if ((r == Helios.FIFO_THR_REG || r == Helios.FIFO_STS_REG)
                  && ((regs[Helios.EXP_FIFO_REG + 8] & 1) != 0))
                out.format("  %2s: %02x  %s\n", r, regs[r], regDesc[r + 19]);
            else
                out.format("  %2s: %02x  %s\n", r, regs[r], regDesc[r]);
        }
        for (int p = 0; p < 3; p++) {
            out.format("\nPage %s: %s\n", p, pageDesc[p]);
            for (int r = Helios.N_REGS - 4; r < Helios.N_REGS; r++) {
                int ri = 4 * p + r;
                out.format("  %2s: %02x  %s\n", r, regs[ri], regDesc[ri]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Display counter interrupt statistics
    **
    ***************************************************************************
    */
    void cntrShow(int option)
    {
        boolean times = false, counts = false, comb = false;
        if ((option & 0x01) != 0) times = true;
        if ((option & 0x02) != 0) counts = true;
        if ((option & 0x04) != 0) comb = true;
        long intvl = cntrStop;
        float rate = 0.0F;
        int count = cntrCount;
        if (intvl == 0)
            intvl = System.nanoTime();
        intvl -= cntrStart;
        if (intvl != 0)
            rate = 1.0e9F * count / intvl;
        out.println("Interrupt Parameters:");
        out.println("  Req. Rate    = " + cntrRate);
        out.println("  Count        = " + cntrCount);
        out.println("  Actual Rate  = " + rate);
        if (times) {
            out.print("  Interval samples (nsec):");
            long prev = cntrSavst;
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 3) == 0) out.println();
                out.format("%14s", cntrTimes[j] - prev);
                prev = cntrTimes[j];
            }
            out.println();
        }
        if (counts) {
            out.print("  Counter samples:");
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 7) == 0) out.println();
                out.format("%7s", cntrCounts[j]);
            }
            out.println();
        }
        if (comb) {
            out.println("  Combined samples:");
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "Intvl(ns)", "Cntr",
                           "Dvn(ns)", "Dly(ns)");
            out.println();
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "---------", "----",
                           "-------", "-------");
            long prev = cntrSavst;
            int expTime = (int)(1000000000.0 / cntrRate);
            boolean low
              = (hel.readB(Helios.INT_CTL_REG) & Helios.CNTR1_USE_LOW) != 0;
            int mult = 1000000000
                        / (low ? Helios.CNTR1_LOW_FREQ : Helios.CNTR_HIGH_FREQ);
            int cntMax = expTime / mult;
            for (int j = 0; j < cntrSaved; j++) {
                if ((j & 1) == 0) out.println();
                int period = (int)(cntrTimes[j] - prev);
                int delay = cntMax - cntrCounts[j];
                if (delay < 0) delay += 0x10000;
                out.format("%13s %5s %8s %8s", period, cntrCounts[j],
                           period - expTime, mult * delay);
                prev = cntrTimes[j];
            }
            out.println();
        }
        if (times || counts || comb) {
            cntrSaved = 0;
            cntrSavst = cntrLast;
        }
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to report interrupts
    **
    ***************************************************************************
    */
    private void reportInt(int value, Object parm)
    {
        Helios heli = (Helios)parm;

        out.format("Flags = %02x, DIO lines = %06x\n", value,
                   heli.readL(Helios.PORTA_REG) & 0xffffff);
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to process counter interrupts
    **
    ***************************************************************************
    */
    private void cntrProc(int value, Object parm)
    {
        cntrLast = System.nanoTime();
        if (cntrSaved < MAX_CNTR_SAVED) {
            cntrTimes[cntrSaved] = cntrLast;
            cntrCounts[cntrSaved++] = hel.cntrRead();
        }
        cntrCount++;
    }

}
