package org.lsst.ccs.drivers.iocard;

import java.io.IOException;
import java.util.Scanner;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.Output;
import org.lsst.ccs.utilities.sa.ConsOut;

/**
 ***************************************************************************
 **
 **  \file  TestAccesDio.java
 **
 **  Program to test the Java DSC universal driver routines
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAccesDio implements CmndProc.Dispatch {

    private final static int
        CMD_INIT        = 0,
        CMD_DIOCONF     = 1,
        CMD_DIOIN       = 2,
        CMD_DIOOUT      = 3,
        CMD_DIOINBIT    = 4,
        CMD_DIOOUTBIT   = 5,
        CMD_DIOSETBIT   = 6,
        CMD_DIOCLRBIT   = 7,
        CMD_DIOSHOW     = 8,
        CMD_INTATTACH   = 9,
        CMD_INTDETACH   = 10,
        CMD_INTSTART    = 11,
        CMD_INTSTOP     = 12,
        CMD_INTSHOW     = 13,
        CMD_SHOW        = 14,
        CMD_REGSHOW     = 15,
        NUM_CMDS        = 16;

    private final static int
        BASE_ADDR = 0x340,
        IRQ_LEVEL = 6;

    private static final String[] helpInit = {
        "Initialize Acces DIO card",
        "init <addr> <irq>",
        "addr   The base address of the card (initially 0x340)",
        "irq    The IRQ level of the card (initially 6)",
    };

    private static final String[] helpShow = {
        "Show initialization parameters",
        "show",
    };

    private static final String[] helpRegshow = {
        "Show register contents",
        "regshow",
    };

    private static final String[] helpDioconf = {
        "Set digital I/O parameters",
        "dioconf <value>",
        "value    The value to set",
    };

    private static final String[] helpDioin = {
        "Read and display a byte from a digital I/O port",
        "dioin <port>",
        "port    The digital I/O port to read",
    };

    private static final String[] helpDioout = {
        "Write a byte to a digital I/O port",
        "dioout <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private static final String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read",
        "bitnum  The number of the bit to read",
    };

    private static final String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private static final String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to set",
    };

    private static final String[] helpDioclrbit = {
        "Clear a bit in a digital I/O port",
        "dioclrbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to clear",
    };

    private static final String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private static final String[] helpIntattach = {
        "Attach to interrupt processing",
        "intattach <pmask>",
        "pmask  A bit mask of the ports (0 - 2) to trigger interrupts",
    };

    private static final String[] helpIntdetach = {
        "Detach from interrupt processing",
        "intdetach",
    };

    private static final String[] helpIntstart = {
        "Start generating interrupts",
        "intstart <port> <rate>",
        "port  The port to use for generating interrupts",
        "rate  The rate of interrupt generation (number / second)",
    };

    private static final String[] helpIntstop = {
        "Stop generating interrupts",
        "intstop",
    };

    private static final String[] helpIntshow = {
        "Show generated interrupt statistics",
        "intshow",
    };

    private final Output out;
    private final AccesDio acc;
    private final CmndProc proc;
    private final TestIocard tioc;

    private int   baseAddr = BASE_ADDR,
                  irqLevel = IRQ_LEVEL,
                  dioPort = 0,
                  dioConf = 0,
                  dioValue = 0,
                  dioBit = 0,
                  intPort = 0,
                  intMask = 0x07;

    private float intRate = 1F;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    TestAccesDio(Output iOut) throws IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("init",        CMD_INIT,        helpInit);
        cmnd.add("show",        CMD_SHOW,        helpShow);
        cmnd.add("regshow",     CMD_REGSHOW,     helpRegshow);
        cmnd.add("dioconf",     CMD_DIOCONF,     helpDioconf);
        cmnd.add("dioin",       CMD_DIOIN,       helpDioin);
        cmnd.add("dioout",      CMD_DIOOUT,      helpDioout);
        cmnd.add("dioinbit",    CMD_DIOINBIT,    helpDioinbit);
        cmnd.add("diooutbit",   CMD_DIOOUTBIT,   helpDiooutbit);
        cmnd.add("diosetbit",   CMD_DIOSETBIT,   helpDiosetbit);
        cmnd.add("dioclrbit",   CMD_DIOCLRBIT,   helpDioclrbit);
        cmnd.add("dioshow",     CMD_DIOSHOW,     helpDioshow);
        cmnd.add("intattach",   CMD_INTATTACH,   helpIntattach);
        cmnd.add("intdetach",   CMD_INTDETACH,   helpIntdetach);
        cmnd.add("intstart",    CMD_INTSTART,    helpIntstart);
        cmnd.add("intstop",     CMD_INTSTOP,     helpIntstop);
        cmnd.add("intshow",     CMD_INTSHOW,     helpIntshow);

        acc = new AccesDio();
        proc = new CmndProc();
        proc.add(this, cmnd);
        tioc = new TestIocard(acc, proc, out);
    }

    TestAccesDio() throws IOException
    {
        this(null);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws IOException
    {
        TestAccesDio test = new TestAccesDio();
        test.tioc.run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Routine to dispatch a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int cCode, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (cCode) {

        case CMD_INIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) baseAddr = (Integer)args[0];
                if ((found & 0x02) != 0) irqLevel = (Integer)args[1];
                acc.init(baseAddr, irqLevel);
            }
            break;

        case CMD_SHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Initialization parameters:");
                out.format("  Base address     = %04x\n", baseAddr);
                out.format("  Card size        = %04x\n", AccesDio.N_REGS);
                out.println("  IRQ level        = " + irqLevel);
            }
            break;

        case CMD_REGSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                regShow();
            }
            break;

        case CMD_DIOCONF:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dioConf = (Integer)args[0];
                acc.dioConfig(dioConf);
            }
            break;

        case CMD_DIOIN:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                int value = acc.dioInp(dioPort);
                out.format("Value = 0x%02x\n", value);
            }
            break;

        case CMD_DIOOUT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioValue = (Integer)args[1];
                acc.dioOut(dioPort, dioValue);
            }
            break;

        case CMD_DIOINBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                int value = acc.dioInpBit(dioPort, dioBit);
                out.println("Value = " + value);
            }
            break;

        case CMD_DIOOUTBIT:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                if ((found & 0x04) != 0) dioValue = (Integer)args[2];
                acc.dioOutBit(dioPort, dioBit, dioValue);
            }
            break;

        case CMD_DIOSETBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                acc.dioSetBit(dioPort, dioBit);
            }
            break;

        case CMD_DIOCLRBIT:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0) dioPort = (Integer)args[0];
                if ((found & 0x02) != 0) dioBit = (Integer)args[1];
                acc.dioClrBit(dioPort, dioBit);
            }
            break;

        case CMD_DIOSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Digital I/O parameters:");
                out.format("  Configuration = %02x\n", dioConf);
                out.println("  Port          = " + dioPort);
                out.format("  Output value  = %02x\n", dioValue);
                out.println("  Bit number    = " + dioBit);
            }
            break;

        case CMD_INTATTACH:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0) intMask = (Integer)args[0];
                acc.attachInt(intMask, this, "reportInt", acc);
            }
            break;

        case CMD_INTDETACH:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                acc.detachInt();
            }
            break;

        case CMD_INTSTART:
            if ((found = CmndProc.scanArgs(scan, "if", args)) >= 0) {
                if ((found & 0x01) != 0) intPort = (Integer)args[0];
                if ((found & 0x02) != 0) intRate = (Float)args[1];
            }
            break;

        case CMD_INTSTOP:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
            }
            break;

        case CMD_INTSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Callback routine to report interrupts
    **
    ***************************************************************************
    */
    private void reportInt(int value, Object parm)
    {
        AccesDio dio = (AccesDio)parm;

        out.format("Flags = %02x, DIO lines = %06x\n", value,
                   dio.readL(AccesDio.PORTA_REG) & 0xffffff);
    }


   /**
    ***************************************************************************
    **
    **  Routine to display the contents of a board's registers
    **
    ***************************************************************************
    */
    private final static String[] regDesc = {
        "|G0 A7 |G0 A6 |G0 A5 |G0 A4 |G0 A3 |G0 A2 |G0 A1 |G0 A0 |",
        "|G0 B7 |G0 B6 |G0 B5 |G0 B4 |G0 B3 |G0 B2 |G0 B1 |G0 B0 |",
        "|G0 C7 |G0 C6 |G0 C5 |G0 C4 |G0 C3 |G0 C2 |G0 C1 |G0 C0 |",
        "|                G0 Control (Write only)                |",
        "|G1 A7 |G1 A6 |G1 A5 |G1 A4 |G1 A3 |G1 A2 |G1 A1 |G1 A0 |",
        "|G1 B7 |G1 B6 |G1 B5 |G1 B4 |G1 B3 |G1 B2 |G1 B1 |G1 B0 |",
        "|G1 C7 |G1 C6 |G1 C5 |G1 C4 |G1 C3 |G1 C2 |G1 C1 |G1 C0 |",
        "|                G1 Control (Write only)                |",
        "|         G0 Enable/Disable Buffer (Write only)         |",
        "|         G1 Enable/Disable Buffer (Write only)         |",
        "|  -   |  -   |  -   |  -   |  -   |  -   |G1 TEN|G0 TEN|",
        "|  -   |G1 IDC|G1 IDB|G1 IDA|  -   |G0 IDC|G0 IDB|G0 IDA|",
        "|  -   |  -   |  -   |G0 EA |G0 ECH|  -   |G0 EB |G0 ECL|",
        "|  -   |  -   |  -   |G1 EA |G1 ECH|  -   |G1 EB |G1 ECL|",
        "|                       Not Used                        |",
        "|G1 ICH|G1 ICL|G1 IB |G1 IA |G0 ICH|G0 DCL|G0 IB |G0 IA |",
        "|                    Counter/Timer 0                    |",
        "|                    Counter/Timer 1                    |",
        "|                    Counter/Timer 2                    |",
        "|            Counter/Timer Mode (Write only)            |",
        "|                Counter/Timer Interrupt                |",
        "|               Counter/Timer IRQ Status                |",
        "|             Counter/Timer Interrupt Clear             |",
        "|                       Not Used                        |",
        "|G1 ICH|G1 ICL|G1 IB |G1 IA |G0 ICH|G0 DCL|G0 IB |G0 IA |",
    };

    private void regShow()
    {
        for (int r = 0; r < AccesDio.N_REGS; r++)
            out.format("  %2s: %02x  %s\n", r, acc.readB(r), regDesc[r]);
    }

}
