package org.lsst.ccs.drivers.iocard;

import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;


/**
 **************************************************************************
 **
 **  \file testAcces2.java
 **
 **  Program to test the Acces DIO card's interrupt capabilities
 **
 **  \author Owen H Saxton
 **
 **************************************************************************
 */
public class TestAcces2 {

    private final static int
        BASE_ADDR = 0x340,
        IRQ_LEVEL = 6,
        N_REGS    = 25,
        PORTA_REG = 0,
        PORTB_REG = 1,
        PORTC_REG = 2,
        CONF_REG  = 3,
        SETUP_REG = 11,
        STATE_REG = 15,
        RESET_REG = 24;

    private final static Output out = new ConsOut();

    private Iocard ioc = new Iocard();


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestAcces2 test = new TestAcces2();
        int intvl = 100;
        if (args.length >= 1)
            intvl = Integer.valueOf(args[0]);
        test.run(intvl, args.length > 1);
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    ***************************************************************************
    */
    public void run(int intvl, boolean rev)
    {
        /*
        **  Initialize
        */
        ioc.init(BASE_ADDR, N_REGS, IRQ_LEVEL);
        ioc.writeB(CONF_REG, 0x80);    // All output

        /*
        **  Test interrupts a few times
        */
        testIntr(0, 9, intvl, rev);
        testIntr(8, 9, intvl, rev);
        testIntr(16, 9, intvl, rev);
        testIntr(24, 9, intvl, rev);

        /*
        **  Terminate
        */
        ioc.term();
    }


   /**
    **************************************************************************
    **
    **  Interrupt test routine
    **
    **************************************************************************
    */
    void testIntr(int first, int count, int intvl, boolean rev)
    {
        /*
        **  Attach to interrupts and enable them
        */
        ioc.attachInt(0, Iocard.OPT_CHECK, RESET_REG, 0x0f,
                      RESET_REG, 0, 0, this, "procIntr", (Boolean)rev);
        ioc.writeB(SETUP_REG, rev ? 0x03 : 0x05);

        /*
        **  Write values to enabled register
        */
        for (int j = first; j < (first + count); j++) {
            ioc.writeB(rev ? PORTC_REG : PORTB_REG, j);
            try {
                Thread.sleep(intvl);
            }
            catch (InterruptedException e) {
            }
        }

        /*
        **  Disable interrupts and detach
        */
        ioc.writeB(SETUP_REG, 0x07);   // No interrupts
        ioc.detachInt(0);
    }


   /**
    **************************************************************************
    **
    **  Interrupt callback routine
    **
    **************************************************************************
    */
    void procIntr(int value, Object parm)
    {
        boolean rev = (Boolean)parm;
        out.format("Flag = %02x, Data = %02x\n", value,
                   ioc.readB(rev ? PORTC_REG : PORTB_REG));
    }

}
