package org.lsst.ccs.drivers.indi.sbig;

import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.indilib.i4j.Constants;
import org.indilib.i4j.Constants.SwitchStatus;
import org.indilib.i4j.client.INDIBLOBElement;
import org.indilib.i4j.client.INDIDevice;
import org.indilib.i4j.client.INDIDeviceListener;
import org.indilib.i4j.client.INDIElement;
import org.indilib.i4j.client.INDIElementListener;
import org.indilib.i4j.client.INDIProperty;
import org.indilib.i4j.client.INDIServerConnection;
import org.indilib.i4j.client.INDIValueException;
import org.indilib.i4j.protocol.url.INDIURLStreamHandlerFactory;

/**
 *
 * @author tonyj
 */
public class SBIGCamera implements AutoCloseable {

    private static final Logger LOG = Logger.getLogger(SBIGCamera.class.getName());

    static {
        INDIURLStreamHandlerFactory.init();
    }

    private final INDIServerConnection indiServerConnection;
    private final INDIDevice sbig;
    private INDIBLOBElement blobElement;

    public SBIGCamera() throws IOException {
        this("localhost");
    }

    public SBIGCamera(String host) throws IOException {
        this(host, Constants.INDI_DEFAULT_PORT);
    }

    public SBIGCamera(String host, int port) throws IOException {
        indiServerConnection = new INDIServerConnection(host, port);
        indiServerConnection.connect();
        try {
            indiServerConnection.askForDevices();
            sbig = indiServerConnection.waitForDevice("SBIG CCD", 2);
            if (sbig == null) {
                throw new IOException("No SBIG CCD device found");
            }
            sbig.blobsEnable(Constants.BLOBEnables.ALSO);
            sbig.addINDIDeviceListener(new DeviceListener());
            INDIElement connect = sbig.getElement("CONNECTION", "CONNECT");
            if ("OFF".equals(connect.getValueAsString())) {
                try {
                    connect.setDesiredValue(SwitchStatus.ON);
                    connect.getProperty().sendChangesToDriver();
                    Thread.sleep(1000);
                } catch (INDIValueException | InterruptedException x) {
                    throw new IOException("Error connecting to device", x);
                }
            }
            INDIProperty<?> property = sbig.waitForProperty("CCD1", 2);
            blobElement = (INDIBLOBElement) property.getElement("CCD1");
            if (blobElement == null) {
                throw new IOException("Unable to access element CCC1/CCD1");
            }
        } catch (IOException x) {
            indiServerConnection.disconnect();
            throw x;
        }
    }
    
    
    public static void main(String[] args) throws IOException, InterruptedException, INDIValueException {
        try (SBIGCamera sbigCamera = new SBIGCamera("lsst-uno12")) {
            File file = sbigCamera.exposeAndWriteFile(5.0, "FRAME_LIGHT", "test.fits");
            System.out.println(file);
            System.out.println(sbigCamera.getTemperature());
            System.out.println(sbigCamera.isCooling());
            sbigCamera.setCooling(false);
            System.out.println(sbigCamera.isCooling());
        }
    }
    
    public File exposeAndWriteFile(double seconds, String frameType, String file) throws IOException {
        File f = new File(file);
        exposeAndWriteFile(seconds, frameType, f);
        return f;
    }

    public void exposeAndWriteFile(double seconds, String frameType, File file) throws IOException {
        CompletableFuture<INDIBLOBElement> expose = expose(seconds, frameType);
        try {
            INDIBLOBElement blob = expose.get((long) (seconds+10), TimeUnit.SECONDS);
            blob.getValue().saveBLOBData(file);
            LOG.log(Level.INFO, "Wrote {0} of size {1}", new Object[]{file, blob.getValue().getSize()});
        } catch (InterruptedException x) {
            throw new InterruptedIOException("Interruption while waiting for exposure");
        } catch (ExecutionException ex) {
            throw new IOException("Error taking exposure", ex.getCause());
        } catch (TimeoutException ex) {
            throw new IOException("Exposure timed out", ex);
        }
    }

    public CompletableFuture<INDIBLOBElement> expose(double seconds, String frameType) throws IOException {
        CompletableFuture<INDIBLOBElement> future = new CompletableFuture<>();
        try {
            blobElement.addINDIElementListener(new ElementListener(future));
            
            INDIProperty<?> frameProperty = sbig.getProperty("CCD_FRAME_TYPE");
            if (!Arrays.asList(frameProperty.getElementNames()).contains(frameType)) {
                throw new IOException("Frame type frameType not in supported types "+Arrays.toString(frameProperty.getElementNames()));
            }
            INDIElement frameElement = frameProperty.getElement(frameType);
            frameElement.setDesiredValue(SwitchStatus.ON);
            frameProperty.sendChangesToDriver();
            
            INDIProperty<?> exposureProperty = sbig.getProperty("CCD_EXPOSURE");
            INDIElement element = exposureProperty.getElement("CCD_EXPOSURE_VALUE");
            element.setDesiredValue(seconds);
            exposureProperty.sendChangesToDriver();
        } catch (INDIValueException x) {
            future.completeExceptionally(x);
        }
        return future;
    }
    
    public double getTemperature() {
        INDIProperty<?> property = sbig.getProperty("CCD_TEMPERATURE");
        INDIElement element = property.getElement("CCD_TEMPERATURE_VALUE");
        return (double) element.getValue();
    }
    
    public boolean isCooling() {
        INDIProperty<?> property = sbig.getProperty("CCD_COOLER");
        INDIElement element = property.getElement("COOLER_ON");
        return element.getValue() == SwitchStatus.ON;
    }

    public void setCooling(boolean on) throws IOException {
        try {
            INDIProperty<?> property = sbig.getProperty("CCD_COOLER");
            INDIElement element = property.getElement(on ? "COOLER_ON" : "COOLER_OFF");
            element.setDesiredValue(SwitchStatus.ON);
            property.sendChangesToDriver();
        } catch (INDIValueException x) {
            throw new IOException("Unable to set cooling ", x);
        }
    }
    
    public INDIProperty<?> getProperty(String name) {
        return sbig.getProperty(name);
    }
    
    public String[] getProperties() {
        return sbig.getPropertyNames();
    }
    
    @Override
    public void close() throws IOException {
        indiServerConnection.disconnect();
    }


    
    static class DeviceListener implements INDIDeviceListener {
        @Override
        public void newProperty(INDIDevice device, INDIProperty<?> property) {
        }

        @Override
        public void removeProperty(INDIDevice device, INDIProperty<?> property) {
        }

        @Override
        public void messageChanged(INDIDevice device) {
            String lastMessage = device.getLastMessage();
            LOG.log(Level.INFO, lastMessage);
        }
        
    }
    
    static class ElementListener implements INDIElementListener {

        private final CompletableFuture<INDIBLOBElement> future;

        private ElementListener(CompletableFuture<INDIBLOBElement> future) {
            this.future = future;
        }
        
        @Override
        public void elementChanged(INDIElement element) {
            future.complete((INDIBLOBElement) element);
        }    
    }


}

