package org.lsst.ccs.drivers.i2c;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for communicating with an I2C device
 *
 *  @author  Owen Saxton
 */
public class I2c {

    /**
     *  Implementation interface.
     */
    static interface Impl {
        
        void open(ConnType connType, String ident, int param)
            throws DriverException;

        void close() throws DriverException;

        default void setTimeout(double timeout) throws DriverException
        {
            throw new DriverException("setTimeout method not implemented");
        }

        default void write(int addr, int value) throws DriverException
        {
            throw new DriverException("Write single method not implemented");
        }

        default int read(int addr) throws DriverException
        {
            throw new DriverException("Read single method not implemented");
        }

        default int read(int addr, byte[] buff, int count)
            throws DriverException
        {
            throw new DriverException("Read multiple method not implemented");
        }

        default void write(int addr, int reg, byte[] buff, int count)
            throws DriverException
        {
            throw new DriverException("Write method not implemented");
        }

        default int read(int addr, int reg, byte[] buff, int count)
            throws DriverException
        {
            throw new DriverException("Read method not implemented");
        }

        default void write2(int addr, int reg, byte[] buff, int count)
            throws DriverException {
            throw new DriverException("Write method not implemented");
        }

        default int read2(int addr, int reg, byte[] buff, int count)
            throws DriverException
        {
            throw new DriverException("Read method not implemented");
        }

    }

    /**
     *  Inner class for storing open connection data.
     */
    static class OpenDesc {

        Impl impl;
        int refCount = 0;

    }

    /**
     *  Constants & data.
     */
    public enum ConnType {
        SERIALUSB, FTDIUSB;
    }

    private static final Map<String, OpenDesc> openMap = new HashMap<>();
    private OpenDesc desc;


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type
     *  @param  ident     The identification
     *  @param  param     The parameter
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident, int param) throws DriverException
    {
        if (desc != null) {
            throw new DriverException("Connection already open");
        }
        desc = getOpen(connType, ident);
    }


    /**
     *  Closes the connection.
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        checkOpen();
        dropOpen(desc);
        desc = null;
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  timeout  The timeout (secs).  0 means no timeout.
     *  @throws  DriverException
     */
    public void setTimeout(double timeout) throws DriverException
    {
        checkOpen();
        desc.impl.setTimeout(timeout);
    }


    /**
     *  Writes a byte to a device without registers.
     *
     *  @param  addr   The I2C address
     *  @param  value  The byte value to write
     *  @throws  DriverException
     */
    public void write(int addr, int value) throws DriverException
    {
        checkOpen();
        desc.impl.write(addr, value);
    }


    /**
     *  Reads a byte from a device without registers.
     *
     *  @param  addr  The I2C address
     *  @return  The read value, or -1 if timeout
     *  @throws  DriverException
     */
    public int read(int addr) throws DriverException
    {
        checkOpen();
        return desc.impl.read(addr);
    }


    /**
     *  Reads multiple bytes from a device without registers.
     *
     *  @param  addr   The I2C address
     *  @param  buff   The read buffer
     *  @param  count  The number of bytes to read
     *  @return  The number of bytes read.  If less than count, a timeout occurred.
     *  @throws  DriverException
     */
    public int read(int addr, byte[] buff, int count) throws DriverException
    {
        checkOpen();
        return desc.impl.read(addr, buff, count);
    }


    /**
     *  Reads multiple bytes from a device without registers.
     *
     *  @param  addr  The I2C address
     *  @param  buff  The read buffer
     *  @return  The number of bytes read.  If less than the buffer size, a timeout occurred.
     *  @throws  DriverException
     */
    public int read(int addr, byte[] buff) throws DriverException
    {
        return read(addr, buff, buff.length);
    }


    /**
     *  Writes multiple bytes to a device with 1-byte register numbers.
     *
     *  @param  addr   The I2C address
     *  @param  reg    The register number
     *  @param  buff   The write buffer
     *  @param  count  The number of bytes to write
     *  @throws  DriverException
     */
    public void write(int addr, int reg, byte[] buff, int count) throws DriverException
    {
        checkOpen();
        desc.impl.write(addr, reg, buff, count);
    }


    /**
     *  Writes multiple bytes to a device with 1-byte register numbers.
     *
     *  @param  addr  The I2C address
     *  @param  reg   The register number
     *  @param  buff  The write buffer
     *  @throws  DriverException
     */
    public void write(int addr, int reg, byte[] buff) throws DriverException
    {
        write(addr, reg, buff, buff.length);
    }


    /**
     *  Reads multiple bytes from a device with 1-byte register numbers.
     *
     *  @param  addr   The I2C address
     *  @param  reg    The register number
     *  @param  buff   The read buffer
     *  @param  count  The number of bytes to read
     *  @return  The number of bytes read.  If less than count, a timeout occurred.
     *  @throws  DriverException
     */
    public int read(int addr, int reg, byte[] buff, int count) throws DriverException
    {
        checkOpen();
        return desc.impl.read(addr, reg, buff, count);
    }


    /**
     *  Reads multiple bytes from a device with 1-byte register numbers.
     *
     *  @param  addr  The I2C address
     *  @param  reg   The register number
     *  @param  buff  The read buffer
     *  @return  The number of bytes read.  If less than the buffer size, a timeout occurred.
     *  @throws  DriverException
     */
    public int read(int addr, int reg, byte[] buff) throws DriverException
    {
        return read(addr, reg, buff, buff.length);
    }


    /**
     *  Writes multiple bytes to a device with 2-byte register numbers.
     *
     *  @param  addr   The I2C address
     *  @param  reg    The register number
     *  @param  buff   The write buffer
     *  @param  count  The number of bytes to write
     *  @throws  DriverException
     */
    public void write2(int addr, int reg, byte[] buff, int count) throws DriverException
    {
        checkOpen();
        desc.impl.write2(addr, reg, buff, count);
    }


    /**
     *  Writes multiple bytes to a device with 2-byte register numbers.
     *
     *  @param  addr  The I2C address
     *  @param  reg   The register number
     *  @param  buff  The write buffer
     *  @throws  DriverException
     */
    public void write2(int addr, int reg, byte[] buff) throws DriverException
    {
        write2(addr, reg, buff, buff.length);
    }


    /**
     *  Reads multiple bytes from a device with 2-byte register numbers.
     *
     *  @param  addr   The I2C address
     *  @param  reg    The register number
     *  @param  buff   The read buffer
     *  @param  count  The number of bytes to read
     *  @return  The number of bytes read.  If less than count, a timeout occurred.
     *  @throws  DriverException
     */
    public int read2(int addr, int reg, byte[] buff, int count)
        throws DriverException
    {
        checkOpen();
        return desc.impl.read2(addr, reg, buff, count);
    }


    /**
     *  Reads multiple bytes from a device with 2-byte register numbers.
     *
     *  @param  addr  The I2C address
     *  @param  reg   The register number
     *  @param  buff  The read buffer
     *  @return  The number of bytes read.  If less than the buffer size, a timeout occurred.
     *  @throws  DriverException
     */
    public int read2(int addr, int reg, byte[] buff) throws DriverException
    {
        return read2(addr, reg, buff, buff.length);
    }


    /**
     *  Gets the open descriptor.
     *
     *  Multiple I2C devices may share the same physical connection.  If a
     *  connection is already open for the device, its descriptor is returned;
     *  otherwise a new connection is made.
     *
     *  @param  connType  The connection type
     *  @param  ident     The device identifier
     *  @return  The open descriptor
     *  @throws  DriverException
     */
    private static synchronized OpenDesc getOpen(ConnType connType, String ident) throws DriverException
    {
        String key = connType + ":" + ident;
        OpenDesc desc = openMap.get(key);
        if (desc == null) {
            Impl impl;
            switch (connType) {

            case FTDIUSB:
            case SERIALUSB:
                impl = new I2cImplUsb();
                break;

            default:
                throw new DriverException("Invalid connection type");
            }
            desc = new OpenDesc();
            desc.impl = impl;
            openMap.put(key, desc);
        }
        if (desc.refCount <= 0) {
            desc.impl.open(connType, ident, 0);
        }
        desc.refCount++;

        return desc;
    }


    /**
     *  Drops an open descriptor.
     *
     *  @param  desc  The open descriptor
     *  @throws  DriverException
     */
    private static synchronized void dropOpen(OpenDesc desc) throws DriverException
    {
        if (--desc.refCount <= 0) {
            desc.impl.close();
        }
    }


    /**
     *  Checks whether a connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        if (desc == null) {
            throw new DriverException("Connection not open");
        }
    }

}
