package org.lsst.ccs.drivers.i2c;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 **************************************************************************
 *
 *  Routines for communicating with an I2C device via the USB-I2C adaptor
 *
 *  @author  Owen Saxton
 *
 **************************************************************************
 */
public class I2cImplUsb implements I2c.Impl {

   /**
    *  Constants and data.
    */
    private static final int DEFAULT_BAUDRATE = 19200;
    private static final double DEFAULT_TIMEOUT = 0.5;
    private static final byte
        I2C_SGL = 0x53,
        I2C_MUL = 0x54,
        I2C_AD1 = 0x55,
        I2C_AD2 = 0x56,
        I2C_USB = 0x5a;

    private final Ascii asc = new Ascii();


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: USBFTDI or USBSERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  param     Open parameter (not used)
    *
    *  @throws  DriverException
    */
    @Override
    public void open(I2c.ConnType connType, String ident, int param)
        throws DriverException
    {
        Ascii.ConnType cType = (connType == I2c.ConnType.FTDIUSB)
                                 ? Ascii.ConnType.FTDI : Ascii.ConnType.SERIAL;
        int dchar = Ascii.makeDataCharacteristics(Ascii.DataBits.EIGHT,
                                                  Ascii.StopBits.TWO,
                                                  Ascii.Parity.NONE,
                                                  Ascii.FlowCtrl.NONE);
        try {
            asc.open(cType, ident, DEFAULT_BAUDRATE, dchar);
            asc.setTimeout(DEFAULT_TIMEOUT);
        }
        catch (DriverException e) {
            asc.closeSilent();
            throw e;
        }
    }


   /**
    *  Closes the connection.
    *
    *  @throws  DriverException
    */
    @Override
    public void close() throws DriverException
    {
        asc.close();
    }


   /**
    *  Sets the read timeout.
    *
    *  @param  timeout  The timeout (secs).  0 means no timeout.
    *
    *  @throws  DriverException
    */
    @Override
    public void setTimeout(double timeout) throws DriverException
    {
        asc.setTimeout(timeout);
    }


   /**
    *  Writes a byte to a device without registers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  value  The byte value to write
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized void write(int addr, int value) throws DriverException
    {
        byte[] cmnd = {I2C_SGL, (byte)(addr & 0xfe), (byte)value};
        asc.writeBytes(cmnd);
    }


   /**
    *  Reads a byte from a device without registers.
    *
    *  @param  addr  The I2C address
    *
    *  @return  The unsigned read value, or -1 if a timeout occurred
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized int read(int addr) throws DriverException
    {
        asc.flush();
        byte[] cmnd = {I2C_SGL, (byte)(addr | 0x01)};
        asc.writeBytes(cmnd);
        byte[] resp = new byte[1];
        return readBytes(resp, 1) > 0 ? resp[0] & 0xff : -1;
    }


   /**
    *  Reads multiple bytes from a device without registers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  buff   The read buffer
    *
    *  @param  count  The number of bytes to read
    *
    *  @return  The number of bytes read.  If less than count, a timeout
    *           occurred.
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized int read(int addr, byte[] buff, int count)
        throws DriverException
    {
        asc.flush();
        byte[] cmnd = {I2C_MUL, (byte)(addr | 0x01), (byte)count};
        asc.writeBytes(cmnd);
        return readBytes(buff, count);
    }


   /**
    *  Writes multiple bytes to a device with 1-byte register numbers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  reg    The register number
    *
    *  @param  buff   The write buffer
    *
    *  @param  count  The number of bytes to write
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized void write(int addr, int reg, byte[] buff, int count)
        throws DriverException
    {
        byte[] cmnd = new byte[count + 4];
        cmnd[0] = I2C_AD1;
        cmnd[1] = (byte)(addr & 0xfe);
        cmnd[2] = (byte)reg;
        cmnd[3] = (byte)count;
        System.arraycopy(buff, 0, cmnd, 4, count);
        asc.writeBytes(cmnd);
    }


   /**
    *  Reads multiple bytes from a device with 1-byte register numbers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  reg    The register number
    *
    *  @param  buff   The read buffer
    *
    *  @param  count  The number of bytes to read
    *
    *  @return  The number of bytes read.  If less than count, a timeout
    *           occurred.
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized int read(int addr, int reg, byte[] buff, int count)
        throws DriverException
    {
        asc.flush();
        byte[] cmnd = {I2C_AD1, (byte)(addr | 0x01), (byte)reg, (byte)count};
        asc.writeBytes(cmnd);
        return readBytes(buff, count);
    }


   /**
    *  Writes multiple bytes to a device with 2-byte register numbers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  reg    The register number
    *
    *  @param  buff   The read buffer
    *
    *  @param  count  The number of bytes to write
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized void write2(int addr, int reg, byte[] buff, int count)
        throws DriverException
    {
        byte[] cmnd = new byte[count + 5];
        cmnd[0] = I2C_AD1;
        cmnd[1] = (byte)(addr & 0xfe);
        cmnd[2] = (byte)(reg >> 8);
        cmnd[3] = (byte)reg;
        cmnd[4] = (byte)count;
        System.arraycopy(buff, 0, cmnd, 5, count);
        asc.writeBytes(cmnd);
    }


   /**
    *  Reads multiple bytes from a device with 2-byte register numbers.
    *
    *  @param  addr   The I2C address
    *
    *  @param  reg    The register number
    *
    *  @param  buff   The read buffer
    *
    *  @param  count  The number of bytes to read
    *
    *  @return  The number of bytes read.  If less than count, a timeout
    *           occurred.
    *
    *  @throws  DriverException
    */
    @Override
    public synchronized int read2(int addr, int reg, byte[] buff, int count)
        throws DriverException
    {
        asc.flush();
        byte[] cmnd = {I2C_AD2, (byte)(addr | 0x01), (byte)(reg >> 8), 
                       (byte)reg, (byte)count};
        asc.writeBytes(cmnd);
        return readBytes(buff, count);
    }


   /**
    *  Reads bytes.
    *
    *  @param  buff   The read buffer
    *
    *  @param  count  The number of bytes to read
    *
    *  @return  The number of bytes read.  If less than count, a timeout
    *           occurred.
    *
    *  @throws  DriverException
    */
    private int readBytes(byte[] buff, int count) throws DriverException
    {
        int nread = 0;
        byte[] buff1 = new byte[count];
        while (nread < count) {
            try {
                nread += asc.readBytes(buff1, nread);
            }
            catch (DriverTimeoutException e) {
            }
        }
        System.arraycopy(buff1, 0, buff, 0, nread);

        return nread;
    }

}
