package org.lsst.ccs.drivers.i2c;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *************************************************************************** 
 ** 
 **  Program to test the USB I2C device driver
 ** 
 **  @author Owen Saxton
 ** 
 *************************************************************************** 
 */
public class TestI2c {

  private final I2c i2c = new I2c();


   /**
    *************************************************************************** 
    ** 
    **  Constructor
    ** 
    *************************************************************************** 
    */
    public TestI2c()
    {
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    **  @param  connType  The type of connection: ftdi or serial
    **
    **  @param  ident     The identification: serial no. (ftdi) or port name
    **                    (serial)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     I2c.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        i2c.open(connType, ident, 0);
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes the connection.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        i2c.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes one byte.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  value  The value to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="writeone", description="Write one byte")
    public void writeOne(@Argument(name="addr", description="I2C bus address")
                         int addr,
                         @Argument(name="value", description="Value to write")
                         int value) throws DriverException
    {
        i2c.write(addr, value);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads one byte.
    ** 
    **  @param  addr  The I2C address
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readone", description="Read one byte")
    public String readOne(@Argument(name="addr", description="I2C bus address")
                          int addr) throws DriverException
    {
        return "Value = " + i2c.read(addr);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads multiple bytes.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  count  The number of bytes to read
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readmult", description="Read multiple byte")
    public String readMult(@Argument(name="addr", description="I2C bus address")
                           int addr,
                           @Argument(name="count", description="Byte count")
                           int count) throws DriverException
    {
        byte[] data = new byte[count];
        return formatData(data, i2c.read(addr, data));
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes bytes to 1-byte register number.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  reg    The register number
    **
    **  @param  value  The values to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="write", description="Write bytes (1-byte register number")
    public void write(@Argument(name="addr", description="I2C bus address")
                      int addr,
                      @Argument(name="reg", description="Register number")
                      int reg,
                      @Argument(name="value", description="Value to write")
                      int... value) throws DriverException
    {
        byte[] bvalue = new byte[value.length];
        for (int j = 0; j < bvalue.length; j++) {
            bvalue[j] = (byte)value[j];
        }
        i2c.write(addr, reg, bvalue);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads bytes from 1-byte register number.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  reg    The register number
    **
    **  @param  count  The number of bytes to read
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read", description="Read bytes (1-byte register number")
    public String read(@Argument(name="addr", description="I2C bus address")
                       int addr,
                       @Argument(name="reg", description="Register number")
                       int reg,
                       @Argument(name="count", description="Byte count")
                       int count) throws DriverException
    {
        byte[] data = new byte[count];
        return formatData(data, i2c.read(addr, reg, data));
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes bytes to 2-byte register number.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  reg    The register number
    **
    **  @param  value  The values to write
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="write2", description="Write bytes (2-byte register number")
    public void write2(@Argument(name="addr", description="I2C bus address")
                       int addr,
                       @Argument(name="reg", description="Register number")
                       int reg,
                       @Argument(name="value", description="Value to write")
                       int... value) throws DriverException
    {
        byte[] bvalue = new byte[value.length];
        for (int j = 0; j < bvalue.length; j++) {
            bvalue[j] = (byte)value[j];
        }
        i2c.write2(addr, reg, bvalue);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads bytes from 2-byte register number.
    ** 
    **  @param  addr   The I2C address
    **
    **  @param  reg    The register number
    **
    **  @param  count  The number of bytes to read
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read2", description="Read bytes (2-byte register number")
    public String read2(@Argument(name="addr", description="I2C bus address")
                        int addr,
                        @Argument(name="reg", description="Register number")
                        int reg,
                        @Argument(name="count", description="Byte count")
                        int count) throws DriverException
    {
        byte[] data = new byte[count];
        return formatData(data, i2c.read2(addr, reg, data));
    }


   /**
    *************************************************************************** 
    ** 
    **  Formats read data for display.
    ** 
    **  @param  data   The array of data
    **
    **  @param  count  The number of bytes read
    **
    **  @return  The result string
    **
    *************************************************************************** 
    */
    private static String formatData(byte[] data, int count)
    {
        if (count == 0) {
            return "No data read";
        }
        else {
            StringBuilder buff = new StringBuilder();
            buff.append(count);
            buff.append(" bytes read:");
            for (int j = 0; j < count; j++) {
                if (j % 20 == 0) {
                    buff.append("\n  ");
                }
                buff.append(String.format(" %02x", data[j]));
            }
            return buff.toString();
        }
    }

}
