package org.lsst.ccs.drivers.hameg;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the Hameg HMP Series device driver
 * 
 *  @author Owen Saxton
 */
public class TestHMP extends TestScpi {

    private final HMPSeries hmp;
    private int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


    /**
     *  Constructor
     */
    public TestHMP()
    {
        super(new HMPSeries());
        hmp = (HMPSeries)scpi;
    }


    /**
     *  Shows the number of channels.
     *
     *  @return  The result string
     */
    @Command(name="shownumchannels", description="Show the number of channels")
    public String showNumChannels()
    {
        return "No. of channels = " + hmp.getNumChannels();
    }


    /**
     *  Sets the channel.
     *
     *  @param  chan  The channel number
     */
    @Command(name="setchannel", description="Set the current channel")
    public void setChannel(@Argument(name="channel", description="Channel number")
                           int chan)
    {
        chanNum = chan;
    }


    /**
     *  Shows the set channel.
     *
     *  @return  The result string
     */
    @Command(name="showchannel", description="Show the set channel")
    public String showChannel()
    {
        return "Channel = " + chanNum;
    }


    /**
     *  Sets the voltage.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        hmp.setVoltage(value, chanNum);
    }


    /**
     *  Sets the voltage.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value,
                           @Argument(name="channel", description="Channel number")
                           int chan) throws DriverException
    {
        hmp.setVoltage(value, chan);
    }


    /**
     *  Sets the voltage to the maximum.
     * 
     *  @throws  DriverException
     */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage() throws DriverException
    {
        hmp.setVoltage(HMPSeries.MAX_VALUE, chanNum);
    }


    /**
     *  Sets the voltage to the maximum.
     * 
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        hmp.setVoltage(HMPSeries.MAX_VALUE, chan);
    }


    /**
     *  Shows the maximum voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage() throws DriverException
    {
        return "Maximum voltage = " + hmp.getMaximumVoltage(chanNum);
    }


    /**
     *  Shows the maximum voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage(@Argument(name="channel", description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum voltage = " + hmp.getMaximumVoltage(chan);
    }


    /**
     *  Shows the set voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage() throws DriverException
    {
        return "Voltage = " + hmp.getVoltage(chanNum);
    }


    /**
     *  Shows the set voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + hmp.getVoltage(chan);
    }


    /**
     *  Reads the actual voltage.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + hmp.readVoltage(chanNum);
    }


    /**
     *  Reads the actual voltage.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + hmp.readVoltage(chan);
    }


    /**
     *  Sets the voltage limit.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setvoltlimit", description="Set the  voltage limit")
    public void setVoltLimit(@Argument(name="value", description="Voltage limit to set")
                             double value) throws DriverException
    {
        hmp.setVoltageLimit(value, chanNum);
    }


    /**
     *  Sets the voltage limit.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setvoltlimit", description="Set the voltage limit")
    public void setVoltLimit(@Argument(name="value", description="Voltage limit to set")
                             double value,
                             @Argument(name="channel", description="Channel number")
                             int chan) throws DriverException
    {
        hmp.setVoltageLimit(value, chan);
    }


    /**
     *  Shows the voltage limit.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltlimit", description="Show the voltage limit")
    public String showVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + hmp.getVoltageLimit(chanNum);
    }


    /**
     *  Shows the voltage limit.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvoltlimit", description="Show the voltage limit")
    public String showVoltLimit(@Argument(name="channel", description="Channel number")
                                int chan) throws DriverException
    {
        return "Voltage limit = " + hmp.getVoltageLimit(chan);
    }


    /**
     *  Shows the voltage tripped state.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvolttrip", description="Show the voltage tripped state")
    public String showVoltTrip() throws DriverException
    {
        return "Voltage tripped = " + hmp.isVoltageTripped(chanNum);
    }


    /**
     *  Shows the voltage tripped state.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showvolttrip", description="Show the voltage tripped state")
    public String showVoltTrip(@Argument(name="channel", description="Channel number")
                               int chan) throws DriverException
    {
        return "Voltage tripped = " + hmp.isVoltageTripped(chan);
    }


    /**
     *  Clears the voltage tripped state.
     * 
     *  @throws  DriverException
     */
    @Command(name="clearvolttrip", description="Clear the voltage tripped state")
    public void clearVoltTrip() throws DriverException
    {
        hmp.clearVoltageTrip(chanNum);
    }


    /**
     *  Clears the voltage tripped state.
     * 
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Command(name="clearvolttrip", description="Clear the voltage tripped state")
    public void clearVoltTrip(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        hmp.clearVoltageTrip(chan);
    }


    /**
     *  Sets the current.
     * 
     *  @param  value  The value to set
     *  @throws  DriverException
     */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        hmp.setCurrent(value, chanNum);
    }


    /**
     *  Sets the current.
     * 
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value,
                           @Argument(name="channel", description="Channel number")
                           int chan) throws DriverException
    {
        hmp.setCurrent(value, chan);
    }


    /**
     *  Sets the current to the maximum.
     * 
     *  @throws  DriverException
     */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent() throws DriverException
    {
        hmp.setVoltage(HMPSeries.MAX_VALUE, chanNum);
    }


    /**
     *  Sets the current to the maximum.
     * 
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        hmp.setVoltage(HMPSeries.MAX_VALUE, chan);
    }


    /**
     *  Shows the maximum current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent() throws DriverException
    {
        return "Maximum current = " + hmp.getMaximumCurrent(chanNum);
    }


    /**
     *  Shows the maximum current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent(@Argument(name="channel", description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum current = " + hmp.getMaximumCurrent(chan);
    }


    /**
     *  Shows the set current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent() throws DriverException
    {
        return "Current = " + hmp.getCurrent(chanNum);
    }


    /**
     *  Shows the set current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + hmp.getCurrent(chan);
    }


    /**
     *  Reads the actual current.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + hmp.readCurrent(chanNum);
    }


    /**
     *  Reads the actual current.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent(@Argument(name="channel", description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + hmp.readCurrent(chan);
    }


    /**
     *  Turns the output on or off.
     *
     *  @param  state  The state to set: on or off
     *  @throws  DriverException
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        hmp.setOutput(state == OnOff.ON, chanNum);
    }


    /**
     *  Turns the output on or off.
     * 
     *  @param  state  The state to set: on or off
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off")
                          OnOff state,
                          @Argument(name="channel", description="Channel number")
                          int chan) throws DriverException
    {
        hmp.setOutput(state == OnOff.ON, chan);
    }


    /**
     *  Shows the output state.
     * 
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(hmp.getOutput(chanNum));
    }


    /**
     *  Shows the output state.
     * 
     *  @param  chan  The channel number
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput(@Argument(name="channel", description="Channel number")
                             int chan) throws DriverException
    {
        return "Output state = " + getOnOff(hmp.getOutput(chan));
    }


    /**
     *  Locks the front panel.
     * 
     *  @throws  DriverException
     */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        hmp.lockPanel(true);
    }


    /**
     *  Unlocks the front panel.
     * 
     *  @throws  DriverException
     */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        hmp.lockPanel(false);
    }


    /**
     *  Sounds the beeper.
     *
     *  @throws  DriverException
     */
    @Command(name="beep", description="Sound the beeper")
    public void beep() throws DriverException
    {
        hmp.beep();
    }


    /**
     *  Converts a boolean to on/off.
     */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
