package org.lsst.ccs.drivers.hameg;

import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *****************************************************************************
 *
 *  Routines for controlling a Hameg HMP series power supply.
 *
 *  All four models (HMP2020, HMP2030, HMP4030 and HMP4040) are supported via
 *  the serial interface.  The data characteristics of the line must be set
 *  to the power supply defaults of 8 bit, no parity, 1 stop bit and RTS/CTS
 *  handshaking.  The default baud rate is 9600.
 *
 *  @author  Owen Saxton
 *
 *****************************************************************************
 */
public class HMPSeries extends Scpi implements PowerSupplyDriver {

    public static final int
        DEFAULT_BAUDRATE = 9600,
        MIN_CHANNEL      = 1,
        MAX_CHANNEL      = 4,
        NUM_CHANNELS     = MAX_CHANNEL - MIN_CHANNEL + 1;
    public static final double MAX_VALUE = Double.POSITIVE_INFINITY;

    private final Timer timer = new Timer("Timer_HMP", true);
    private final long[]
        onDelay = new long[NUM_CHANNELS],
        offDelay = new long[NUM_CHANNELS];
    private final SetOutput[] outTask = new SetOutput[NUM_CHANNELS];
    private int maxChannel, numChannels;


   /**
    *  Timer task to implement the delayed setting of the output state.
    */
    class SetOutput extends TimerTask {

        private final int chan;
        private final String command;

        SetOutput(String command, int chan)
        {
            this.command = command;
            this.chan = chan;
        }
    
        @Override
        public void run()
        {
            try {
                writeChan(command, chan);
            }
            catch (DriverException e) {
            }
            outTask[chan - MIN_CHANNEL] = null;
        }
    }


   /**
    *  Constructor.
    */
    public HMPSeries() {
        super(Option.NO_NET);   // Disallow network connections
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: CONN_TYPE_FTDI or
    *                    CONN_TYPE_SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate
    *
    *  @param  dataChar  The data characteristics
    *
    *  @throws  DriverException
    */
    @Override
    public void open(Scpi.ConnType connType, String ident, int baudRate,
                     int dataChar) throws DriverException
    {
        super.open(connType, ident, baudRate, dataChar);
        try {
            checkIdentification("HAMEG", CHECK_EQUALS, "HMP", CHECK_STARTS_WITH);
            String model = getIdentification()[1];
            try {
                numChannels = Integer.valueOf(model.substring(5, 6));
            }
            catch (NumberFormatException e) {
                throw new DriverException("Unrecognized model: " + model);
            }
            if (numChannels > NUM_CHANNELS) {
                throw new DriverException("Unsupported model: " + model);
            }
            maxChannel = MIN_CHANNEL + numChannels - 1;
        }
        catch (DriverException | NumberFormatException e) {
            close();
            maxChannel = MIN_CHANNEL - 1;
            numChannels = 0;
            throw e;
        }
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: CONN_TYPE_FTDI or
    *                    CONN_TYPE_SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate
    *
    *  @throws  DriverException
    */
    @Override
    public void open(Scpi.ConnType connType, String ident, int baudRate)
        throws DriverException
    {
        int dataChar = makeDataCharacteristics(DataBits.EIGHT, StopBits.ONE,
                                               Parity.NONE, FlowCtrl.RTS);
        open(connType, ident, baudRate, dataChar);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: CONN_TYPE_FTDI or
    *                    CONN_TYPE_SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @throws  DriverException
    */
    public void open(Scpi.ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The connection type: CONN_TYPE_FTDI or
    *                    CONN_TYPE_SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @throws  DriverException
    */
    @Deprecated
    public void open(int connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE);
    }


   /**
    *  Gets the number of channels.
    *
    *  @return  The number of channels
    */
    public int getNumChannels()
    {
        return numChannels;
    }


   /**
    *  Turns output on or off for a channel.
    *
    *  @param  on    Turn on if true, off if false
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        checkChannel(chan);
        int index = chan - MIN_CHANNEL;
        long delay = on ? onDelay[index] : offDelay[index];
        String command = "OUTP " + (on ? "1" : "0");
        SetOutput task = outTask[index];
        if (task != null) {
            task.cancel();
            outTask[index] = null;
        }
        if (delay == 0) {
            writeChan(command, chan);
        }
        else {
            outTask[index] = new SetOutput(command, chan);
            timer.schedule(outTask[index], delay);
        }
    }


   /**
    *  Gets the output state of a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The output state
    *
    *  @throws  DriverException
    */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return readIntegerChan("OUTP?", chan) != 0;
    }


   /**
    *  Sets the power-on delay for a channel.
    *
    *  @param  time  The delay (sec)
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        onDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


   /**
    *  Sets the power-off delay for a channel.
    *
    *  @param  time  The delay (sec)
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        offDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


   /**
    *  Sets the voltage for a channel.
    *
    *  @param  value  The value to set.  The maximum value may be set by
    *                 using the constant MAX_VALUE.
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeChan("VOLT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


   /**
    *  Gets the set voltage for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT?", chan);
    }


   /**
    *  Gets the maximum voltage for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The maximum voltage
    *
    *  @throws  DriverException
    */
    public double getMaximumVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT? MAX", chan);
    }


   /**
    *  Reads the voltage for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:VOLT?", chan);
    }


   /**
    *  Sets the current for a channel.
    *
    *  @param  value  The value to set.  The maximum value may be set by
    *                 using the constant MAX_VALUE.
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeChan("CURR " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


   /**
    *  Gets the set current for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The current
    *
    *  @throws  DriverException
    */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR?", chan);
    }


   /**
    *  Gets the maximum current for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The maximum current
    *
    *  @throws  DriverException
    */
    public double getMaximumCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR? MAX", chan);
    }


   /**
    *  Reads the current for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The current
    *
    *  @throws  DriverException
    */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:CURR?", chan);
    }


   /**
    *  Sets the voltage limit for a channel.
    *
    *  @param  value  The value to set
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    public void setVoltageLimit(double value, int chan) throws DriverException
    {
        writeChan("VOLT:PROT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


   /**
    *  Gets the voltage limit for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDoubleChan("VOLT:PROT?", chan);
    }


   /**
    *  Gets the voltage tripped state for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The tripped state: true or false
    *
    *  @throws  DriverException
    */
    public boolean isVoltageTripped(int chan) throws DriverException
    {
        return readIntegerChan("VOLT:PROT:TRIP?", chan) != 0;
    }


   /**
    *  Clears the voltage tripped state for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    public void clearVoltageTrip(int chan) throws DriverException
    {
        writeChan("VOLT:PROT:CLEAR", chan);
    }


   /**
    *  Locks or unlocks the front panel.
    *
    *  @param  lock  True to lock the panel; false to unlock it
    *
    *  @throws  DriverException
    */
    public void lockPanel(boolean lock) throws DriverException
    {
        writeCommand(lock ? "SYST:RWL" : "SYST:LOC");
    }


   /**
    *  Activates the beeper.
    *
    *  @throws  DriverException
    */
    public void beep() throws DriverException
    {
        writeCommand("SYST:BEEP");
    }


   /**
    *  Checks a channel for validity.
    */
    private void checkChannel(int chan) throws DriverException
    {
        if (chan < MIN_CHANNEL || chan > maxChannel) {
            throw new DriverException("Invalid channel number");
        }
    }


   /**
    *  Writes a channel select command.
    */
    private void writeSelect(int chan) throws DriverException
    {
        checkChannel(chan);
        writeCommand("INST:NSEL " + chan);
    }


   /**
    *  Writes a command to a channel.
    */
    private synchronized void writeChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        writeCommand(instr);
    }


   /**
    *  Writes a command to a channel and reads the double result.
    */
    private synchronized double readDoubleChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        return readDouble(instr);
    }


   /**
    *  Writes a command to a channel and reads the integer result.
    */
    private synchronized int readIntegerChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        return readInteger(instr);
    }

}
