package org.lsst.ccs.drivers.gpvacmon;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;

import java.text.DecimalFormat;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 **************************************************************************
 **
 ** General access routines for the GP835 device
 *
 ** @author Homer Neal
 * **************************************************************************
 */
public class GP835 extends Scpi {

    /**
     ** Public constants *
     *
     */
    public enum onOff {

        OFF, ON;
    }

    final boolean DEBUG = true;

    /**
     ** Private constants *
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private String terminator = "\r";
    private int timeout = 1000;

    /**
     * As specified in the manual pg 13, Current limits *
     */
    private double minCurrent = 3.00;
    private double maxCurrent = 24.00;
    /**
     * As specified in the manual pge 13, Power limits *
     */
    private int minPower = 40;
    private int maxPower = 300;

    public GP835() throws DriverException {
    }

    /**
     **
     ** Opens a connection.
     *
     * *
     ** @param serialname The device name of the serial port
     ** @param baud The baud rate
     *
     */
    public void open(String serialname, int baud) throws DriverException {
        init();
        System.out.println("opening SERIAL connection to the GP Vacuum Quality Monitor");
        System.out.println("serialname = "+serialname);
        System.out.println("baud = "+baud);
        super.open(Scpi.CONN_TYPE_SERIAL, serialname, baud);
    }

    /**
     ** Opens an FTDI connection.
     */
    public void openftdi(String serialname, int baud) throws DriverException {
        init();
        System.out.println("opening FTDI connection to the GP Vacuum Quality Monitor");
        open(Scpi.CONN_TYPE_FTDI, serialname, baud);

    }

    /**
     ** Initializes device.
     *
     */
    public void init() {
        setTerminator(terminator);
    }

    /**
     ** Closes the connection. * * @throws DriverException *
     *
     */
    public void close() throws DriverException {
        super.close();
    }

    /**
     * reset to factory defaults *
     */
    public void reset() throws DriverException {
        writeGP835("RST");
    }

    /**
     * Return the full Status byte *
     */
    public String getStatus() throws DriverException {
        return readGP835("STB?");
    }

    /**
     * Strips off "STB" if present and converts hex code to int *
     */
    public int stripStatus(String status) throws DriverException {
        if (status.contains("STB")) {
            return (Integer.parseInt(status.substring(3), 16));
        } else {
            return (Integer.parseInt(status, 16));
        }
    }

    /**
     * Return the status response as an int
     */
    public int getState() throws DriverException {
        return stripStatus(getStatus());
    }

    /**
     * Return the error register *
     */
    public int stripError(String err) throws DriverException {
        if (err.contains("ESR")) {
            return (Integer.parseInt(err.substring(3), 16));
        } else {
            return (Integer.parseInt(err, 16));
        }
    }

    public String getError() throws DriverException {
        return (readGP835("ESR?"));
    }

    /**
     * Gets the GP835 identity. *
     */
    public String getIdent() throws DriverException {
        return readGP835("*IDN?");
    }

    /**
     * Gets the GP835 report. *
     */
    public String getReport() throws DriverException {
        return readGP835("TEST:REP?");
    }

    /**
     ** return the pressure
     *
     * @throws DriverException * 9.5.6 External Total Pressure To get the total
     * pressure from an external instrument: INST ETPR OUTP ON MEAS:PRES?
     */
    public double getPressure() throws DriverException {
//        return Double.valueOf(readGP835("INST ETPR;OUTP ON;MEAS:PRES?"));
	writeGP835("INST ETPR");
//	writeGP835("OUTP ON");
        return Double.valueOf(readGP835("MEAS:PRES?"));
    }

    /**
     ** return the power to the filament
     *
     */
    public double getFilamentPower() throws DriverException {
        return Double.valueOf(readGP835("INST FIL;MEAS:POW?"));
    }

    /**
     ** return the mass spectrometer scan results
     *
     */
    public String getMSP() throws DriverException {
        return readGP835("INST MSP;FORM:AMU ON;FORM:COUNT ON;OUTP ON ;INIT;FETC?");
    }

    /**
     ** return the test report
     *
     */
    public String getTestReport() throws DriverException {
        return readGP835("TEST:REP?");
    }

    /**
     ** return the currently selected instrument
     *
     */
    public String getSelectedInstrument() throws DriverException {
        return readGP835("INST:SEL?");
    }

    /**
     ** Sets the gauge on
     *
     * @throws DriverException *
     */
    public void setGaugeOn() throws DriverException {
//        writeGP835("INST MSP;OUTP ON");
       	writeGP835("INST ETPR");
	writeGP835("OUTP ON");
    }

    /**
     ** Sets the gauge off --- usually to avoid light
     *
     * @throws DriverException *
     */
    public void setGaugeOff() throws DriverException {
//        writeGP835("INST MSP;OUTP OFF");
       	writeGP835("INST ETPR");
	writeGP835("OUTP OFF");
       	writeGP835("INST MSP");
	writeGP835("OUTP OFF");
    }

     /**
     ** Writes a command.
     *
     * @param command The command to write, excluding terminator
     * @throws DriverException *
     *
     */
    public synchronized void writeGP835(String command) throws DriverException {
        // write(command + terminator);
        write(command);
    }

    /**
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     *
     ** @throws DriverTimeoutException *
     */
    public synchronized String readGP835() throws DriverException {
        return (read());
    }

    /**
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     *
     ** @throws DriverTimeoutException *
     */
    public synchronized String readGP835(String command) throws DriverException {
//        System.out.println("readGP835 called with command ("+command+")");
        writeGP835(command);
        String response = readGP835();
//        System.out.println("response = "+response);
        return response;
    }

    /**
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     *
     ** @throws DriverTimeoutException *
     */
    public synchronized int readIntegerGP835(String command) throws DriverException {
        writeGP835(command);
        return Integer.getInteger(readGP835());
    }

    public double[][] readAMU() throws DriverException {
        double[][] amuarray = new double[5000][2];
        String buff;
        buff = getMSP();

        int blen = buff.length();

        int ichar = 0;

        int icrv = buff.indexOf(" CURVE ");
        if (icrv > 0) {
            int istart = buff.indexOf(" VALues ");
            if (istart > 0) {
                istart += 8;
            }
            ichar = istart;
            ByteBuffer curv;
            curv = ByteBuffer.wrap(buff.substring(istart).getBytes());
            curv.order(ByteOrder.LITTLE_ENDIAN);
            ShortBuffer mspdat = curv.asShortBuffer();

            double val;
            int idx = 0;
            
            while ((ichar + 1) < blen) {
                char thischar = buff.charAt(ichar);
                char nextchar = buff.charAt(ichar + 1);
                if (thischar != '\n' && nextchar != '\n') {
                    val = mspdat.get();
                    amuarray[idx][1] = val;
                    idx++;
                    ichar += 2;
                } else {
                    break;
                }
                System.out.println("ADC count = " + val);

            }
        
        }
        return(amuarray);
    }
    /*
     public double[][] readAMU() throws DriverException {
     readIntegerGP835("*OPC?");  // any other operations to complete
     writeGP835("FORM:ALL ON,ON");
     writeGP835("INIT");       // get measurements

     int krdy = 0;
     System.out.println("Beginning of readBuffer: krdy = " + krdy);
     while (krdy != 1) { // this should have been set false by accumbuffer
     System.out.println("In loop of readBuffer: krdy = " + krdy);
     try {
     krdy = readIntegerGP835("*OPC?");  // wait for read operations to complete
     } catch (Exception ex) {
     krdy = 0;                         // obviously we are not ready yet
     }
     System.out.println("krdy = " + krdy);
     if (krdy != 1) {
     System.out.println("sleep for a second then recheck.");
     try {
     Thread.currentThread().sleep(1000);//sleep for 1000 ms
     } catch (InterruptedException ex) {
     System.out.println("Cannot sleep because of exception: " + ex);
     }
     }
     // in case conditions are never satisfied, count on the timeout to free us
     }
     if (DEBUG) {
     System.out.println("About to do double pair array read.");
     }
     double[][] buff = readDoublePairArray("FETCH?"); // request all stored readings
     if (DEBUG) {
     System.out.println("Fetched data.");
     }
     return buff;
     }
     */

    /**
     **************************************************************************
     **
     ** Reads a double pair array with SCPI error checking after writing a
     * command. * * @param command The command to write, excluding terminator
     *
     *
     * * @return The returned array of double float values * * @throws
     * DriverException
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public double[][] readDoublePairArray(String command) throws DriverException {
        try {
            if (DEBUG) {
                System.out.println("Doing String Array read.");
            }
            String[] reply = readStringArray(command);
            if (DEBUG) {
                /*
                 for (String str : reply) {
                 System.out.println("reply string = " + str);
                 }
                 */
                System.out.println("Retrieved string of values from buffer. length = " + reply.length);
            }
            double[][] dReply = new double[2][reply.length / 2];
            int ridx = 0;
            for (int j = 0; j < reply.length; j += 2) {
//                System.out.println("reply = (" + reply[j] + ")");
                dReply[0][ridx] = Double.valueOf(reply[j]);
                dReply[1][ridx] = Double.valueOf(reply[j + 1]);
                ridx++;
            }
            if (DEBUG) {
                System.out.println("Binary array prepared.");
            }
            return dReply;
        } catch (NumberFormatException e) {
            throw new DriverException(e);
        }

    }

}
