package org.lsst.ccs.drivers.gpio;

import java.util.List;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A general purpose GPIO driver. This driver has been tested on UNO 1483 and Raspberry Pi 4, but should work
 * on other Linux systems with minimal changes.
 *
 * Originally written for the older <code>/sys/class</code> GPIO kernel interface, it was subsequently ported
 * to be an abstract class with concrete implementations running over both the popular <code>libgpiod</code>
 * v1 library and the <code>advec</code> vendor kernel module specific to the Advantech UNO 1483.
 *
 * @author tonyj
 * @author fritzm
 */
public abstract class GPIODriver {

    /**
     * Enumerate the GPIO chips on this machine
     *
     * @return The list of GPIO chips
     * @throws DriverException If an IO error occurs while enumerating the chips
     */
    public abstract List<? extends GPIOChip> enumerateChips() throws DriverException;

    /**
     * Enumerate the currently exported GPIO channels on this machine
     *
     * @return The list of exported channels
     * @throws DriverException If the channels cannot be enumerated.
     */
    public abstract List<? extends GPIOChannel> enumerateExportedChannels() throws DriverException;

    /**
     * Get a specified GPIO channel
     *
     * @param channel The channel to fetch
     * @return The Channel
     * @throws DriverException If the GPIO can not be fetched, for example if
     * the channel number is out of range.
     */
    public abstract GPIOChannel getChannel(int channel) throws DriverException;

    /**
     * Test if the given channel is already exported
     *
     * @param channel The channel to test
     * @return <code>true</code> if and only if the channel has already been
     * exported.
     */
    public abstract boolean isExported(int channel);

    /**
     * Export the specified channel.
     *
     * @param channel The channel number
     * @return The exported channel
     * @throws DriverException If the channel is invalid, or cannot be exported
     */
    public abstract GPIOChannel export(int channel) throws DriverException;

    /**
     * Export all of the channels in the given range. Channels which are already
     * exported are not re-exported.
     *
     * @param base The first channel to export
     * @param ngpio The number of channels to export
     * @return The list of exported channels. including any which were already
     * exported.
     * @throws DriverException If an error occurs.
     */
    public abstract List<? extends GPIOChannel> exportRange(int base, int ngpio) throws DriverException;

    /**
     * Unexport the specified channel
     *
     * @param channel The channel number
     * @throws DriverException If the channel is invalid, or cannot be un
     * exported
     */
    public abstract void unexport(int channel) throws DriverException;

    /**
     * Unexport all of the channels in the given range. Channels which are not
     * already exported are ignored.
     *
     * @param base The first channel to unexport
     * @param ngpio The number of channels to unexport
     * @throws DriverException If an error occurs.
     */
    public abstract void unexportRange(int base, int ngpio) throws DriverException;

    /**
     * A class representing a single GPIO Channel
     */
    public abstract static class GPIOChannel {

        protected final int chip;
        protected final int channel;
        protected final String name;
        protected final int offset;

        /**
         * Enumeration for specifying the direction of a channel
         */
        public enum Direction {

            /**
             * A readable channel
             */
            IN,

            /**
             * A writable channel
             */
            OUT
        };

        public GPIOChannel(int chip, int channel, String name, int offset) {
            this.chip = chip;
            this.channel = channel;
            this.name = name;
            this.offset = offset;
        }

        /**
         * Get the direction for this channel
         *
         * @return The direction
         * @throws DriverException If the direction cannot be determined
         */
        public abstract Direction getDirection() throws DriverException;

        /**
         * Set the direction for this channel
         *
         * @param dir The direction to be set.
         * @throws DriverException If the direction cannot be set.
         */
        public abstract void setDirection(Direction dir) throws DriverException;

        /**
         * Set the channel (to 1)
         *
         * @throws DriverException If the channel cannot be set
         */
        public abstract void set() throws DriverException;

        /**
         * Clear the channel (to 0)
         *
         * @throws DriverException
         */
        public abstract void clear() throws DriverException;

        /**
         * Read the current value of the channel
         *
         * @return <code>true</code> if the channel is set (value == 1)
         * @throws DriverException If an error occurs
         */
        public abstract boolean read() throws DriverException;

        /**
         * Write a new value to this channel
         *
         * @param value The value to write
         * @throws DriverException If an error occurs
         */
        public abstract void write(boolean value) throws DriverException;

        /**
         * Locks this channel for exclusive access. This method works by taking
         * an exclusive lock on the value file, which speeds up all subsequent
         * read/write operations on the channel.
         *
         * @throws DriverException If the lock cannot be obtained.
         */
        public abstract void lock() throws DriverException;

        /**
         * Unlocks the specified channel. If the channel is not locked the
         * method does nothing.
         *
         * @throws DriverException If the channel unlock fails
         */
        public abstract void unlock() throws DriverException;

        /**
         * Return the chip number for this channel.
         *
         * @return The chip number
         */
        public int getChip() {
            return this.chip;
        }

        /**
         * Return the channel number for this channel.
         *
         * @return The channel number
         */
        public int getChannel() {
            return this.channel;
        }

        /**
         * Return the human readable name for this channel.
         *
         * @return The channel name
         */
        public String getName() {
            return this.name;
        }

        /**
         * Return the offset within chip for this channel.
         *
         * @return The channel offset
         */
        public int getOffset() {
            return this.offset;
        }

        @Override
        public String toString() {
            return "GPIOChannel{" + "channel=" + channel + ", name=" + name + ", chip=" + chip + ", offset=" + offset + "}";
        }
    }

    /**
     * A class representing a GPIOChip
     */
    public class GPIOChip {

        private final int base;
        private final int ngpio;
        private final String name;
        private final String label;

        protected GPIOChip(int base, int ngpio, String name, String label) throws DriverException {
            this.base = base;
            this.ngpio = ngpio;
            this.name = name;
            this.label = label;
        }

        /**
         * Get the base channel number for this chip
         *
         * @return The base number
         */
        public int getBase() {
            return this.base;
        }

        /**
         * Get the number of GPIO channels supported by this chip
         *
         * @return The number of channels
         */
        public int getNgpio() {
            return this.ngpio;
        }

        /**
         * Get the human readable name associated with this chip
         * @return The name
         */
        public String getName() {
            return this.name;
        }

        /**
         * Get the human readable label associated with this chip
         * @return The label
         */
        public String getLabel() {
            return this.label;
        }

        /**
         * Export all of the channels associated with this chip. Channels which
         * are already exported are not re-exported (to avoid errors).
         *
         * @return The list of channels exported, including any which were
         * already exported.
         * @throws DriverException If an error occurs
         */
        public List<? extends GPIOChannel> exportAll() throws DriverException {
            return exportRange(base, ngpio);
        }

        /**
         * Unexport all of the channels associated with this chip. Channels which
         * are not exported are ignored.
         *
         * @throws DriverException If an error occurs
         */
        public void unexportAll() throws DriverException {
            unexportRange(base, ngpio);
        }

        @Override
        public String toString() {
            return "GPIOChip{" + "base=" + base + ", ngpio=" + ngpio + ", name=" + name + ", label=" + label + "}";
        }
    }

}
