package org.lsst.ccs.drivers.gpio;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.lsst.ccs.drivers.commons.DriverException;

import com.sun.jna.LastErrorException;
import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.PointerType;
import com.sun.jna.Structure;

/**
 * Concrete class implementing <code>GPIODriver</code> leveraging a JNA wrapper for the <code>libgpiod</code>
 * v1 library. This should work for any harware with drivers which support the official new character device
 * GPIO kernel interface. A notable exception is the Avantech UNO 1484 which does _not_ support the newer
 * kernel interface, and which therefore has its own specific concrete class also in this package.)
 *
 * @see <a href="https://docs.kernel.org/userspace-api/gpio/chardev.html">GPIO Character Device Userspace
 * API</a>
 */
public class GPIODriverLibgpiod extends GPIODriver {

    private final List<GPIOChipLibgpiod> GPIOChipsByIndex = new ArrayList<>();
    private final List<GPIOChannelLibgpiod> GPIOChannelsByIndex = new ArrayList<>();
    private final Set<Integer> GPIOExportedChannels = new HashSet<>();

    /**
     *  JNA wrapper for libgpiod v1
     */
    private interface gpiod extends Library {

        public final gpiod INSTANCE = (gpiod)Native.load("gpiod", gpiod.class);

        public static final int LINE_DIRECTION_INPUT = 1;
        public static final int LINE_DIRECTION_OUTPUT = 2;

        public static final int GPIOD_LINE_REQUEST_DIRECTION_AS_IS = 1;
        public static final int GPIOD_LINE_REQUEST_DIRECTION_INPUT = 2;
        public static final int GPIOD_LINE_REQUEST_DIRECTION_OUTPUT = 3;

        public final class gpiod_chip_ptr extends PointerType {}
        public final class gpiod_chip_iter_ptr extends PointerType {}
        public final class gpiod_line_ptr extends PointerType {}
        public final class gpiod_line_iter_ptr extends PointerType {}

        public class gpiod_line_request_config extends Structure {

            @SuppressWarnings("unused")
            public String consumer;

            @SuppressWarnings("unused")
            public int request_type;

            @SuppressWarnings("unused")
            public int flags;

            @Override
            protected List<String> getFieldOrder() {
                return Arrays.asList("consumer", "request_type", "flags");
            }

        }

        public static class gpiod_line_request_config_ptr
            extends gpiod_line_request_config implements Structure.ByReference {}

        public gpiod_chip_iter_ptr gpiod_chip_iter_new() throws LastErrorException;
        public gpiod_chip_ptr gpiod_chip_iter_next(gpiod_chip_iter_ptr iter);
        public void gpiod_chip_iter_free(gpiod_chip_iter_ptr iter);

        public gpiod_chip_ptr gpiod_chip_open_by_number(int chipNum);
        public String gpiod_chip_name(gpiod_chip_ptr chip);
        public String gpiod_chip_label(gpiod_chip_ptr chip);
        public int gpiod_chip_num_lines(gpiod_chip_ptr chip);
        public gpiod_line_ptr gpiod_chip_get_line(gpiod_chip_ptr chip, int offset);

        public gpiod_line_iter_ptr gpiod_line_iter_new(gpiod_chip_ptr chip) throws LastErrorException;
        public gpiod_line_ptr gpiod_line_iter_next(gpiod_line_iter_ptr iter);
        public void gpiod_line_iter_free(gpiod_line_iter_ptr iter);

        public String gpiod_line_name(gpiod_line_ptr line);
        public int gpiod_line_offset(gpiod_line_ptr line);
        public int gpiod_line_direction(gpiod_line_ptr line);
        public int gpiod_line_request(gpiod_line_ptr line, gpiod_line_request_config_ptr config, int defaultVal) throws LastErrorException;
        public void gpiod_line_release(gpiod_line_ptr line);
        public int gpiod_line_get_value(gpiod_line_ptr line);
        public void gpiod_line_set_value(gpiod_line_ptr line, int value);
        public void gpiod_line_close_chip(gpiod_line_ptr line);

    }

     public GPIODriverLibgpiod() throws DriverException {
        super();
        // We enumerate and build object wrappers for all chips/channels at driver construction (though we
        // defer holding on to any open descriptors until "export").  This is so we can establish a proper
        // mapping between flat "channel number" (in the old sys/class sense) and chip/offset as used by the
        // new chardev interface.
        int chipNum = 0;
        int channelNum = 0;
        gpiod.gpiod_chip_iter_ptr chipIter = gpiod.INSTANCE.gpiod_chip_iter_new();
        gpiod.gpiod_chip_ptr chip;
        while((chip = gpiod.INSTANCE.gpiod_chip_iter_next(chipIter)) != null) {
            this.GPIOChipsByIndex.add(new GPIOChipLibgpiod(chip, channelNum));
            gpiod.gpiod_line_iter_ptr lineIter = gpiod.INSTANCE.gpiod_line_iter_new(chip);
            gpiod.gpiod_line_ptr line;
            while((line = gpiod.INSTANCE.gpiod_line_iter_next(lineIter)) != null) {
                this.GPIOChannelsByIndex.add(new GPIOChannelLibgpiod(line, chipNum, channelNum));
                ++channelNum;
            }
            ++chipNum;
            gpiod.INSTANCE.gpiod_line_iter_free(lineIter);
        }
        gpiod.INSTANCE.gpiod_chip_iter_free(chipIter);
    }

    public List<? extends GPIOChip> enumerateChips() throws DriverException {
        return this.GPIOChipsByIndex;
    }

    public List<? extends GPIOChannel> enumerateExportedChannels() throws DriverException {
        return GPIOExportedChannels.stream().map(GPIOChannelsByIndex::get).collect(Collectors.toList());
    }

    public GPIOChannelLibgpiod getChannel(int channel) throws DriverException {
        if ((channel < 0) || (channel >= GPIOChannelsByIndex.size())) {
            throw new DriverException("Invalid channel " + channel);
        } else {
            return this.GPIOChannelsByIndex.get(channel);
        }
    }

    public boolean isExported(int channel) {
        return GPIOExportedChannels.contains(channel);
    }

    public GPIOChannelLibgpiod export(int channel) throws DriverException {
        GPIOChannelLibgpiod chan = getChannel(channel);
        chan.export();
        GPIOExportedChannels.add(channel);
        return chan;
    }

    public List<? extends GPIOChannel> exportRange(int base, int ngpio) throws DriverException {
        List<GPIOChannelLibgpiod> result = new ArrayList<>();
        for (int channel = base; channel < base + ngpio; channel++) {
            if (!isExported(channel)) {
                result.add(export(channel));
            } else {
                result.add(getChannel(channel));
            }
        }
        return result;
    }

    public void unexport(int channel) throws DriverException {
        getChannel(channel).unexport();
        GPIOExportedChannels.remove(channel);
    }

    public void unexportRange(int base, int ngpio) throws DriverException {
        for (int channel = base; channel < base + ngpio; channel++) {
            if (isExported(channel)) {
                unexport(channel);
            }
        }
    }

    public static final class GPIOChannelLibgpiod extends GPIOChannel {

        private gpiod.gpiod_line_ptr line = null;

        private GPIOChannelLibgpiod(gpiod.gpiod_line_ptr line, int chip, int channel) {
            super(
                chip,
                channel,
                gpiod.INSTANCE.gpiod_line_name(line),
                gpiod.INSTANCE.gpiod_line_offset(line)
            );

        }

        private void assertExported() throws DriverException {
            if (this.line == null) {
                throw new DriverException("Invalid operation on unexported channel");
            }
        }

        private void export() throws DriverException {
            // We repurpose "export" here to open and cache a gpiod_line_ptr (and its implicit gpiod_chip_ptr)
            // and request the line as required by libgpiod.
            if (this.line == null) {
                gpiod.gpiod_chip_ptr chipPtr = gpiod.INSTANCE.gpiod_chip_open_by_number(this.chip);
                this.line = gpiod.INSTANCE.gpiod_chip_get_line(chipPtr, this.offset);
                gpiod.gpiod_line_request_config_ptr config = new gpiod.gpiod_line_request_config_ptr();
                config.consumer = gpiod.INSTANCE.toString();
                config.request_type = gpiod.GPIOD_LINE_REQUEST_DIRECTION_AS_IS;
                config.flags = 0;
                gpiod.INSTANCE.gpiod_line_request(this.line, config, 0);
            }
        }

        private void unexport() throws DriverException {
            if (this.line != null) {
                gpiod.INSTANCE.gpiod_line_release(this.line);
                gpiod.INSTANCE.gpiod_line_close_chip(this.line);
                this.line = null;
            }
        }

        public Direction getDirection() throws DriverException {
            assertExported();
            int dir = gpiod.INSTANCE.gpiod_line_direction(this.line);
            switch(dir) {
                case gpiod.LINE_DIRECTION_INPUT: return Direction.IN;
                case gpiod.LINE_DIRECTION_OUTPUT: return Direction.OUT;
                default: throw new DriverException("Invalid direction " + dir);
            }
        }

        public void setDirection(Direction dir) throws DriverException {
            assertExported();
            // libgpiod requires a release/re-request to modify line direction at the current time
            gpiod.INSTANCE.gpiod_line_release(this.line);
            gpiod.gpiod_line_request_config_ptr config = new gpiod.gpiod_line_request_config_ptr();
            config.consumer = gpiod.INSTANCE.toString();
            config.flags = 0;
            switch(dir) {
                case IN:
                    config.request_type = gpiod.GPIOD_LINE_REQUEST_DIRECTION_INPUT;
                    gpiod.INSTANCE.gpiod_line_request(this.line, config, 0);
                    break;
                case OUT:
                    config.request_type = gpiod.GPIOD_LINE_REQUEST_DIRECTION_OUTPUT;
                    gpiod.INSTANCE.gpiod_line_request(this.line, config, 0);
                    break;
            }
        }

        public void set() throws DriverException {
            assertExported();
            gpiod.INSTANCE.gpiod_line_set_value(this.line, 1);
        }

        public void clear() throws DriverException {
            assertExported();
            gpiod.INSTANCE.gpiod_line_set_value(this.line, 0);
        }

        public boolean read() throws DriverException {
            assertExported();
            return (gpiod.INSTANCE.gpiod_line_get_value(this.line) == 1);
        }

        public void write(boolean value) throws DriverException {
            assertExported();
            gpiod.INSTANCE.gpiod_line_set_value(this.line, value ? 1 : 0);
        }

        public void lock() throws DriverException {
        }

        public void unlock() throws DriverException {
        }

    }


    public final class GPIOChipLibgpiod extends GPIOChip {

        private GPIOChipLibgpiod(gpiod.gpiod_chip_ptr chip, int base) throws DriverException {
            super(
                base,
                gpiod.INSTANCE.gpiod_chip_num_lines(chip),
                gpiod.INSTANCE.gpiod_chip_name(chip),
                gpiod.INSTANCE.gpiod_chip_label(chip)
            );
        }

    }

}
