package org.lsst.ccs.drivers.gpio;

import java.io.FileDescriptor;
import java.io.RandomAccessFile;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.lsst.ccs.drivers.commons.DriverException;

import com.sun.jna.LastErrorException;
import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.Structure;

/**
 * Concrete class implementing <code>GPIODriver</code> for the Advantech UNO-1483.  Leverages the
 * <code>advec</code>vendor kernel module via JNA-wrapped <code>ioctl()</code> calls.
 */
public class GPIODriverUno1483 extends GPIODriver {

    private final List<GPIOChip> GPIOChipsByIndex = new ArrayList<>();
    private final List<GPIOChannelUno1483> GPIOChannelsByIndex = new ArrayList<>();

    private RandomAccessFile advgpio;
    private int fd;

    /** JNA wrapper for libc (used to get ioctl()) */
    private interface libc extends Library {
        libc INSTANCE = Native.load("c", libc.class);
        int ioctl(int fd, int request, Pointer data) throws LastErrorException;
    }

    /** JNA wrapper for two-word structure used as data arg in Advantech GPIO driver ioctl() calls */
    public static class GpioIoctlData extends Structure {

        public int[] data = new int[2];

        @Override
        protected List<String> getFieldOrder() {
            return Arrays.asList("data");
        }

    }

    /* Constants used in Advantech GPIO driver ioctl() calls */

    public static final int SETGPIODIR = 0x7007;
    public static final int GETGPIODIR = 0x7008;
    public static final int ECSETGPIOSTATUS = 0x7009;
    public static final int ECGETGPIOSTATUS = 0x700a;

    public static final int DIR_IN = 0x80;
    public static final int DIR_OUT = 0x40;
    public static final int OUTPUT_LOW = 0;
    public static final int OUTPUT_HIGH	= 1;

    public GPIODriverUno1483() throws DriverException {

        super();

        try {
            advgpio = new RandomAccessFile("/dev/advgpio", "rw");
            Field fdField = FileDescriptor.class.getDeclaredField("fd");
            fdField.setAccessible(true); // Allow access to the private field
            fd = (int)fdField.get(advgpio.getFD());
        } catch (Exception e) {
            throw new DriverException("Could not open Uno1483 GPIO driver", e);
        }

        // Advantech driver doesn't support enumeration; statically initialize available chip/channels for
        // known fixed hardware here instead...

        GPIOChipsByIndex.add(new GPIOChip(0, 8, "gpiochip0", "Advantech GPIO"));

        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 0, "DI0", 0));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 1, "DI1", 1));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 2, "DI2", 2));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 3, "DI3", 3));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 4, "DO0", 4));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 5, "DO1", 5));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 6, "DO2", 6));
        GPIOChannelsByIndex.add(new GPIOChannelUno1483(0, 7, "DO3", 7));

    }

    public List<? extends GPIOChip> enumerateChips() throws DriverException {
        return this.GPIOChipsByIndex;
    }

    public List<? extends GPIOChannel> enumerateExportedChannels() throws DriverException {
        return GPIOChannelsByIndex;
    }

    public GPIOChannelUno1483 getChannel(int channel) throws DriverException {
        if ((channel < 0) || (channel >= GPIOChannelsByIndex.size())) {
            throw new DriverException("Invalid channel " + channel);
        } else {
            return this.GPIOChannelsByIndex.get(channel);
        }
    }

    public boolean isExported(int channel) {
        return true;
    }

    public GPIOChannelUno1483 export(int channel) throws DriverException {
        return getChannel(channel);
    }

    public List<? extends GPIOChannel> exportRange(int base, int ngpio) throws DriverException {
        List<GPIOChannelUno1483> result = new ArrayList<>();
        for (int channel = base; channel < base + ngpio; channel++) {
            result.add(getChannel(channel));
        }
        return result;
    }

    public void unexport(int channel) throws DriverException {
    }

    public void unexportRange(int base, int ngpio) throws DriverException {
    }

    public final class GPIOChannelUno1483 extends GPIOChannel {

        private GPIOChannelUno1483(int chip, int channel, String name, int offset) {
            super(chip, channel, name, offset);
        }

        public Direction getDirection() throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.data[1] = 0;
            data.write();
            try {
                libc.INSTANCE.ioctl(fd, GETGPIODIR, data.getPointer());
            } catch (LastErrorException e) {
                throw new DriverException("Error in GETGPIODIR ioctl", e);
            }
            data.read();
            switch(data.data[1]) {
                case DIR_IN: return Direction.IN;
                case DIR_OUT: return Direction.OUT;
                default: throw new DriverException("Bad value returned from GETGPIODIR " + data.data[1]);
            }
        }

        public void setDirection(Direction dir) throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.write();
            switch(dir) {
                case IN: data.data[1] = DIR_IN; break;
                case OUT: data.data[1] = DIR_OUT; break;
                default: throw new DriverException("Bad value passed to SETGPIODIR");
            }
            try {
                libc.INSTANCE.ioctl(fd, SETGPIODIR, data.getPointer());
            } catch (LastErrorException e) {
                throw new DriverException("Error in SETGPIODIR ioctl", e);
            }
        }

        public void set() throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.data[1] = OUTPUT_HIGH;
            data.write();
            try {
                libc.INSTANCE.ioctl(fd, ECSETGPIOSTATUS, data.getPointer());
            } catch (LastErrorException e) {
                throw new DriverException("Error in ECSETGPIOSTATUS ioctl", e);
            }
        }

        public void clear() throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.data[1] = OUTPUT_LOW;
            data.write();
            try {
                libc.INSTANCE.ioctl(fd, ECSETGPIOSTATUS, data.getPointer());
            } catch (LastErrorException e) {
                throw new DriverException("Error in ECSETGPIOSTATUS ioctl", e);
            }
        }

        public boolean read() throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.data[1] = 0;
            data.write();
            try {
                libc.INSTANCE.ioctl(fd, ECGETGPIOSTATUS, data.getPointer());
            } catch (LastErrorException e) {
                throw new DriverException("Error in ECGETGPIOSTATUS ioctl", e);
            }
            data.read();
            return data.data[1] != OUTPUT_LOW;
        }

        public void write(boolean value) throws DriverException {
            GpioIoctlData data = new GpioIoctlData();
            data.data[0] = this.offset;
            data.data[1] = value ? OUTPUT_HIGH : OUTPUT_LOW;
            data.write();
            try {
                libc.INSTANCE.ioctl(fd, ECSETGPIOSTATUS, data.getPointer());
             } catch(LastErrorException e) {
                throw new DriverException("Error in ECSETGPIOSTATUS ioctl", e);
            }
        }

        public void lock() throws DriverException {
        }

        public void unlock() throws DriverException {
        }

    }

}
