package org.lsst.ccs.drivers.gpio;

import java.util.List;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.gpio.GPIODriver.GPIOChannel;
import org.lsst.ccs.drivers.gpio.GPIODriver.GPIOChannel.Direction;
import org.lsst.ccs.drivers.gpio.GPIODriver.GPIOChip;

/**
 * Simple interactive test program
 *
 * @author tonyj
 */
public class TestGPIODriver {

    private final GPIODriver gpio;

    public TestGPIODriver() throws DriverException {
        gpio = new GPIODriver();
    }

    @Command(name = "enumerateChips", description = "Enumerate available chips")
    public List<GPIOChip> enumerateChips() throws DriverException {
        return gpio.enumerateChips();
    }

    @Command(name = "enumerateChannels", description = "Enumerate available chips")
    public List<GPIOChannel> enumerateChannels() throws DriverException {
        return gpio.enumerateExportedChannels();
    }

    @Command(name = "export", description = "Export a single channel")
    public GPIOChannel export(@Argument(name = "channel") int channel) throws DriverException {
        return gpio.export(channel);
    }

    @Command(name = "unexport", description = "Export a single channel")
    public void unexport(@Argument(name = "channel") int channel) throws DriverException {
        gpio.unexport(channel);
    }

    @Command(name = "setChannel", description = "Set a single channel")
    public void set(@Argument(name = "channel") int channel) throws DriverException {
        gpio.getChannel(channel).set();
    }

    @Command(name = "clearChannel", description = "Clear a single channel")
    public void clear(@Argument(name = "channel") int channel) throws DriverException {
        gpio.getChannel(channel).clear();
    }

    @Command(name = "readChannel", description = "Read a single channel")
    public boolean read(@Argument(name = "channel") int channel) throws DriverException {
        return gpio.getChannel(channel).read();
    }

    @Command(name = "readAll", description = "Read all exported channels channel")
    public void readAll() throws DriverException {
        for (GPIOChannel channel : enumerateChannels()) {
            System.out.printf("Channel %d direction=%s value %s\n", channel.getChannel(), channel.getDirection(), channel.read());
        }
    }

    @Command(name = "lock", description = "Lock a single channel")
    public void lock(@Argument(name = "channel") int channel) throws DriverException {
        gpio.getChannel(channel).lock();
    }

    @Command(name = "unlock", description = "Unlock a single channel")
    public void unlock(@Argument(name = "channel") int channel) throws DriverException {
        gpio.getChannel(channel).unlock();
    }

    @Command(name = "getDirection", description = "Clear a single channel")
    public Direction getDirection(@Argument(name = "channel") int channel) throws DriverException {
        return gpio.getChannel(channel).getDirection();
    }

    @Command(name = "setDirection", description = "Clear a single channel")
    public void setDirection(@Argument(name = "channel") int channel, @Argument(name = "direction") Direction dir) throws DriverException {
        gpio.getChannel(channel).setDirection(dir);
    }

    @Command(name = "exportRange", description = "Export all channels in specified range")
    public void exportRange(@Argument(name = "start") int start, @Argument(name = "n") int n) throws DriverException {
        gpio.exportRange(start, n);
    }

    @Command(name = "unexportRange", description = "Export all channels in specified range")
    public void unexportRange(@Argument(name = "start") int start, @Argument(name = "n") int n) throws DriverException {
        gpio.unexportRange(start, n);
    }

    @Command(name = "pollTest", description = "Repeatedly read a channel")
    public void pollTest(@Argument(name = "channel") int channel, @Argument(name = "n", defaultValue = "1000") int n) throws DriverException {
        GPIOChannel gpioChannel = gpio.getChannel(channel);
        long start = System.currentTimeMillis();
        for (int i = 0; i < n; i++) {
            gpioChannel.read();
        }
        long stop = System.currentTimeMillis();
        System.out.printf("Polled channel %d %d times, took=%dms\n", channel, n, stop - start);

        start = System.currentTimeMillis();
        gpioChannel.lock();
        for (int i = 0; i < n; i++) {
            gpioChannel.read();
        }
        gpioChannel.unlock();
        stop = System.currentTimeMillis();
        System.out.printf("Polled channel %d %d times, took=%dms\n", channel, n, stop - start);
    }
}
