package org.lsst.ccs.drivers.gpio;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A general purpose GPIO driver. This driver has been tested on UNO 1483 and
 * raspberry pi 3, but should work on other Linux systems with minimal changes.
 *
 * @see <a href="https://elinux.org/GPIO">Linux GPIO documentation</a>
 * @author tonyj
 */
public class GPIODriver {

    private final Path root;
    private final Map<Integer, GPIOChannel> channelMap = new HashMap<>();
    private final Predicate<String> chipPattern = Pattern.compile("gpiochip\\d+").asPredicate();
    private final Predicate<String> channelPattern = Pattern.compile("gpio\\d+").asPredicate();

    /**
     * Create an instance of GPIO driver
     *
     * @throws DriverException If the system does not support GPIO
     */
    public GPIODriver() throws DriverException {
        File rootFile = new File("/sys/class/gpio");
        if (!rootFile.isDirectory()) {
            throw new DriverException("GPIO is not supported on this system");
        }
        root = rootFile.toPath();
    }

    private GPIOChip readChip(Path dir) throws WrappedDriverException {
        try {
            return new GPIOChip(dir);
        } catch (DriverException ex) {
            throw new WrappedDriverException(ex);
        }
    }

    /**
     * Enumerate the GPIO chips on this machine
     *
     * @return The list of GPIO chips
     * @throws DriverException If an IO error occurs while enumerating the chips
     */
    public List<GPIOChip> enumerateChips() throws DriverException {
        try (Stream<Path> stream = Files.list(root).filter(t -> chipPattern.test(t.getFileName().toString()))) {
            return stream.map(p -> readChip(p)).collect(Collectors.toList());
        } catch (WrappedDriverException ex) {
            throw ex.getWrappedException();
        } catch (IOException ex) {
            throw new DriverException("Unexpected IO exception", ex);
        }
    }

    private GPIOChannel getChannel(Path dir) {
        int channel = Integer.parseInt(dir.getFileName().toString().substring(4));
        return getChannelFromMap(channel, dir);
    }

    /**
     * Enumerate the currently exported GPIO channels on this machine
     *
     * @return The list of exported channels
     * @throws DriverException If an IO exception occurs during the enumeration
     */
    public List<GPIOChannel> enumerateExportedChannels() throws DriverException {
        try (Stream<Path> stream = Files.list(root).filter(t -> channelPattern.test(t.getFileName().toString()))) {
            return stream.map(p -> getChannel(p)).collect(Collectors.toList());
        } catch (IOException ex) {
            throw new DriverException("Unexpected IO exception", ex);
        }
    }

    /**
     * Get a specified GPIO channel
     *
     * @param channel The channel to fetch
     * @return The Channel
     * @throws DriverException If the GPIO can not be fetched, for example if it
     * has not been exported.
     */
    public GPIOChannel getChannel(int channel) throws DriverException {
        Path path = root.resolve("gpio" + channel);
        if (!path.toFile().isDirectory()) {
            throw new DriverException("Invalid channel " + channel + " (not exported?)");
        }
        return getChannelFromMap(channel, path);
    }

    private GPIOChannel getChannelFromMap(int channel, Path path) {
        synchronized (channelMap) {
            GPIOChannel result = channelMap.get(channel);
            if (result == null) {
                result = new GPIOChannel(channel, path);
                channelMap.put(channel, result);
            }
            return result;
        }
    }

    /**
     * Test if the given channel is already exported
     *
     * @param channel The channel to test
     * @return <code>true</code> if and only if the channel has already been
     * exported.
     */
    public boolean isExported(int channel) {
        Path path = root.resolve("gpio" + channel);
        return path.toFile().isDirectory();
    }

    /**
     * Export the specified channel.
     *
     * @param channel The channel number
     * @return The exported channel
     * @throws DriverException If the channel is invalid, or cannot be exported
     */
    public GPIOChannel export(int channel) throws DriverException {
        writeToFile(root.resolve("export"), Integer.toString(channel));
        return getChannel(channel);
    }

    /**
     * Export all of the channels in the given range. Channels which are already
     * exported are not re-exported.
     *
     * @param base The first channel to export
     * @param ngpio The number of channels to export
     * @return The list of exported channels. including any which were already
     * exported.
     * @throws DriverException If an error occurs.
     */
    public List<GPIOChannel> exportRange(int base, int ngpio) throws DriverException {
        List<GPIOChannel> result = new ArrayList<>();
        for (int channel = base; channel < base + ngpio; channel++) {
            if (!isExported(channel)) {
                result.add(export(channel));
            } else {
                result.add(getChannel(channel));
            }
        }
        return result;
    }

    /**
     * Unexport all of the channels in the given range. Channels which are not
     * already exported are ignored.
     *
     * @param base The first channel to unexport
     * @param ngpio The number of channels to unexport
     * @throws DriverException If an error occurs.
     */
    public void unexportRange(int base, int ngpio) throws DriverException {
        for (int channel = base; channel < base + ngpio; channel++) {
            if (isExported(channel)) {
                unexport(channel);
            }
        }
    }

    /**
     * Unexport the specified channel
     *
     * @param channel The channel number
     * @throws DriverException If the channel is invalid, or cannot be un
     * exported
     */
    public void unexport(int channel) throws DriverException {
        writeToFile(root.resolve("unexport"), Integer.toString(channel));
    }

    private static String getFirstLine(Path file) throws DriverException {
        try (Stream<String> lines = Files.lines(file)) {
            return lines.findFirst().get();
        } catch (NoSuchElementException | IOException x) {
            throw new DriverException("Error reading " + file, x);
        }
    }

    private static void writeToFile(Path file, String line) throws DriverException {
        try {
            Files.write(file, line.getBytes());
        } catch (IOException x) {
            throw new DriverException("Error writing " + line + " to " + file, x);
        }
    }

    /**
     * A class representing a single GPIO Channel
     */
    public static final class GPIOChannel {

        private final int channel;
        private final Path path;
        private RandomAccessFile randomAccessValueFile;
        private final Path valuePath;
        private static final byte ONE_BYTE = "1".getBytes()[0];
        private static final byte ZERO_BYTE = "0".getBytes()[0];
        private FileLock lock;

        /**
         * Enumeration for specifying the direction of a channel
         */
        public enum Direction {

            /**
             * A readable channel
             */
            IN,

            /**
             * A writable channel
             */
            OUT
        };

        private GPIOChannel(int channel, Path path) {
            this.channel = channel;
            this.path = path;
            this.valuePath = path.resolve("value");
        }

        /**
         * Get the direction for this channel
         *
         * @return The direction
         * @throws DriverException If the direction cannot be determined
         */
        public Direction getDirection() throws DriverException {
            return Direction.valueOf(getFirstLine(path.resolve("direction")).toUpperCase());
        }

        /**
         * Set the direction for this channel
         *
         * @param dir The direction to be set.
         * @throws DriverException If the direction cannot be set.
         */
        public void setDirection(Direction dir) throws DriverException {
            writeToFile(path.resolve("direction"), dir.toString().toLowerCase());
        }

        /**
         * Set the channel (to 1)
         *
         * @throws DriverException If the channel cannot be set
         */
        public void set() throws DriverException {
            write(true);
        }

        /**
         * Clear the channel (to 0)
         *
         * @throws DriverException
         */
        public void clear() throws DriverException {
            write(false);
        }

        /**
         * Read the current value of the channel
         *
         * @return <code>true</code> if the channel is set (value == 1)
         * @throws DriverException If an error occurs
         */
        public boolean read() throws DriverException {
            if (randomAccessValueFile != null) {
                try {
                    randomAccessValueFile.seek(0);
                    return ONE_BYTE == randomAccessValueFile.readByte();
                } catch (IOException ex) {
                    throw new DriverException("Error reading channel", ex);
                }
            } else {
                return "1".equals(getFirstLine(valuePath));
            }
        }

        /**
         * Write a new value to this channel
         *
         * @param value The value to write
         * @throws DriverException If an error occurs
         */
        public void write(boolean value) throws DriverException {
            if (randomAccessValueFile != null) {
                try {
                    randomAccessValueFile.seek(0);
                    randomAccessValueFile.write(value ? ONE_BYTE : ZERO_BYTE);
                } catch (IOException ex) {
                    throw new DriverException("Error reading channel", ex);
                }
            } else {
                writeToFile(valuePath, value ? "1" : "0");
            }
        }

        /**
         * Locks this channel for exclusive access. This method works by taking
         * an exclusive lock on the value file, which speeds up all subsequent
         * read/write operations on the channel.
         *
         * @throws DriverException If the lock cannot be obtained.
         */
        public void lock() throws DriverException {
            try {
                randomAccessValueFile = new RandomAccessFile(valuePath.toFile(), "rws");
                lock = randomAccessValueFile.getChannel().lock();
            } catch (IOException ex) {
                throw new DriverException("Unable to lock file", ex);
            }
        }

        /**
         * Unlocks the specified channel. If the channel is not locked the
         * method does nothing.
         *
         * @throws DriverException If the channel unlock fails
         */
        public void unlock() throws DriverException {
            try {
                RandomAccessFile local = this.randomAccessValueFile;
                if (local != null) {
                    this.randomAccessValueFile = null;
                    lock.release();
                    local.close();
                }
            } catch (IOException ex) {
                throw new DriverException("Unable to unlock file", ex);
            }
        }

        /**
         * Return the channel number for this channel.
         *
         * @return The channel number
         */
        public int getChannel() {
            return channel;
        }

        @Override
        public String toString() {
            return "GPIOChannel{" + "channel=" + channel + '}';
        }
    }

    /**
     * A class representing a GPIOChip
     */
    public final class GPIOChip {

        private final int base;
        private final int ngpio;
        private final String label;

        private GPIOChip(Path dir) throws DriverException {
            try {
                base = Integer.parseInt(getFirstLine(dir.resolve("base")));
                ngpio = Integer.parseInt(getFirstLine(dir.resolve("ngpio")));
                label = getFirstLine(dir.resolve("label"));
            } catch (NumberFormatException ex) {
                throw new DriverException("Error parsing chip " + dir, ex);
            }
        }

        /**
         * Get the base channel number for this chip
         *
         * @return The base number
         */
        public int getBase() {
            return base;
        }

        /**
         * Get the number of GPIO channels supported by this chip
         *
         * @return The number of channels
         */
        public int getNgpio() {
            return ngpio;
        }

        /**
         * Get the human readable label associated with this chip
         * @return The label
         */
        public String getLabel() {
            return label;
        }

        /**
         * Export all of the channels associated with this chip. Channels which
         * are already exported are not re-exported (to avoid errors).
         *
         * @return The list of channels exported, including any which were
         * already exported.
         * @throws DriverException If an error occurs
         */
        public List<GPIOChannel> exportAll() throws DriverException {
            return exportRange(base, ngpio);
        }

        /**
         * Export all of the channels associated with this chip/ Channels which
         * are not exported are ignored.
         *
         * @throws DriverException If an error occurs
         */
        public void unexportAll() throws DriverException {
            unexportRange(base, ngpio);
        }

        @Override
        public String toString() {
            return "GPIOChip{" + "base=" + base + ", ngpio=" + ngpio + ", label=" + label + '}';
        }
    }

    private final static class WrappedDriverException extends RuntimeException {

        private static final long serialVersionUID = 1L;

        WrappedDriverException(DriverException x) {
            initCause(x);
        }

        private DriverException getWrappedException() {
            return (DriverException) getCause();
        }
    }
}
