package org.lsst.ccs.drivers.ftdi;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *****************************************************************************
 **
 **  Program to enable direct talking to a device using the FTDI chip.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */
public class TalkFtdi {

    private final static PrintStream out = System.out;
    private final static String[] terms = {"", "\r\n", "\r", "\n"};
    private final Thread readW = new Thread(new Reader());
    private final Ftdi ftd = new Ftdi();
    private int nRead = 0, nByte = 0;
    private boolean open;

    enum Parity {

        N(Ftdi.PARITY_NONE),
        O(Ftdi.PARITY_ODD),
        E(Ftdi.PARITY_EVEN),
        M(Ftdi.PARITY_MARK),
        S(Ftdi.PARITY_SPACE);

        int value;

        Parity(int value)
        {
            this.value = value;
        }

        int getValue()
        {
            return value;
        }
    }

    enum Flow {

        N(Ftdi.FLOW_CONTROL_NONE),
        R(Ftdi.FLOW_CONTROL_RTS_CTS),
        D(Ftdi.FLOW_CONTROL_DTR_DSR),
        X(Ftdi.FLOW_CONTROL_XON_XOFF);

        int value;

        Flow(int value)
        {
            this.value = value;
        }

        int getValue()
        {
            return value;
        }

    }

    private enum Terminator {
        NONE, CRLF, CR, LF;
    }



   /**
    **************************************************************************
    **
    **  Inner class for holding command line options.
    **
    **************************************************************************
    */
    private static class Options {

        @Option(name="-n", metaVar="<node>",
                usage="The network node running an FTDI server (default: local device)")
        private String node;

        @Option(name="-s",  metaVar="<serial>",
                usage="The (partial) serial number of the device (default: match all)")
        private String serial;

        @Option(name="-i", metaVar="<index>",
                usage="Index in the list of matching devices (default 0)")
        private int index = 0;

        @Option(name="-b", metaVar="<baudRate>",
                usage="Baud rate (default 115200)")
        private int baud = 115200;

        @Option(name="-p", metaVar="<parity>",
                usage="Parity: n(one) (default), o(dd), e(ven), m(ark) or s(pace)")
        private Parity parity = Parity.N;

        @Option(name="-d8", usage="Eight data bits (default)")
        private boolean dataBits8;

        @Option(name="-d7", usage="Seven data bits")
        private boolean dataBits7;

        @Option(name="-s1", usage="One stop bit (default)")
        private boolean stopBits1;

        @Option(name="-s2", usage="Two stop bits")
        private boolean stopBits2;

        @Option(name="-f",  metaVar="<flow>",
                usage="Flow control: n(one) (default), r(ts/cts), d(tr/dsr) or x(on/off)")
        private Flow flow = Flow.N;

        @Option(name="-t", metaVar="<term>",
                usage="Sent line terminator: none, crlf, cr or lf (default)")
        private Terminator term = Terminator.LF;

        @Option(name="-help", usage="Display this help")
        private boolean showHelp;
    }


   /**
    **************************************************************************
    **
    **  Inner class to implement device reading thread.
    **
    **************************************************************************
    */
    private class Reader implements Runnable {

        byte[] data = new byte[1024];

        @Override
        public void run()
        {
            while (true) {
                int leng = 0;
                try {
                    leng = ftd.getQueueStatus();
                    if (leng == 0) {
                        ftd.awaitEvent(0);
                        leng = ftd.getQueueStatus();
                    }
                    if (leng > data.length) {
                        leng = data.length;
                    }
                    if (!open) break;
                    leng = ftd.read(data, 0, leng);
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                    }
                    break;
                }
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


   /**
    **************************************************************************
    **
    **  Main program.
    **
    **  @param  args  The command-line arguments
    **
    **  @throws  Exception
    **
    **************************************************************************
    */
    public static void main(String[] args)
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
            if (optns.showHelp) {
                showHelp(parser);
            }
            else {
                (new TalkFtdi()).run(optns);
            }
        }
        catch (CmdLineException e) {
            if (!optns.showHelp) {
                out.println(e.getMessage());
            }
            showHelp(parser);
        }
        catch (Exception e) {
            out.println(e.getMessage());
        }
        System.exit(0);
    }


   /**
    **************************************************************************
    **
    **  Sends commands to the device.
    **
    **  Loops reading and processing each new typed command line.
    **
    **************************************************************************
    */
    private void run(Options optns) throws Exception
    {
        ftd.open(optns.node, optns.index, optns.serial);
        ftd.setBaudrate(optns.baud);
        int dataBits = (optns.dataBits7 && !optns.dataBits8)
                         ? Ftdi.DATABITS_7 : Ftdi.DATABITS_8;
        int stopBits = (optns.stopBits2 && !optns.stopBits1)
                         ? Ftdi.STOPBITS_2 : Ftdi.STOPBITS_1;
        ftd.setDataCharacteristics(dataBits, stopBits,
                                   optns.parity.getValue());
        ftd.setFlowControl(optns.flow.getValue());
        ftd.enableEvents(Ftdi.EVENT_RXCHAR);
        open = true;
        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();
        String term = terms[optns.term.ordinal()];
        out.println("Connected. Type CTRL-D to exit, CR for read stats.");

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            }
            else {
                ftd.write((line + term).getBytes());
            }
        }

        open = false;
        ftd.close();
    }


   /**
    ***************************************************************************
    **
    **  Displays help.
    **
    ***************************************************************************
    */
    private static void showHelp(CmdLineParser parser)
    {
        out.println("Usage: TalkFtdi [options...]");
        out.println("Options:");
        parser.printUsage(out);
    }

}
