package org.lsst.ccs.drivers.ftdi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ******************************************************************************
 **
 **  Remotely accesses a device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class FtdiClient implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private Socket srvSock;
    private InputStream srvIn;
    private OutputStream srvOut;
    private final Map<Integer, ArrayBlockingQueue> queueMap = new HashMap<>();
    private ThreadLocal<Integer> threadId;
    private ThreadLocal<ArrayBlockingQueue> threadQueue;
    private int currThreadId = 0;


   /**
    ***************************************************************************
    **
    **  Inner class implementing a network reading thread.
    **
    ***************************************************************************
    */
    private class Reader extends Thread {

        @Override
        public void run()
        {
            DriverException excp = null;
            while (true) {
                try {
                    byte[] header = new byte[FtdiServer.POSN_DATA];
                    int leng = 0, recLeng = header.length;
                    while (leng < recLeng) {
                        int nread = srvIn.read(header, leng, recLeng - leng);
                        if (nread < 0) break;
                        leng += nread;
                    }
                    if (leng < recLeng) {
                        throw new DriverException("Server disconnected");
                    }
                    if (Convert.bytesToInt(header, FtdiServer.POSN_MAGIC)
                          != FtdiServer.MAGIC_NUMBER) {
                        System.out.print("Header ="); 
                        for (int j = 0; j < header.length; j++) {
                            System.out.format(" %02x", header[j]);
                        }
                        System.out.println();
                        throw new DriverException("Invalid magic number");
                    }
                    recLeng = Convert.bytesToShort(header,
                                                   FtdiServer.POSN_LENGTH);
                    byte[] reply = new byte[recLeng];
                    System.arraycopy(header, 0, reply, 0, leng);
                    while (leng < recLeng) {
                        int nread = srvIn.read(reply, leng, recLeng - leng);
                        if (nread < 0) break;
                        leng += nread;
                    }
                    if (leng < recLeng) {
                        throw new DriverException("Server disconnected");
                    }
                    int context = Convert.bytesToShort(reply,
                                                       FtdiServer.POSN_CONTEXT);
                    queueMap.get(context).offer(reply);
                }
                catch (DriverException e) {
                    excp = e;
                    closeNetSilent();
                    break;
                }
                catch (IOException e) {
                    if (closeNetSilent()) {
                        excp = new DriverException(e);
                    }
                    break;
                }
            }
            if (excp != null) {
                Iterator queues = queueMap.values().iterator();
                while (queues.hasNext()) {
                    ((ArrayBlockingQueue)queues.next()).offer(excp);
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  This is an invalid operation.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(int index, String serial) throws DriverException
    {
        throw new DriverException("Invalid local open call");
    }


   /**
    ***************************************************************************
    **
    **  Opens a remote device.
    **
    **  @param  node    The name of the node where the device is located.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial)
        throws DriverException
    {
        try {
            srvSock = new Socket(node, FtdiServer.SERVER_PORT);
            srvIn = srvSock.getInputStream();
            srvOut = srvSock.getOutputStream();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
        queueMap.clear();
        threadId = new ThreadLocal();
        threadQueue = new ThreadLocal();
        Reader rdr = new Reader();
        rdr.setDaemon(true);
        rdr.start();
        byte[] bSerial = serial == null ? new byte[0] : serial.getBytes();
        int lSerial = bSerial.length;
        byte[] rqst = new byte[FtdiServer.POSN_SERIAL + lSerial];
        Convert.intToBytes(index, rqst, FtdiServer.POSN_INDEX);
        System.arraycopy(bSerial, 0, rqst, FtdiServer.POSN_SERIAL, lSerial);
        try {
            send(FtdiServer.FUNC_OPEN, rqst);
            receive();
        }
        catch (DriverException e) {
            closeNetSilent();
            throw e;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_DATA];
        send(FtdiServer.FUNC_CLOSE, rqst);
        closeNet();
    }


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate ro set
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setBaudrate(int baudrate) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_BAUDRATE + 4];
        Convert.intToBytes(baudrate, rqst, FtdiServer.POSN_BAUDRATE);
        send(FtdiServer.FUNC_BAUDRATE, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set
    **
    **  @param  stopBits    The encoded number of stop bits to set
    **
    **  @param  parity      The encoded parity value to set
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setDataCharacteristics(int wordLength, int stopBits,
                                       int parity)
        throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_PARITY + 4];
        Convert.intToBytes(wordLength, rqst, FtdiServer.POSN_WORDLENG);
        Convert.intToBytes(stopBits, rqst, FtdiServer.POSN_STOPBITS);
        Convert.intToBytes(parity, rqst, FtdiServer.POSN_PARITY);
        send(FtdiServer.FUNC_DATACHAR, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Sets the flow control.
    **
    **  @param  flow  The encoded flow control value to set
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setFlowControl(int flow) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_FLOWCTRL + 4];
        Convert.intToBytes(flow, rqst, FtdiServer.POSN_FLOWCTRL);
        send(FtdiServer.FUNC_FLOWCTRL, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms)
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms)
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_XMITTMO + 4];
        Convert.intToBytes(rcveTimeout, rqst, FtdiServer.POSN_RCVETMO);
        Convert.intToBytes(xmitTimeout, rqst, FtdiServer.POSN_XMITTMO);
        send(FtdiServer.FUNC_TIMEOUTS, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the DTR line.
    **
    **  @param  set  True to set the line; false to clear it.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setDtr(boolean set) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_DTRSET + 1];
        rqst[FtdiServer.POSN_DTRSET] = (byte)(set ? 1 : 0);
        send(FtdiServer.FUNC_DTR, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the RTS line.
    **
    **  @param  set  True to set the line; false to clear it.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setRts(boolean set) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_RTSSET + 1];
        rqst[FtdiServer.POSN_RTSSET] = (byte)(set ? 1 : 0);
        send(FtdiServer.FUNC_RTS, rqst);
        receive();
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws DriverException
    {
        return read(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data
    **
    **  @param  offset  The offset in the array to the start of the data
    **
    **  @param  count   The maximum number of bytes to read
    **
    **  @return  The number of bytes read
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data, int offset, int count) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_READLENG + 4];
        Convert.intToBytes(count, rqst, FtdiServer.POSN_READLENG);
        send(FtdiServer.FUNC_READ, rqst);
        byte[] reply = receive();
        int nread = reply.length - FtdiServer.POSN_READDATA;
        System.arraycopy(reply, FtdiServer.POSN_READDATA, data, offset, nread);

        return nread;
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws DriverException
    {
        return write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write
    **
    **  @param  offset  The offset in the array to the start of the data
    **
    **  @param  count   The number of bytes to write
    **
    **  @return  The number of bytes written
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data, int offset, int count) throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_WRITEDATA + count];
        System.arraycopy(data, offset, rqst, FtdiServer.POSN_WRITEDATA, count);
        send(FtdiServer.FUNC_WRITE, rqst);

        return Convert.bytesToInt(receive(), FtdiServer.POSN_WRITELENG);
    }


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read
    **
    **  @return  The number of bytes in the read queue
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getQueueStatus() throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_DATA];
        send(FtdiServer.FUNC_QUEUESTAT, rqst);

        return Convert.bytesToInt(receive(), FtdiServer.POSN_QUEUESTAT);
    }


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getModemStatus() throws DriverException
    {
        byte[] rqst = new byte[FtdiServer.POSN_DATA];
        send(FtdiServer.FUNC_MODEMSTAT, rqst);

        return Convert.bytesToInt(receive(), FtdiServer.POSN_MODEMSTAT);
    }


   /**
    ***************************************************************************
    **
    **  Sends a request to the server.
    **
    **  @param  rqst  A byte array containing the request.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void send(int function, byte[] rqst) throws DriverException
    {
        int context;
        Integer id = threadId.get();
        if (id == null) {
            context = currThreadId++;
            threadId.set(context);
            threadQueue.set(new ArrayBlockingQueue(1));
            queueMap.put(context, threadQueue.get());
        }
        else {
            context = id;
        }
        Convert.intToBytes(FtdiServer.MAGIC_NUMBER, rqst,
                           FtdiServer.POSN_MAGIC);
        Convert.shortToBytes((short)rqst.length, rqst, FtdiServer.POSN_LENGTH);
        Convert.shortToBytes((short)function, rqst, FtdiServer.POSN_FUNCTION);
        Convert.shortToBytes((short)context, rqst, FtdiServer.POSN_CONTEXT);
        try {
            srvOut.write(rqst);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives a reply from the server.
    **
    **  @return  A byte array containing the reply.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private byte[] receive() throws DriverException
    {
        Object replyObj = null;
        try {
            replyObj = threadQueue.get().take();
        }
        catch (InterruptedException e) {
        }
        if (replyObj instanceof DriverException) {
            throw (DriverException)replyObj;
        }
        byte[] reply = (byte[])replyObj;
        if (Convert.bytesToShort(reply, FtdiServer.POSN_FUNCTION)
              == FtdiServer.FUNC_EXCEPTION) {
            String text = new String(reply, FtdiServer.POSN_EXCPTEXT,
                                     reply.length - FtdiServer.POSN_EXCPTEXT);
            throw new DriverException(text);
        }

        return reply;
    }


   /**
    ***************************************************************************
    **
    **  Closes the network connection silently.
    **
    **  @return  Whether the connection was open
    **
    ***************************************************************************
    */
    private boolean closeNetSilent()
    {
        boolean wasOpen = true;

        try {
            wasOpen = closeNet();
        }
        catch (DriverException e) {
        }

        return wasOpen;
    }


   /**
    ***************************************************************************
    **
    **  Closes the network connection.
    **
    **  @return  Whether the connection was open
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private boolean closeNet() throws DriverException
    {
        boolean wasOpen = false;
        Exception ei = null;

        try {
            if (srvSock != null) {
                wasOpen = true;
                srvSock.close();
            }
        }
        catch (IOException e) {
            ei = e;
        }
        srvSock = null;

        if (ei != null) {
            throw new DriverException(ei);
        }

        return wasOpen;
    }

}
