package org.lsst.ccs.drivers.ftdi;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Accesses a device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Ftdi implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    /** Word length value: 8 bits */
    public final static int DATABITS_8 = 8;

    /** Word length value: 7 bits */
    public final static int DATABITS_7 = 7;

    /** Stop bits value: 1 */
    public final static int STOPBITS_1 = 0;

    /** Stop bits value: 2 */
    public final static int STOPBITS_2 = 2;

    /** Parity value: none */
    public final static int PARITY_NONE  = 0;

    /** Parity value: odd */
    public final static int PARITY_ODD   = 1;

    /** Parity value: even */
    public final static int PARITY_EVEN  = 2;

    /** Parity value: mark */
    public final static int PARITY_MARK  = 3;

    /** Parity value: space */
    public final static int PARITY_SPACE = 4;

    /** Modem status bit: CTS */
    public final static int MODEM_STATUS_CTS = 0x10;

    /** Modem status bit: DSR */
    public final static int MODEM_STATUS_DSR = 0x20;

    /** Modem status bit: RI */
    public final static int MODEM_STATUS_RI = 0x40;

    /** Modem status bit: DCD */
    public final static int MODEM_STATUS_DCD = 0x80;

    /** Flow control value: none */
    public final static int FLOW_CONTROL_NONE = 0x0000;

    /** Flow control value: RTS/CTS */
    public final static int FLOW_CONTROL_RTS_CTS = 0x0100;

    /** Flow control value: DTR/DSR */
    public final static int FLOW_CONTROL_DTR_DSR = 0x0200;

    /** Flow control value: XON/XOFF */
    public final static int FLOW_CONTROL_XON_XOFF = 0x0400;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private FtdiClient client;
    private FtdiLocal local;
    private FtdiInterface ftdi;


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(int index, String serial) throws DriverException
    {
        checkNotOpen();
        if (local == null) {
            local = new FtdiLocal();
        }
        local.open(index, serial);
        ftdi = local;
    }


   /**
    ***************************************************************************
    **
    **  Opens a local or remote device.
    **
    **  @param  node    The name of the node where the device is located, or
    **                  null to specify a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial)
        throws DriverException
    {
        if (node == null) {
            open(index, serial);
        }
        else {
            checkNotOpen();
            if (client == null) {
                client = new FtdiClient();
            }
            client.open(node, index, serial);
            ftdi = client;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        checkOpen();
        try {
            ftdi.close();
        }
        finally {
            ftdi = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setBaudrate(int baudrate) throws DriverException
    {
        checkOpen();
        ftdi.setBaudrate(baudrate);
    }


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set.
    **
    **  @param  stopBits    The encoded number of stop bits to set.
    **
    **  @param  parity      The encoded parity value to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setDataCharacteristics(int wordLength, int stopBits,
                                       int parity)
        throws DriverException
    {
        checkOpen();
        ftdi.setDataCharacteristics(wordLength, stopBits, parity);
    }


   /**
    ***************************************************************************
    **
    **  Sets the flow control.
    **
    **  @param  flow  The encoded flow control value to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setFlowControl(int flow) throws DriverException
    {
        checkOpen();
        ftdi.setFlowControl(flow);
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws DriverException
    {
        checkOpen();
        ftdi.setTimeouts(rcveTimeout, xmitTimeout);
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the DTR line.
    **
    **  @param  set  True to set the line; false to clear it.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setDtr(boolean set) throws DriverException
    {
        checkOpen();
        ftdi.setDtr(set);
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the RTS line.
    **
    **  @param  set  True to set the line; false to clear it.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setRts(boolean set) throws DriverException
    {
        checkOpen();
        ftdi.setRts(set);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws DriverException
    {
        checkOpen();
        return ftdi.read(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The maximum number of bytes to read.
    **
    **  @return  The number of bytes read.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data, int offset, int count) throws DriverException
    {
        checkOpen();
        return ftdi.read(data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws DriverException
    {
        checkOpen();
        return ftdi.write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The number of bytes to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data, int offset, int count) throws DriverException
    {
        checkOpen();
        return ftdi.write(data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read, without
    **  execution being blocked.
    **
    **  @return  The number of bytes in the read queue.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getQueueStatus() throws DriverException
    {
        checkOpen();
        return ftdi.getQueueStatus();
    }


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getModemStatus() throws DriverException
    {
        checkOpen();
        return ftdi.getModemStatus();
    }


   /**
    ***************************************************************************
    **
    **  Checks that the device is not open
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void checkNotOpen() throws DriverException
    {
        if (ftdi != null) {
            throw new DriverException("Device already open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks that the device is open
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void checkOpen() throws DriverException
    {
        if (ftdi == null) {
            throw new DriverException("Device not open");
        }
    }

}
