package org.lsst.ccs.drivers.ftdi;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *****************************************************************************
 **
 **  Program to enable direct talking to a device using the FTDI chip.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */
public class TalkFtdi {

    private final static PrintStream out = System.out;
    private final Thread readW = new Thread(new Reader());
    private final Ftdi ftd = new Ftdi();
    private int nRead = 0, nByte = 0;
    private boolean open;


   /**
    **************************************************************************
    **
    **  Inner class to implement device reading thread.
    **
    **************************************************************************
    */
    private class Reader implements Runnable {

        byte[] data = new byte[1024];

        @Override
        public void run()
        {
            while (true) {
                int leng;
                try {
                    leng = ftd.read(data, 0, 1);
                    int nread = ftd.getQueueStatus();
                    if (nread > data.length - leng) {
                        nread = data.length - leng;
                    }
                    if (!open) break;
                    leng += ftd.read(data, leng, nread);
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                    }
                    break;
                }
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


   /**
    **************************************************************************
    **
    **  Inner class for holding command line options.
    **
    **************************************************************************
    */
    private static class Options {

        @Option(name="-n", usage="node name")
        private String node;

        @Option(name="-s", usage="serial number")
        private String serial;

        @Option(name="-i", usage="device index")
        private int index = 0;

        @Option(name="-b", usage="baud rate")
        private int baud = 115200;

    }


   /**
    **************************************************************************
    **
    **  Main program.
    **
    **  @param  args  The command-line arguments
    **
    **  @throws  Exception
    **
    **************************************************************************
    */
    public static void main(String[] args) throws Exception
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
        }
        catch (CmdLineException e) {
            out.println(e.getMessage());
            return;
        }
        (new TalkFtdi()).run(optns);
        System.exit(0);
    }


   /**
    **************************************************************************
    **
    **  Sends commands to the device.
    **
    **  Loops reading and processing each new typed command line.
    **
    **************************************************************************
    */
    private void run(Options optns) throws Exception
    {
        ftd.open(optns.node, optns.index, optns.serial);
        ftd.setBaudrate(optns.baud);
        ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                   Ftdi.PARITY_NONE);
        open = true;

        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            }
            else {
                ftd.write(line.getBytes());
            }
        }

        open = false;
        ftd.close();
    }

}
