package org.lsst.ccs.drivers.ftdi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 ******************************************************************************
 **
 **  Serves access to a device which uses the FTDI chip
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class FtdiServer extends Thread {

   /**
    ***************************************************************************
    **
    **  Package constants.
    **
    ***************************************************************************
    */
    final static int
        SERVER_PORT    = 9001,
        MAGIC_NUMBER   = 0x71e5f39c,

        FUNC_EXCEPTION = 0,
        FUNC_OPEN      = 1,
        FUNC_CLOSE     = 2,
        FUNC_READ      = 3,
        FUNC_WRITE     = 4,
        FUNC_TIMEOUTS  = 5,
        FUNC_BAUDRATE  = 6,
        FUNC_DATACHAR  = 7,
        FUNC_QUEUESTAT = 8,
        FUNC_MODEMSTAT = 9,

        POSN_MAGIC     = 0,
        POSN_LENGTH    = POSN_MAGIC + 4,
        POSN_FUNCTION  = POSN_LENGTH + 2,
        POSN_CONTEXT   = POSN_FUNCTION + 2,
        POSN_DATA      = POSN_CONTEXT + 2,
        POSN_EXCPTEXT  = POSN_DATA,
        POSN_INDEX     = POSN_DATA,
        POSN_SERIAL    = POSN_INDEX + 4,
        POSN_READLENG  = POSN_DATA,
        POSN_READDATA  = POSN_DATA,
        POSN_WRITEDATA = POSN_DATA,
        POSN_WRITELENG = POSN_DATA,
        POSN_RCVETMO   = POSN_DATA,
        POSN_XMITTMO   = POSN_RCVETMO + 4,
        POSN_BAUDRATE  = POSN_DATA,
        POSN_WORDLENG  = POSN_DATA,
        POSN_STOPBITS  = POSN_WORDLENG + 4,
        POSN_PARITY    = POSN_STOPBITS + 4,
        POSN_QUEUESTAT = POSN_DATA,
        POSN_MODEMSTAT = POSN_DATA;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private final static Logger log = Logger.getLogger("ftdiServer");
    private Socket cliSock;
    private final InputStream cliIn;
    private final OutputStream cliOut;
    private final FtdiLocal ftdi = new FtdiLocal();
    private final BlockingQueue<byte[]> readQ = new ArrayBlockingQueue<>(2);


   /**
    ***************************************************************************
    **
    **  Runs a device reader thread.
    **
    ***************************************************************************
    */
    private class Reader extends Thread {

        @Override
        public void run()
        {
            IOException excp = null;
            while (true) {
                byte[] rqst;
                try {
                    rqst = readQ.take();
                }
                catch (InterruptedException e) {
                    break;
                }
                try {
                    int leng = Convert.bytesToInt(rqst, POSN_READLENG);
                    byte[] data = new byte[leng];
                    int nread = ftdi.read(data);
                    byte[] reply = new byte[POSN_READDATA + nread];
                    System.arraycopy(data, 0, reply, POSN_READDATA, nread);
                    send(rqst, reply);
                }
                catch (DriverException ef) {
                    try {
                        sendException(rqst, ef);
                    }
                    catch (IOException ei) {
                        excp = ei;
                        break;
                    }
                }
                catch (IOException ei) {
                    excp = ei;
                    break;
                }
            }

            if (excp != null) {
                if (closeNetSilent()) {
                    log.warning(excp.toString());
                }
                try {
                    ftdi.close();
                }
                catch (DriverException e) {
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  sock  The network socket to use
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    public FtdiServer(Socket sock) throws IOException
    {
        cliSock = sock;
        cliIn = sock.getInputStream();
        cliOut = sock.getOutputStream();
    }


   /**
    ***************************************************************************
    **
    **  Main program.
    **
    **  @param  args  Command-line arguments
    **
    **  @throws  IOException
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws IOException
    {
        log.setLevel(Level.INFO);
        log.info("Started server");
        ServerSocket sSock = new ServerSocket(SERVER_PORT, 10);
        while (true) {
            Socket sock = sSock.accept();
            FtdiServer srvr = new FtdiServer(sock);
            srvr.setDaemon(true);
            srvr.start();
        }
    }


   /**
    ***************************************************************************
    **
    **  Runs a server thread.
    **
    ***************************************************************************
    */
    @Override
    public void run()
    {
        String client = cliSock.getInetAddress().getHostName();
        int port = cliSock.getPort();
        log.info("Opened connection to " + client + ":" + port);
        Reader rdr = new Reader();
        rdr.setDaemon(true);
        rdr.start();
        IOException excp = null;
        boolean running = true;

        while (running) {
            byte rqst[] = null;
            try {
                byte reply[];
                rqst = receive();
                int rLeng = Convert.bytesToShort(rqst, POSN_LENGTH);
                int func = Convert.bytesToShort(rqst, POSN_FUNCTION);

                switch (func) {

                case FUNC_OPEN:
                    int index = Convert.bytesToInt(rqst, POSN_INDEX);
                    String serial = null;
                    if (rLeng > POSN_SERIAL) {
                        serial = new String(rqst, POSN_SERIAL,
                                            rLeng - POSN_SERIAL);
                    }
                    ftdi.open(index, serial);
                    sendAck(rqst);
                    break;

                case FUNC_CLOSE:
                    ftdi.close();
                    closeNet();
                    running = false;
                    break;

                case FUNC_READ:
                    readQ.offer(rqst);
                    break;

                case FUNC_WRITE:
                    int nwrite = ftdi.write(rqst, POSN_WRITEDATA,
                                            rLeng - POSN_WRITEDATA);
                    reply = new byte[POSN_WRITELENG + 4];
                    Convert.intToBytes(nwrite, reply, POSN_WRITELENG);
                    send(rqst, reply);
                    break;

                case FUNC_TIMEOUTS:
                    int rcveTmo = Convert.bytesToInt(rqst, POSN_RCVETMO);
                    int xmitTmo = Convert.bytesToInt(rqst, POSN_XMITTMO);
                    ftdi.setTimeouts(rcveTmo, xmitTmo);
                    sendAck(rqst);
                    break;

                case FUNC_BAUDRATE:
                    int baudrate = Convert.bytesToInt(rqst, POSN_BAUDRATE);
                    ftdi.setBaudrate(baudrate);
                    sendAck(rqst);
                    break;

                case FUNC_DATACHAR:
                    int wordleng = Convert.bytesToInt(rqst, POSN_WORDLENG);
                    int stopbits = Convert.bytesToInt(rqst, POSN_STOPBITS);
                    int parity = Convert.bytesToInt(rqst, POSN_PARITY);
                    ftdi.setDataCharacteristics(wordleng, stopbits, parity);
                    sendAck(rqst);
                    break;

                case FUNC_QUEUESTAT:
                    reply = new byte[POSN_QUEUESTAT + 4];
                    Convert.intToBytes(ftdi.getQueueStatus(), reply,
                                       POSN_QUEUESTAT);
                    send(rqst, reply);
                    break;

                case FUNC_MODEMSTAT:
                    reply = new byte[POSN_MODEMSTAT + 4];
                    Convert.intToBytes(ftdi.getModemStatus(), reply,
                                       POSN_MODEMSTAT);
                    send(rqst, reply);
                    break;

                default:
                }
            }
            catch (DriverException ed) {
                try {
                    sendException(rqst, ed);
                }
                catch (IOException ei) {
                    excp = ei;
                    running = false;
                }
            }
            catch (IOException ei) {
                excp = ei;
                running = false;
            }
        }

        if (excp != null) {
            if (closeNetSilent()) {
                log.warning(excp.toString());
            }
            try {
                ftdi.close();
            }
            catch (DriverException ef) {
            }
        }

        rdr.interrupt();
        log.info("Closed connection to " + client + ":" + port);
    }


   /**
    ***************************************************************************
    **
    **  Receives a request from the client.
    **
    ***************************************************************************
    */
    private byte[] receive() throws IOException
    {
        byte[] header = new byte[POSN_DATA];
        int leng = 0, recLeng = header.length;
        while (leng < recLeng) {
            int nread = cliIn.read(header, leng, recLeng - leng);
            if (nread < 0) break;
            leng += nread;
        }
        if (leng < recLeng) {
            throw new IOException("Client disconnected");
        }
        if (Convert.bytesToInt(header, POSN_MAGIC) != MAGIC_NUMBER) {
            throw new IOException("Invalid magic number");
        }
        recLeng = Convert.bytesToShort(header, POSN_LENGTH);
        byte[] rqst = new byte[recLeng];
        System.arraycopy(header, 0, rqst, 0, leng);
        while (leng < recLeng) {
            int nread = cliIn.read(rqst, leng, recLeng - leng);
            if (nread < 0) break;
            leng += nread;
        }
        if (leng < recLeng) {
            throw new IOException("Client disconnected");
        }

        return rqst;
    }


   /**
    ***************************************************************************
    **
    **  Sends a reply to the client.
    **
    ***************************************************************************
    */
    private void send(byte[] rqst, byte[] reply) throws IOException
    {
        Convert.intToBytes(MAGIC_NUMBER, reply, POSN_MAGIC);
        Convert.shortToBytes((short)reply.length, reply, POSN_LENGTH);
        System.arraycopy(rqst, POSN_FUNCTION, reply, POSN_FUNCTION, 2);
        System.arraycopy(rqst, POSN_CONTEXT, reply, POSN_CONTEXT, 2);
        cliOut.write(reply);
    }


   /**
    ***************************************************************************
    **
    **  Sends an acknowledgment to the client.
    **
    ***************************************************************************
    */
    private void sendAck(byte[] rqst) throws IOException
    {
        byte[] reply = new byte[POSN_DATA];
        send(rqst, reply);
    }


   /**
    ***************************************************************************
    **
    **  Sends an exception response to the client.
    **
    ***************************************************************************
    */
    private void sendException(byte[] rqst, Exception e) throws IOException
    {
        if (rqst == null) return;
        byte[] text = e.getMessage().getBytes();
        byte[] reply = new byte[POSN_EXCPTEXT + text.length];
        System.arraycopy(text, 0, reply, POSN_EXCPTEXT, text.length);
        Convert.shortToBytes((short)FUNC_EXCEPTION, rqst, POSN_FUNCTION);
        send(rqst, reply);
    }


   /**
    ***************************************************************************
    **
    **  Closes the network connections silently.
    **
    ***************************************************************************
    */
    private boolean closeNetSilent()
    {
        boolean wasOpen = true;

        try {
            wasOpen = closeNet();
        }
        catch (IOException e) {
        }

        return wasOpen;
    }


   /**
    ***************************************************************************
    **
    **  Closes the network connections.
    **
    ***************************************************************************
    */
    private boolean closeNet() throws IOException
    {
        boolean wasOpen = false;
        try {
            if (cliSock != null) {
                wasOpen = true;
                cliSock.close();
            }
        }
        finally {
            cliSock = null;
        }

        return wasOpen;
    }

}
