package org.lsst.ccs.drivers.ftdi;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Accesses a locally-connected device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class FtdiLocal implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private long handle;

   /**
    ***************************************************************************
    **
    **  Static Initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Ftdi");
    }


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null, restricts the list of
    **                  available devices to those with a serial number
    **                  containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(int index, String serial)
        throws DriverException
    {
        if (handle != 0) {
            throw new DriverException("Device is already open");
        }
        handle = openJNI(index, serial);
    }


   /**
    ***************************************************************************
    **
    **  Opens a remote device.
    **
    **  This is an invalid operation.
    **
    **  @param  node    The name of the node where the device is located, or
    **                  null to specify a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial)
        throws DriverException
    {
        throw new DriverException("Invalid remote open call");
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        long hndl = getHandle();
        handle = 0;
        closeJNI(hndl);
    }


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setBaudrate(int baudrate) throws DriverException
    {
        setBaudrateJNI(getHandle(), baudrate);
    }


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set..
    **
    **  @param  stopBits    The encoded number of stop bits to set..
    **
    **  @param  parity      The encoded parity value to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setDataCharacteristics(int wordLength, int stopBits, int parity)
        throws DriverException
    {
        setDataCharacteristicsJNI(getHandle(), wordLength, stopBits, parity);
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws DriverException
    {
        setTimeoutsJNI(getHandle(), rcveTimeout, xmitTimeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws DriverException
    {
        return readJNI(getHandle(), data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The maximum number of bytes to read.
    **
    **  @return  The number of bytes read.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data, int offset, int count) throws DriverException
    {
        return readJNI(getHandle(), data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws DriverException
    {
        return writeJNI(getHandle(), data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The number of bytes to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data, int offset, int count) throws DriverException
    {
        return writeJNI(getHandle(), data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read.
    **
    **  @return  The number of bytes in the read queue.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getQueueStatus() throws DriverException
    {
        return getQueueStatusJNI(getHandle());
    }


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    @Override
    public int getModemStatus() throws DriverException
    {
        return getModemStatusJNI(getHandle());
    }


   /**
    ***************************************************************************
    **
    **  Gets the handle and checks whether the device is open.
    **
   **  @throws DriverException
    **
    ***************************************************************************
    */
    private long getHandle() throws DriverException
    {
        long hndl = handle;
        if (hndl == 0) {
            throw new DriverException("Device is not open");
        }
        return hndl;
    }


   /**
    ***************************************************************************
    **
    **  JNI method declarations
    **
    ***************************************************************************
    */
    private native long openJNI(int index, String serial)
        throws DriverException;

    private native void closeJNI(long handle) throws DriverException;

    private native void setBaudrateJNI(long handle, int baudrate)
        throws DriverException;

    private native void setDataCharacteristicsJNI(long handle, int wordLength,
                                                  int stopBits, int parity)
        throws DriverException;

    private native void setTimeoutsJNI(long handle, int rcveTimeout,
                                       int xmitTimeout)
        throws DriverException;

    private native int readJNI(long handle, byte[] data, int offset, int count)
        throws DriverException;

    private native int writeJNI(long handle, byte[] data, int offset, int count)
        throws DriverException;

    private native int getQueueStatusJNI(long handle) throws DriverException;

    private native int getModemStatusJNI(long handle) throws DriverException;

}
