package org.lsst.ccs.drivers.ftdi;

/**
 ***************************************************************************
 **
 **  Accesses a device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Ftdi implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    /** Word length value: 8 bits */
    public final static int DATABITS_8 = 8;

    /** Word length value: 7 bits */
    public final static int DATABITS_7 = 7;

    /** Stop bits value: 1 */
    public final static int STOPBITS_1 = 0;

    /** Stop bits value: 2 */
    public final static int STOPBITS_2 = 2;

    /** Parity value: none */
    public final static int PARITY_NONE  = 0;

    /** Parity value: odd */
    public final static int PARITY_ODD   = 1;

    /** Parity value: even */
    public final static int PARITY_EVEN  = 2;

    /** Parity value: mark */
    public final static int PARITY_MARK  = 3;

    /** Parity value: space */
    public final static int PARITY_SPACE = 4;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private FtdiClient client;
    private FtdiLocal local;
    private FtdiInterface ftdi;


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void open(int index, String serial) throws FtdiException
    {
        checkNotOpen();
        if (local == null) {
            local = new FtdiLocal();
        }
        local.open(index, serial);
        ftdi = local;
    }


   /**
    ***************************************************************************
    **
    **  Opens a local or remote device.
    **
    **  @param  node    The name of the node where the device is located, or
    **                  null to specify a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial) throws FtdiException
    {
        if (node == null) {
            open(index, serial);
        }
        else {
            checkNotOpen();
            if (client == null) {
                client = new FtdiClient();
            }
            client.open(node, index, serial);
            ftdi = client;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void close() throws FtdiException
    {
        checkOpen();
        try {
            ftdi.close();
        }
        finally {
            ftdi = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setBaudrate(int baudrate) throws FtdiException
    {
        checkOpen();
        ftdi.setBaudrate(baudrate);
    }


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set.
    **
    **  @param  stopBits    The encoded number of stop bits to set.
    **
    **  @param  parity      The encoded parity value to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setDataCharacteristics(int wordLength, int stopBits,
                                       int parity)
        throws FtdiException
    {
        checkOpen();
        ftdi.setDataCharacteristics(wordLength, stopBits, parity);
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws FtdiException
    {
        checkOpen();
        ftdi.setTimeouts(rcveTimeout, xmitTimeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws FtdiException
    {
        checkOpen();
        return ftdi.read(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The maximum number of bytes to read.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data, int offset, int count) throws FtdiException
    {
        checkOpen();
        return ftdi.read(data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws FtdiException
    {
        checkOpen();
        return ftdi.write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The number of bytes to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data, int offset, int count) throws FtdiException
    {
        checkOpen();
        return ftdi.write(data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read, without
    **  execution being blocked.
    **
    **  @return  The number of bytes in the read queue.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int getQueueStatus() throws FtdiException
    {
        checkOpen();
        return ftdi.getQueueStatus();
    }


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int getModemStatus() throws FtdiException
    {
        checkOpen();
        return ftdi.getModemStatus();
    }


   /**
    ***************************************************************************
    **
    **  Checks that the device is not open
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    private void checkNotOpen() throws FtdiException
    {
        if (ftdi != null) {
            throw new FtdiException("Device already open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks that the device is open
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    private void checkOpen() throws FtdiException
    {
        if (ftdi == null) {
            throw new FtdiException("Device not open");
        }
    }

}
